package slitscan.slit;

import java.io.File;

import slitscan.ContextIn;
import slitscan.ContextOut;
import slitscan.Orbit;
import slitscan.OrbitRoot;
import slitscan.Point;

public class NonLinearHorizontalSlit extends OrbitRoot implements Orbit {

	final String func;
	final int ifunc;
	final double a;
	final double b;
	
	public NonLinearHorizontalSlit(File[] in_files, String[] args) {
		this.func = checkArg(args, 0) ? args[0] : "sin";
		if(this.func.equals("log1p"))
			this.ifunc = 1;
		else if(this.func.equals("sqrt"))
			this.ifunc = 2;
		else if(this.func.equals("sqr"))
			this.ifunc = 3;
		else if(this.func.equals("pow"))
			this.ifunc = 4;
		else if(this.func.equals("smoothstep"))
			this.ifunc = 5;
		else 
			this.ifunc = 0;
		this.a = checkArg(args, 1) ? Double.parseDouble(args[1]) : 0.0;
		this.b = checkArg(args, 2) ? Double.parseDouble(args[2]) : 1.0;
	}

	@Override
	public ContextOut buildContextOut(ContextIn ictx) {
		this.ictx = ictx;
		this.octx = new ContextOut(ictx.depth, ictx.width);
		return this.octx;
	}
	
	private double smoothstep(double x) {
		if(x < this.a)
			return 0.0;
		if(x >= this.b)
			return 1.0;
		final double x_normalized = (x - this.a) / (this.b - this.a);
		return x_normalized * x_normalized * (3 - 2 * x_normalized);
	}
	
	@Override
	public void f(Point p) {
		p.sx = p.ty;
		switch(this.ifunc) {
		case 0:
			p.sz = Math.sin((double)p.tx / (double)this.octx.width * Math.PI) * (double)this.ictx.depth;
			break;
		case 1:
			p.sz = Math.log1p((double)p.tx / (double)this.octx.width * 1.72) * (double)this.ictx.depth;
			break;
		case 2:
			p.sz = Math.sqrt((double)p.tx / (double)this.octx.width) * (double)this.ictx.depth;
			break;
		case 3:
			double z = (double)p.tx / (double)this.octx.width;
			p.sz = z * z * (double)this.ictx.depth;
			break;
		case 4:
			p.sz = Math.pow((double)p.tx / (double)this.octx.width, this.a) * (double)this.ictx.depth;
			break;
		case 5:
			p.sz = smoothstep((double)p.tx / (double)this.octx.width) * (double)this.ictx.depth;
			break;
		default:
			p.sz = Math.sin((double)p.tx / (double)this.octx.width * Math.PI) * (double)this.ictx.depth;
		}
		p.sy = (int) (((double)p.tx / this.octx.width) * this.ictx.height);
	}

	@Override
	public String toString() {
		return "NonLinearHorizontalSlit [func=" + this.func + ", a=" + this.a + ", b=" + this.b + "]";
	}

}
