package slitscan.slit;

import java.io.File;
import java.util.Vector;

import slitscan.ContextIn;
import slitscan.ContextOut;
import slitscan.Orbit;
import slitscan.OrbitRoot;
import slitscan.Point;

public class LinearDDA extends OrbitRoot implements Orbit {

	int x0, y0, x1, y1;
	final ArgState arg_x0, arg_y0, arg_x1, arg_y1;
	boolean ascending;
	Vector<Integer> vx, vy;
	int[] tx, ty;
	
	
	public LinearDDA(File[] in_files, String[] args) {
		super();
		this.arg_x0 = checkArgState(args, 0, -1);
		this.arg_y0 = checkArgState(args, 1, -1);
		this.arg_x1 = checkArgState(args, 2, -1);
		this.arg_y1 = checkArgState(args, 3, -1);		
	}
	
	private void append(int x, int y) {
		this.vx.add(x);
		this.vy.add(y);
	}

	// see: https://en.wikipedia.org/wiki/Digital_differential_analyzer_(graphics_algorithm)
	private void ddaLine(int lx0, int ly0, int lx1, int ly1) {
		float dx = (lx1 - lx0);
		float dy = (ly1 - ly0);
		float step;
		if (Math.abs(dx) >= Math.abs(dy))
			step = Math.abs(dx);
		else
			step = Math.abs(dy);
		dx = dx / step;
		dy = dy / step;
		
		float x = x0;
		float y = y0;
		int i = 0;
		while (i <= step) {
			append(Math.round(x), Math.round(y));
			x = x + dx;
			y = y + dy;
			++i;
		}
	}

	private void buildTable() {
		this.vx = new Vector<Integer>();
		this.vy = new Vector<Integer>();
		ddaLine(this.x0, this.y0, this.x1, this.y1);
		this.tx = new int[this.vx.size()];
		this.ty = new int[this.vy.size()];
		for (int i = 0; i < this.tx.length; i++) {
			this.tx[i] = this.vx.get(i);
			this.ty[i] = this.vy.get(i);
		}
		this.vx = null;
		this.vy = null;
	}

	@Override
	public ContextOut buildContextOut(ContextIn ictx) {
		this.ictx = ictx;
		this.x0 = this.arg_x0.getInt(ictx.width);
		this.y0 = this.arg_y0.getInt(ictx.height);
		this.x1 = this.arg_x1.getInt(ictx.width);
		this.y1 = this.arg_y1.getInt(ictx.height);
		if(this.x0 < 0) this.x0 = 0;
		if(this.y0 < 0) this.y0 = 0;
		if(this.x1 < 0) this.x1 = ictx.width;
		if(this.y1 < 0) this.y1 = ictx.height;
		ascending = (this.x1 > this.x0) ^ (this.y1 > this.y0); 
		buildTable();
		this.octx = new ContextOut(ictx.depth, this.tx.length);
		return this.octx;
	}
	
	@Override
	public void f(Point p) {
		p.sy = this.ty[p.ty];
		p.sx = this.tx[p.ty];
		// TODO 4 quadrants ?
		if(ascending)
			p.sz = ((double)p.tx / this.octx.width) * this.ictx.depth;
		else
			p.sz = this.ictx.depth - ((double)p.tx / this.octx.width) * this.ictx.depth;
	}

	@Override
	public String toString() {
		return "LinearDDA [x0=" + this.x0 + ", y0=" + this.y0 + ", x1=" + this.x1 + ", y1=" + this.y1 + "]";
	}

}
