package slitscan.func;

import java.io.File;

import noise.ImprovedNoise;
import slitscan.Orbit;

public class PerlinNoiseFunc extends Func implements Orbit {
	private static final double STEP = 0.01;
	static final double EPSILON = 0.0001;
	
	private final ImprovedNoise noise;
	
	private final double xFrequency;
	private final double yFrequency;
	private final double phase;
	private final long seed;
	private double min = 0;
	private double max = 1;
	private double range;

	public PerlinNoiseFunc(File[] in_files, String[] args) {
		this.seed = checkArg(args, 0) ? Long.parseLong(args[0]) : 1512;
		this.xFrequency = checkArg(args, 1) ? Double.parseDouble(args[1]) : 1.0;
		this.yFrequency = checkArg(args, 2) ? Double.parseDouble(args[2]) : 1.0;
		this.phase = checkArg(args, 3) ? Double.parseDouble(args[3]) : 0.0;
		this.noise = new ImprovedNoise();
		this.noise.seed(this.seed);
		
		this.min = this.noise.noise(0.0, 0.0, this.phase);
		this.max = this.min;
		double x = 0.0;
		while (x < 1.0) {
			double y = 0.0;
			while (y < 1.0) {
				double v = this.noise.noise(this.xFrequency*x, this.yFrequency*y, this.phase);
				if (v < this.min)
					this.min = v;
				if (v > this.max)
					this.max = v;
				y += STEP;
			}
			x += STEP;
		}
		this.range = this.max - this.min;
		assert this.range > 0.0;
	}

	public double g_raw(double x, double y) {
		double v = this.noise.noise(this.xFrequency*x, this.yFrequency*y, this.phase);
		return (v-this.min) / this.range;
	}

	public double g(double x, double y) {
		double v = g_raw(x, y);
		if(v < 0.0) return 0.0;
		if(v >= 1.0) return 1.0 - EPSILON;
		return v;
	}

	@Override
	public String toString() {
		return "PerlinNoiseFunc[xFrequency=" + this.xFrequency + ",yFrequency=" + this.yFrequency + ",phase=" + this.phase + ",seed=" + this.seed + "]";
	}

}
