package slitscan.concombine;

import java.awt.image.BufferedImage;
import java.awt.image.Raster;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.concurrent.CountDownLatch;

import javax.imageio.ImageIO;

import slitscan.CombineRoot;
import slitscan.ConCombine;

class TileMMACombiner extends Thread {

	private final int[] argbArray;
	private final int max_pixel[];
	private final int max_gray[];
	private final int min_pixel[];
	private final int min_gray[];
	private final int r_sum[];
	private final int g_sum[];
	private final int b_sum[];
	private final int in_width;
	private final int in_height;
	private final int start_x;
	private final int len_x;
	private final CountDownLatch scanning;

	public TileMMACombiner(int in_width, int in_height, int sx, int lx, CountDownLatch scanning, int[] argbArray, int[] max_pixel, int[] max_gray, int[] min_pixel, int[] min_gray, int[] r_sum, int[] g_sum, int[] b_sum) {
		this.in_width = in_width;
		this.in_height = in_height;
		this.start_x = sx;
		this.len_x = lx;
		this.scanning = scanning;
		this.argbArray = argbArray;
		this.max_pixel = max_pixel;
		this.max_gray = max_gray;
		this.min_pixel = min_pixel;
		this.min_gray = min_gray;
		this.r_sum = r_sum;
		this.g_sum = g_sum;
		this.b_sum = b_sum;
		
//		System.out.println(this);
	}

	@Override
	public void run() {
		try {
			for(int ix=0; ix<this.len_x; ++ix) {
				for(int iy=0; iy<in_height; ++iy) {
					final int po = iy*in_width + ix + this.start_x;
					final int pi = 3*po;
					int gray = argbArray[pi] + argbArray[pi+1] + argbArray[pi+2];
					if(this.max_gray[po] < gray) {
						this.max_gray[po] = gray;
						this.max_pixel[po] = ((argbArray[pi]   & 0xff) << 16) 
								    | ((argbArray[pi+1] & 0xff) << 8)
								    | ((argbArray[pi+2] & 0xff))
								    | 0xff000000;
					}
					if(this.min_gray[po] > gray) {
						this.min_gray[po] = gray;
						this.min_pixel[po] = ((argbArray[pi]   & 0xff) << 16) 
								    | ((argbArray[pi+1] & 0xff) << 8)
								    | ((argbArray[pi+2] & 0xff))
								    | 0xff000000;
					}
					this.r_sum[po] += argbArray[pi];
					this.g_sum[po] += argbArray[pi+1];
					this.b_sum[po] += argbArray[pi+2];
				}
			}
		} finally {
			this.scanning.countDown();
		}
	}

	@Override
	public String toString() {
		return "TileCombiner [in_width=" + in_width + ", in_height=" + in_height + ", start_x=" + start_x + ", len_x=" + len_x + ", scanning=" + scanning + "]";
	}
}

public class MMA extends CombineRoot implements ConCombine {
	final static int ap = Runtime.getRuntime().availableProcessors();

	File[] in_files;
	int max_pixel[];
    int max_gray[];
	int min_pixel[];
    int min_gray[];
    int r_sum[];
    int g_sum[];
    int b_sum[];
	int count;

	public MMA(File[] in_files, String[] args) {
		super();
		this.in_files = in_files;
	}

	@Override
	public void pre() throws IOException {
		int size = this.octx.width * this.octx.height;
		this.max_pixel = new int[size];
		this.max_gray = new int[size];
		this.min_pixel = new int[size];
		this.min_gray = new int[size];
		Arrays.fill(min_gray, Integer.MAX_VALUE);
		this.r_sum = new int[size];
		this.g_sum = new int[size];
		this.b_sum = new int[size];
		count = 0;
	}

	@Override
	public void combine() throws IOException {
		int[] argbArray = null;
		for(int fx=0; fx<this.in_files.length; ++fx) {
			try {
//				System.out.println("--- " + this.in_files[fx]);
				BufferedImage in_image = ImageIO.read(this.in_files[fx]);
				int in_width = in_image.getWidth();
				int in_height = in_image.getHeight();
				if(argbArray == null)
					argbArray = new int[3*in_width*in_height];
				Raster raster = in_image.getData();
				raster.getPixels(0, 0, in_width, in_height, argbArray);
				
				TileMMACombiner[] parallelCombiner = new TileMMACombiner[ap];
				CountDownLatch combining = new CountDownLatch(parallelCombiner.length);
				int per = in_width / ap;
				int rest = in_width - per * (ap-1);
				int sx1=0;
				for(; sx1<parallelCombiner.length-1; ++sx1) {
					parallelCombiner[sx1] = new TileMMACombiner(in_width, in_height, sx1*per, per, combining,
							                                 argbArray,
							                                 max_pixel, max_gray,
							                                 min_pixel, min_gray,
							                                 r_sum, g_sum, b_sum);
					parallelCombiner[sx1].start();
				}
				parallelCombiner[sx1] = new TileMMACombiner(in_width, in_height, sx1*per, rest, combining,
									                        argbArray,
									                        max_pixel, max_gray,
									                        min_pixel, min_gray,
									                        r_sum, g_sum, b_sum);
				parallelCombiner[sx1].start();

				try {
					combining.await();
				} catch (InterruptedException e) {
					e.printStackTrace();
				}

				++this.count;
				in_image.flush();
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	@Override
	public int[] post(int i) throws IOException {
		switch(i) {
		case 0: return max_pixel;
		case 1: return min_pixel;
		case 2: {
			int pixels[] = new int[this.octx.width * this.octx.height];
			for (int pi = 0; pi < pixels.length; pi++) {
				pixels[pi] =   (((this.r_sum[pi] / this.count) & 0xff) << 16) 
							 | (((this.g_sum[pi] / this.count) & 0xff) << 8)
							 | (((this.b_sum[pi] / this.count) & 0xff))
							 | 0xff000000;
			}
			return pixels;
		}
		}
		return null;
	}

	@Override
	public String name(int i) {
		switch(i) {
		case 0: return "max";
		case 1: return "min";
		case 2: return "avr";
		}
		return "";
	}

	@Override
	public String toString() {
		return "MMA";
	}

}
