package slitscan.concombine;

import java.awt.image.BufferedImage;
import java.awt.image.Raster;
import java.io.File;
import java.io.IOException;
import java.util.concurrent.CountDownLatch;

import javax.imageio.ImageIO;

import slitscan.CombineRoot;
import slitscan.ConCombine;

class TileStopsCombiner extends Thread {

	private final int[] argbArray;
	private double weight = 1.0;
	private final double r_sum[];
	private final double g_sum[];
	private final double b_sum[];
	private final int in_width;
	private final int in_height;
	private final int start_x;
	private final int len_x;
	private final CountDownLatch scanning;

	public TileStopsCombiner(int in_width, int in_height, int sx, int lx, CountDownLatch scanning, int[] argbArray, double weight, double[] r_sum, double[] g_sum, double[] b_sum) {
		this.in_width = in_width;
		this.in_height = in_height;
		this.start_x = sx;
		this.len_x = lx;
		this.scanning = scanning;
		this.argbArray = argbArray;
		this.weight = weight;
		this.r_sum = r_sum;
		this.g_sum = g_sum;
		this.b_sum = b_sum;
	}

	@Override
	public void run() {
		try {
			for(int ix=0; ix<this.len_x; ++ix) {
				for(int iy=0; iy<in_height; ++iy) {
					final int po = iy*in_width + ix + this.start_x;
					final int pi = 3*po;
					this.r_sum[po] += this.weight * argbArray[pi];
					this.g_sum[po] += this.weight * argbArray[pi+1];
					this.b_sum[po] += this.weight * argbArray[pi+2];
				}
			}
		} finally {
			this.scanning.countDown();
		}
	}

	@Override
	public String toString() {
		return "TileStopsCombiner [weight=" + weight + ", in_width=" + in_width + ", in_height=" + in_height + ", start_x=" + start_x + ", len_x=" + len_x + ", scanning=" + scanning + "]";
	}

}

public class Flash extends CombineRoot implements ConCombine {
	final static int ap = Runtime.getRuntime().availableProcessors();

	File[] in_files;
    double r_sum[];
    double g_sum[];
    double b_sum[];
	int count;

	private final int nstops;
	private final int width;
	private final double weight;
	private final int[] lower;
	private final int[] upper;

	public Flash(File[] in_files, String[] args) {
		super();
		this.in_files = in_files;
		this.nstops = checkArg(args, 0) ? Integer.parseInt(args[0]) : -1;
		this.width  = Math.max(0, checkArg(args, 1) ? Integer.parseInt(args[1]) : 1);
		this.weight = checkArg(args, 2) ? Double.parseDouble(args[2]) : 1.0;
		
		if(nstops > 0) {
			this.lower = new int[nstops];
			this.upper = new int[nstops];
			int pos = 0;
			int delta = in_files.length / (nstops + 1);
			int wl = this.width / 2; 
			int wu = this.width - wl; 
			for(int si=0; si<nstops; ++si) {
				pos += delta;
				this.lower[si] = pos - wl;
				this.upper[si] = pos + wu;
			}
		}
		else {
			this.lower = null;
			this.upper = null;
		}
	}

	boolean inside(int fx) {
		for(int si=0; si<this.nstops; ++si) {
			if(fx > this.lower[si] && fx <= this.upper[si])
				return true;
		}
		return false;
	}
	
	@Override
	public void pre() throws IOException {
		int size = this.octx.width * this.octx.height;
		this.r_sum = new double[size];
		this.g_sum = new double[size];
		this.b_sum = new double[size];
		count = 0;
	}

	@Override
	public void combine() throws IOException {
		int[] argbArray = null;
		int flen = this.in_files.length;

		for(int fx=0; fx<flen; ++fx) {
			try {
				double w;
				if(inside(fx)) {
					w = this.weight;
					this.count += this.weight;
				}
				else {
					w = 1.0;
					this.count += 1;
				}
				
				BufferedImage in_image = ImageIO.read(this.in_files[fx]);
				int in_width = in_image.getWidth();
				int in_height = in_image.getHeight();
				if(argbArray == null)
					argbArray = new int[3*in_width*in_height];
				Raster raster = in_image.getData();
				raster.getPixels(0, 0, in_width, in_height, argbArray);
				
				TileStopsCombiner[] parallelCombiner = new TileStopsCombiner[ap];
				CountDownLatch combining = new CountDownLatch(parallelCombiner.length);
				int per = in_width / ap;
				int rest = in_width - per * (ap-1);
				int sx1=0;
				for(; sx1<parallelCombiner.length-1; ++sx1) {
					parallelCombiner[sx1] = new TileStopsCombiner(in_width, in_height, sx1*per, per, combining,
							                                      argbArray,
							                                      w,
							                                      r_sum, g_sum, b_sum);
					parallelCombiner[sx1].start();
				}
				parallelCombiner[sx1] = new TileStopsCombiner(in_width, in_height, sx1*per, rest, combining,
									                          argbArray,
									                          w,
									                          r_sum, g_sum, b_sum);
				parallelCombiner[sx1].start();

				try {
					combining.await();
				} catch (InterruptedException e) {
					e.printStackTrace();
				}

				in_image.flush();
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	@Override
	public int[] post(int i) throws IOException {
		if(i != 0)
			return null;

		int pixels[] = new int[this.octx.width * this.octx.height];
		for (int pi = 0; pi < pixels.length; pi++) {
			pixels[pi] =   (((int)(this.r_sum[pi] / this.count) & 0xff) << 16) 
						 | (((int)(this.g_sum[pi] / this.count) & 0xff) << 8)
						 | (((int)(this.b_sum[pi] / this.count) & 0xff))
						 | 0xff000000;
		}
		return pixels;
	}

	@Override
	public String name(int i) {
		return "";
	}

	@Override
	public String toString() {
		return "Flash [nstops=" + nstops + ", width=" + width + ", weight=" + weight + "]";
	}

}
