package slitscan;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicInteger;

import javax.imageio.ImageIO;

import filesort.FileWriter;


class TileScanner extends Thread {
	final int start_x, len_x, start_y, len_y;
	final Orbit orbit1;
	final Orbit orbit2;
	final double mix;
    final ContextIn ictx;
    final ContextOut octx;
	final CountDownLatch scanning;
	final CountDownLatch merging;
	int count;
	Point anchor;

	public TileScanner(final Orbit orbit1,
			           final Orbit orbit2,
			           final double mix,
			           final ContextIn ictx,
					   final ContextOut octx,
			           final int lx, final int ux, final int ly, final int uy,
			           final CountDownLatch scanning,
			           final CountDownLatch merging) {
		super();
		this.orbit1 = orbit1;
		this.orbit2 = orbit2;
		this.mix = mix;
		this.ictx = ictx;
		this.octx = octx;
		this.start_x = lx;
		this.len_x = ux;
		this.start_y = ly;
		this.len_y = uy;
		this.scanning = scanning;
		this.merging = merging;
	}

	@Override
	public void run() {
		try {
			this.count = 0;
			for(int ix = 0; ix < this.len_x; ++ix) {
				for(int iy = 0; iy < this.len_y; ++iy) {
					Point p1 = new Point(this.anchor, this.start_x+ix, this.start_y+iy);
					this.orbit1.f(p1);
					if(orbit2 != null) {
						Point p2 = new Point(this.anchor, this.start_x+ix, this.start_y+iy);
						this.orbit2.f(p2);
						p1.sx = (int) (mix * p1.sx + (1.0-mix) * p2.sx); 
						p1.sy = (int) (mix * p1.sy + (1.0-mix) * p2.sy); 
						p1.sz = mix * p1.sz + (1.0-mix) * p2.sz; 
					}
					this.anchor = p1;
					++this.count;
				}
			}
		} finally {
			this.scanning.countDown();
		}

		try {
			this.merging.await();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
	}	
}

class TileMapper extends Thread {
	final Point[] points;
    final File[] in_files;
	final int pixels[];
	final CountDownLatch mapping;
	final AtomicInteger load_counter;
	final int start;
	final int len;
	final int stride;

	public TileMapper(final Point[] points,
			          File[] in_files,
			          final int pixels[],
			          final int start,
			          final int len,
			          final int stride,
			          final CountDownLatch mapping,
			          AtomicInteger load_counter) {
		super();
		this.points = points;
		this.in_files = in_files;
		this.pixels = pixels;
		this.start = start;
		this.len = len;
		this.stride = stride;
		this.mapping = mapping;
		this.load_counter = load_counter;
	}

	@Override
	public void run() {
		try {
			int current = Integer.MIN_VALUE;
			BufferedImage in_image = null;
			int width = 0;
			int height = 0;
			for(int px=0; px<this.len; ++px) {
				Point point = this.points[px+this.start];
				if(current != (int)point.sz) {
					try {
						in_image = ImageIO.read(this.in_files[clamp((int)point.sz, this.in_files.length)]);
						width = in_image.getWidth();
						height = in_image.getHeight();
						current = (int)point.sz;
						this.load_counter.incrementAndGet();
					} catch (IOException e) {
						in_image = null;
						e.printStackTrace();
					}
				}
				if(in_image != null) {
					this.pixels[point.ty*this.stride + point.tx] = in_image.getRGB(clamp(point.sx, width),
							                                                       clamp(point.sy, height));
				}
			}
		} finally {
			this.mapping.countDown();
		}

	}

	static int clamp(final int val, final int limit) {
		if(val < 0) return 0;
		if(val >= limit) return limit-1;
		return val;
	}		
}

public class OrbitScanner {
	final static int ap = Runtime.getRuntime().availableProcessors();
	
	public static void work(File[] in_files, Orbit orbit1, Orbit orbit2, double mix, String range_label, String tracker_fn) {
		System.out.println("cores  = " + ap);
		assert ap > 0 : ap;
		assert in_files != null && in_files.length > 0;
		try {
	        System.out.println("images = " + in_files.length);
		    long t0 = System.nanoTime();
			ContextIn ictx = new ContextIn(in_files);
	        System.out.println("in     = " + ictx);
			orbit1.setTracker(ExternalTracker.buildTracker(tracker_fn, ictx));
			ContextOut octx = orbit1.buildContextOut(ictx);
			if(orbit2 != null) orbit2.buildContextOut(ictx);
	        System.out.println("out    = " + octx);
	        long t1 = System.nanoTime();
	        System.out.println("cntx   = " + ((t1-t0) / 1000000.0));

			Point[] points = scan(orbit1, orbit2, mix, ictx, octx);
	        long t2 = System.nanoTime();
	        System.out.println("scan   = " + ((t2-t1) / 1000000.0));
	        
			Arrays.sort(points);
	        long t3 = System.nanoTime();
	        System.out.println("sort   = " + ((t3-t2) / 1000000.0));

			int pixels[] = map(points, in_files, octx);
	        long t4 = System.nanoTime();
	        System.out.println("map    = " + ((t4-t3) / 1000000.0));
			
			BufferedImage out_iamge = new BufferedImage(octx.width, octx.height, BufferedImage.TYPE_INT_ARGB);
			out_iamge.setRGB(0, 0, octx.width, octx.height, pixels, 0, octx.width);
			SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd-HHmmss");
			String m = "";
			if(orbit2 != null) {
				m = "(mix=" + mix + ")" + orbit2;
			}
			String output_name = application.SL.out_path + "orbit-" + orbit1 + m + range_label  + "-" + sdf.format(new Date()) + ".png";
			output_name = output_name.replace(" ", "");
		    ImageIO.write(out_iamge, "png", new File(output_name));
	        long t5 = System.nanoTime();
	        System.out.println("write  = " + ((t5-t4) / 1000000.0));
	        System.out.println("output = " + output_name);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	private static Point[] scan(Orbit orbit1, Orbit orbit2, double mix, ContextIn ictx, ContextOut octx) {
		assert orbit1 != null;
		TileScanner[] parallelScanner = new TileScanner[ap];
		CountDownLatch scanning = new CountDownLatch(parallelScanner.length);
		CountDownLatch[] merging = new CountDownLatch[parallelScanner.length];
		int per = octx.width / ap;
		int rest = octx.width - per * (ap-1);
		int sx1=0;
		for(; sx1<parallelScanner.length-1; ++sx1) {
			merging[sx1] = new CountDownLatch(1);
			parallelScanner[sx1] = new TileScanner(orbit1, orbit2, mix, ictx, octx, sx1*per, per, 0, octx.height, scanning, merging[sx1]);
			parallelScanner[sx1].start();
		}
		sx1 = parallelScanner.length-1;
		merging[sx1] = new CountDownLatch(1);
		parallelScanner[sx1] = new TileScanner(orbit1, orbit2, mix, ictx, octx, sx1*per, rest, 0, octx.height, scanning, merging[sx1]);
		parallelScanner[sx1].start();
		try {
			scanning.await();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}

		int count = 0;
		for(int sx2=0; sx2<parallelScanner.length; ++sx2) {
			count += parallelScanner[sx2].count;
		}
		
		Point[] points = new Point[count];
		int px = 0;
		for(int sx3=0; sx3<parallelScanner.length; ++sx3) {
			Point p = parallelScanner[sx3].anchor;
			while(p != null) {
				points[px++] = p;
				p = p.next;
			}
			merging[sx3].countDown();
		}
		boolean normalize = orbit1 instanceof slitscan.func.Func || orbit2 instanceof slitscan.func.Func; 
		if(normalize) {
			final boolean FUNC_DUMP = false;
			StringBuilder out;
			if(FUNC_DUMP) {
				out = new StringBuilder();
			}
			double maxsz = Double.MIN_VALUE;
			double minsz = Double.MAX_VALUE;
			for (int pi = 0; pi < points.length; pi++) {
				if(points[pi].sz > maxsz) {
					maxsz = points[pi].sz;
				}
				if(points[pi].sz < minsz) {
					minsz = points[pi].sz;
				}
			}
			double diffsz = maxsz - minsz;
			double factorsz = (double)ictx.depth / diffsz;
			for (int pi = 0; pi < points.length; pi++) {
				points[pi].sz = factorsz * (points[pi].sz-minsz);
				if(FUNC_DUMP) {
				//TODO	((slitscan.func.Func) orbit).dump(out, points[pi], ictx, octx);
				}
			}			
			if(FUNC_DUMP) {
				FileWriter.writeFile(application.SL.out_path + "dump", orbit1.getClass().getSimpleName()+".csv", out);
			}
		}
		return points;
	}

	private static int[] map(Point[] points, File[] in_files, ContextOut ctx) {
		assert points != null;
		assert in_files != null;
		int pixels[] = new int[ctx.width * ctx.height];
		int ap2 = ap + ap / 4;
		int per = points.length / ap2;
		int rest = points.length - per * (ap2-1);
		TileMapper[] parallelMapper = new TileMapper[ap2];
		CountDownLatch mapping = new CountDownLatch(parallelMapper.length);
		AtomicInteger load_counter = new AtomicInteger(0);
		for(int mx=0; mx<parallelMapper.length-1; ++mx) {
			parallelMapper[mx] = new TileMapper(points, in_files, pixels, mx*per, per, ctx.width, mapping, load_counter);
			parallelMapper[mx].start();
		}
		parallelMapper[parallelMapper.length-1] = new TileMapper(points, in_files, pixels, (ap2-1)*per, rest, ctx.width, mapping, load_counter);
		parallelMapper[parallelMapper.length-1].start();
		try {
			mapping.await();
	        System.out.println("loads  = " + load_counter.get());
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		return pixels;
	}	

}
