package slitscan;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;


class FramePosition {
	final int frame;
	double x, y;
	int occurrences;

	@Override
	public String toString() {
		return "FramePosition [frame=" + this.frame + ", x=" + this.x + ", y=" + this.y + ", occurrences=" + this.occurrences + "]";
	}

//	FramePosition() {
//		this.frame = 0;
//		this.x = 0;
//		this.y = 0;
//		this.occurrences = 0;
//	}
	
	FramePosition(int frame, double x, double y) {
		this.frame = frame;
		this.x = x;
		this.y = y;
		this.occurrences = 1;
	}

	public FramePosition(FramePosition framePosition) {
		this.frame = framePosition.frame;
		this.x = framePosition.x;
		this.y = framePosition.y;
		this.occurrences = 1;
	}
}

public class ExternalTracker {
//	int frame_length;
	ContextIn ictx;
	FramePosition[] frames;
	
	ExternalTracker(ContextIn ictx) {
		this.ictx = ictx;
		this.frames = new FramePosition[this.ictx.depth];
	}
	
	private int clamp(int f) {
		if(f < 0) return 0;
		if(f >= this.frames.length) return this.frames.length-1;
		return f;
	}
	
	public int xAt(int frame, int factor) {
		return (int) (this.frames[clamp(frame)].x * factor);
	}

//	public int xNearInt(int frame) {
//		double v = (double) frame / (double) this.ictx.width;
//		double min = Double.MAX_VALUE;
//		int mix = 0;
//		for (int fx = 0; fx < this.ictx.depth; ++fx) {
//			double diff = Math.abs(this.frames[fx].x - v);
//			if (diff < min) {
//				min = diff;
//				mix = this.frames[fx].frame;
//			}
//		}
//		return mix;
//	}

	public int xNear(double frame) {
		double min = Double.MAX_VALUE;
		int at = 0;
		for (int ff = 0; ff < this.frames.length; ++ff) {
			double diff = Math.abs(this.frames[ff].x - frame);
			if (diff < min) {
				min = diff;
				at = this.frames[ff].frame;
			}
		}
		return at;
	}

	public int yNear(double frame) {
		double min = Double.MAX_VALUE;
		int at = 0;
		for (int ff = 0; ff < this.frames.length; ++ff) {
			double diff = Math.abs(this.frames[ff].y - frame);
			if (diff < min) {
				min = diff;
				at = this.frames[ff].frame;
			}
		}
		return at;
	}

	public int yAt(int frame, int factor) {
		return (int) (this.frames[clamp(frame)].y * factor);
	}

	private void readTracker(String tracker_fn) {
		// read all lines from tracker file
		// frame number, x position, y position
		try (BufferedReader br = new BufferedReader(new FileReader(tracker_fn))) {
			String line = br.readLine();
			while (line != null) {
				// skip lines containing the tracker name <bpy_struct, MovieTrackingTrack("Track.001")>
				if (!(line.length() < 1 || line.startsWith("<"))) {
					String[] s = line.split(" ");
					if (s.length == 3) {
						int f = Integer.parseInt(s[0]);
						if (f < this.ictx.depth) {
							double x = Double.parseDouble(s[1]);
							double y = Double.parseDouble(s[2]);
							if (this.frames[f] == null) {
								this.frames[f] = new FramePosition(f, x, y);
							} else {
								// there may be some lines having multible trackers 
								this.frames[f].x += x;
								this.frames[f].y += y;
								this.frames[f].occurrences += 1;
							}
						}
					}
				}
				line = br.readLine();
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
		// calculate the average position of multiple points per frame
		for(int ff=0; ff<this.ictx.depth; ++ff) {
			if(this.frames[ff] != null && this.frames[ff].occurrences > 1) {
				this.frames[ff].x /= this.frames[ff].occurrences;
				this.frames[ff].y /= this.frames[ff].occurrences;
				this.frames[ff].occurrences = 1;
			}
		}
	}

	private void intrepolate() {
		// interpolate positions for frames not defined in the tracker file
		int gap_start = -1;
		int gap_end = -1;
		for(int ff=0; ff<this.ictx.depth; ++ff) {
			if(gap_start == -1 && this.frames[ff] == null) {
				gap_start = ff;
			}
			else if(gap_start != -1 && gap_end != -1 && this.frames[ff] != null) {
				gap_end = ff;
				double dx = (this.frames[gap_end].x - this.frames[gap_start-1].x) / (gap_end - gap_start + 1);
				double dy = (this.frames[gap_end].y - this.frames[gap_start-1].y) / (gap_end - gap_start + 1);
				System.out.println(gap_start + " " + gap_end);
				int step = 1;
				System.out.println((gap_start-1) + " " + this.frames[gap_start-1]);
				for(int gx = gap_start; gx < gap_end; ++gx) {
					this.frames[gx] = new FramePosition(this.frames[gap_start-1].frame+step,
							                            this.frames[gap_start-1].x+dx*step,
							                            this.frames[gap_start-1].y+dy*step);
					System.out.println(gx + " " + this.frames[gx]);
					++step;
				}
				System.out.println((gap_end) + " " + this.frames[gap_end]);
				gap_start = -1;
				gap_end = -1;
			}
		}
		for(int ff=1; ff<this.ictx.depth; ++ff) {
			if (this.frames[ff] == null && this.frames[ff-1] != null) {
				this.frames[ff] = new FramePosition(this.frames[ff-1].frame+1,
						                            this.frames[ff-1].x,
						                            this.frames[ff-1].y);
			}
		}
		System.out.println((this.frames.length-1) + " " + this.frames[this.frames.length-1]);
	}

	private void checkConsistency() {
		assert this.frames.length == this.ictx.depth;
		for(int ff=0; ff<this.frames.length; ++ff) {
			assert this.frames[ff] != null;
			assert this.frames[ff].frame == ff;
		}
	}
	
	public void doSmoothing() {
		FramePosition[] sf = new FramePosition[this.frames.length];
		for(int ff=0; ff<this.frames.length; ++ff) {
			sf[ff] = new FramePosition(this.frames[ff]);
		}
		for(int ff=2; ff<this.frames.length-2; ++ff) {
			double x =(   this.frames[ff-2].x
					+ 2 * this.frames[ff-1].x
					+ 4 * this.frames[ff].x
					+ 2 * this.frames[ff+1].x
					+     this.frames[ff+2].x) / 10.0;
			double y =(   this.frames[ff-2].y
					+ 2 * this.frames[ff-1].y
					+ 4 * this.frames[ff].y
					+ 2 * this.frames[ff+1].y
					+     this.frames[ff+2].y) / 10.0;
			sf[ff] = new FramePosition(ff, x, y);
		}
		this.frames = sf;
	}

	public void normalize() {
		double xmin = this.frames[0].x;
		double xmax = this.frames[0].x;
		double ymin = this.frames[0].y;
		double ymax = this.frames[0].y;
		for (int ff = 1; ff < this.frames.length; ++ff) {
			if (this.frames[ff].x < xmin) {
				xmin = this.frames[ff].x;
			}
			if (this.frames[ff].x > xmax) {
				xmax = this.frames[ff].x;
			}
			if (this.frames[ff].y < ymin) {
				ymin = this.frames[ff].y;
			}
			if (this.frames[ff].y > ymax) {
				ymax = this.frames[ff].y;
			}
		}
		final double xrange = xmax-xmin;
		final double yrange = ymax-ymin;
		for (int ff = 0; ff < this.frames.length; ++ff) {
			this.frames[ff].x = (this.frames[ff].x - xmin) / xrange;
			this.frames[ff].y = (this.frames[ff].y - ymin) / yrange;
		}
	}

	void build0(String tracker_fn, ExternalTracker tracker) {
		readTracker(tracker_fn);
		intrepolate();
		checkConsistency();
//		doSmoothing();
		checkConsistency();
	}

	static ExternalTracker buildTracker(String tracker_fn, ContextIn ictx) {
		if(tracker_fn != null && tracker_fn.length() > 0) {
			ExternalTracker tracker = new ExternalTracker(ictx);
			tracker.build0(tracker_fn, tracker);
			return tracker;
		}
		return null;
	}
}
