package slitscan;

import java.awt.image.BufferedImage;
import java.awt.image.WritableRaster;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.concurrent.CountDownLatch;
import javax.imageio.ImageIO;


class AnalysisMapper extends Thread {
	final WritableRaster ref;
	final WritableRaster compare;
	final CountDownLatch mapping;
	final int start;
	final int len;
	final int width;
	final int[] ref_line;
	final int[] compare_line;
	final int ref_bands;
	final int compare_bands;
	long diff;

	public AnalysisMapper(final WritableRaster ref,
						  final WritableRaster compare,
				          final int start,
				          final int len,
				          final CountDownLatch mapping) {
		this.ref = ref;
		this.compare = compare;
		this.start = start;
		this.len = len;
		this.mapping = mapping;
		this.width = ref.getWidth();
		this.ref_bands = ref.getNumBands();
		this.ref_line = new int[ref_bands*width*len];
		this.compare_bands = compare.getNumBands();
		this.compare_line = new int[compare_bands*width*len];
	}

	@Override
	public void run() {
		diff = 0;
		try {
			ref.getPixels(0, 0, width, len, ref_line);
			compare.getPixels(0, 0, width, len, compare_line);
			for(int lx=0; lx < len; ++lx) {
				for(int ix=0; ix<width; ++ix) {
					int rx = ref_bands * (lx * width + ix);
					int r1 = ref_line[rx];
					int g1 = ref_line[rx+1];
					int b1 = ref_line[rx+2];				
					int cx = compare_bands * (lx * width + ix);
					int r2 = compare_line[cx];
					int g2 = compare_line[cx+1];
					int b2 = compare_line[cx+2];

					int rd = Math.abs(r1-r2);
					int gd = Math.abs(g1-g2);
					int bd = Math.abs(b1-b2);
					
					diff += rd+gd+bd;
				}
			}
		} finally {
			this.mapping.countDown();
		}
//		System.out.println("start " + start + " len " + len + " diff " + diff);
	}
}

class DiffDesc {
	public DiffDesc(File f, long d) {
		this.f = f;
		this.d = d;
	}
	final File f;
	final long d;
}

class DiffDescComparator implements Comparator<DiffDesc> {
	  @Override
	  public int compare(DiffDesc diffDesc1, DiffDesc diffDesc2) {
		  return (diffDesc1.d < diffDesc2.d) ? -1
				                             : (diffDesc1.d > diffDesc2.d) ? 1 : 0;
	}
}

public class AnalysisScanner {
	final static int ap = Runtime.getRuntime().availableProcessors();
	
	public static void work(File[] in_files) {
		System.out.println("cores  = " + ap);
		assert ap > 0 : ap;
		assert in_files != null && in_files.length > 1;
		ArrayList<DiffDesc> diffs = new ArrayList<>();
		try {
	        System.out.println("images = " + in_files.length);
		    long t0 = System.nanoTime();
			BufferedImage ref_image = ImageIO.read(in_files[0]);
			WritableRaster ref = ref_image.getRaster();
	        for (int i = 1; i < in_files.length; ++i) {
				BufferedImage in_image = ImageIO.read(in_files[i]);
				long diff = map(ref, in_image);
				diffs.add(new DiffDesc(in_files[i], diff));
			}
	        long t1 = System.nanoTime();
	        System.out.println("analy  = " + ((t1-t0) / 1000000.0));
	        long t2 = System.nanoTime();
	        Collections.sort(diffs, new DiffDescComparator());
	        System.out.println("sort   = " + ((t2-t1) / 1000000.0));
	        for(int bx=0; bx<20 && bx<diffs.size(); ++bx)
	        	System.out.println("best " + bx + " = " + diffs.get(bx).f + " diff = " + diffs.get(bx).d);
			
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	private static long map(WritableRaster ref, BufferedImage in_image) {
		int per = ref.getHeight() / ap;
		int rest = ref.getHeight() - per * (ap-1);
		AnalysisMapper[] parallelMapper = new AnalysisMapper[ap];
		CountDownLatch mapping = new CountDownLatch(parallelMapper.length);
		for(int mx=0; mx<parallelMapper.length-1; ++mx) {
			parallelMapper[mx] = new AnalysisMapper(ref, in_image.getRaster(), mx*per, per, mapping);
			parallelMapper[mx].start();
		}
		parallelMapper[parallelMapper.length-1] = new AnalysisMapper(ref, in_image.getRaster(), (ap-1)*per, rest, mapping);
		parallelMapper[parallelMapper.length-1].start();
		try {
			mapping.await();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		long difference = 0;
		for(int mx=0; mx<parallelMapper.length; ++mx) {
			difference += parallelMapper[mx].diff;
		}
//        System.out.println("differ = " + difference);
		return difference;
	}

}
