/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.util;

import java.awt.image.*;
import kandid.colorator.*;

/**
 * Counter array, that can be painted to an buffered image.
 * @author thomas jourdan
 */
public class Weight {
  private int[][]     weight;
  private int         xSize;
  private int         ySize;
  protected Colorator colorator;

  /** Creates a new counter array.
   * @param xs
   * @param ys 
   * @param colorator
   */
  public Weight(int xs, int ys, Colorator colorator) {
    if (Debug.enabled) assert colorator != null : "colorator is undefined";
    this.colorator = colorator;
    xSize          = xs;
    ySize          = ys;
    weight         = new int[xSize][ySize];
  }

  /**
   * Increments counter at position xp, yp.
   * @param xp
   * @param yp
   */
  public void inc(int xp, int yp) {
    try {
      ++weight[xp][yp];
    } catch (java.lang.ArrayIndexOutOfBoundsException exc) {
//!!      if(Debug.enabled) Debug.stackTrace(exc);
    }
  }

  /**
   * returns counter value at position xp, yp.
   * @param xp
   * @param yp
   * @return value at position xp, yp.
   */
  public int at(int xp, int yp) {
    try {
      return weight[xp][yp];
    } catch (java.lang.ArrayIndexOutOfBoundsException exc) {
//!!      if(Debug.enabled) Debug.stackTrace(exc);
    }
    return 0;
  }

  /**
   * Calculates the maximum.
   * @return maximum value in the counter array.
   */
  public int getMax() {
    int max = 0;
    try {
      for (int dx = 0; dx < xSize; ++dx) {
        for (int dy = 0; dy < ySize; ++dy) {
          if (weight[dx][dy] > max) {
            max = weight[dx][dy];
          }
        }
      }
    } catch (java.lang.ArrayIndexOutOfBoundsException exc) {
      if(Debug.enabled) Debug.stackTrace(exc);
    }

    return max;
  }

  /**
   * Paints the counter array to an image with the same size.
   * Linear relationship between counter values and colors.
   *
   * @param bufImg a buffered image
   */
  public void toImage(BufferedImage bufImg) {
    double max = (double)getMax();
    ColorF32 cout = new ColorF32();
    try {
      for (int dx = 0; dx < xSize; ++dx) {
        for (int dy = 0; dy < ySize; ++dy) {
          colorator.getColor((double)weight[dx][dy] / max, cout);
          bufImg.setRGB(dx, dy, ColorF32.toRGB8(cout));
        }
      }
    } catch (java.lang.ArrayIndexOutOfBoundsException exc) {
      if(Debug.enabled) Debug.stackTrace(exc);
    }
  }

  /**
   * Paints the counter array to an image with the same size.
   * Logarithmic relationship between counter values and colors.
   *
   * @param bufImg a buffered image
   * @param bufColorF32Img 
   */
  public void toImageLog(BufferedImage bufImg, ColorF32[][] bufColorF32Img) {
    double log_max = Math.log(getMax());
    try {
      for (int dx = 0; dx < xSize; ++dx) {
        for (int dy = 0; dy < ySize; ++dy) {
          int pixelWeight = weight[dx][dy];
          if(pixelWeight <= 0) {
            bufImg.setRGB(dx, dy, 0);
          }
          else {
            ColorF32 colorF32 = new ColorF32();
            colorator.getColor(Math.log((double)weight[dx][dy]) / log_max, colorF32);
            bufImg.setRGB(dx, dy, ColorF32.toRGB8(colorF32));
            ColorF32.copy(colorF32, bufColorF32Img[dy][dx]);
          }
        }
      }
    } catch (java.lang.ArrayIndexOutOfBoundsException exc) {
      if(Debug.enabled) Debug.stackTrace(exc);
    }
  }

  /**
   * This entry point is for testing only.
   *
   * @param args
   */
  public static void main(String[] args) {
    if(Debug.enabled) {
      Weight weight = new Weight(100, 50, new GrayColorator());
      assert weight.getMax() == 0: "t1";
      assert weight.at(0, 0) == 0: "t2";
      weight.inc(0, 0);
      weight.inc(99, 49);
      weight.inc(99, 49);
      assert weight.at(99, 49) == 2: "t3";
      assert weight.at(100, 50) == 0: "t4";
      assert weight.getMax() == 2: "t5";
    }
  }
}
