/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.util;

import java.io.CharArrayWriter;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;

/**
 * Helper functions.
 * @author thomas jourdan
 */
public class Util {

  /**
   * Calculates the value of 2 raised by the power of the given exponent.
   *
   * @param exponent
   *
   * @return 2<sup>exponent</sup>
   */
  public static int powerOf2(int exponent) {
    int result = 1;
    while (exponent > 0) {
      result *= 2;
      --exponent;
    }
    return result;
  }

  /**
   * Converts from polar to matrix.
   *
   * @param in1
   * @param in2
   * @param in3
   * @param in4
   * @param in5
   * @param in6
   *
   * @return
   */
  public static double[] polar2matrix(double in1, double in2, double in3, double in4, double in5, double in6) {
    double       result[] = new double[6];
    final double grad2rad = Math.PI / 180.0;
    double       i3       = in3;
    double       i4       = in4;
    double       i5       = in5;
    double       i6       = in6;
    result[0] =  i5 * Math.cos(grad2rad * i3);     // a
    result[1] = -i6 * Math.sin(grad2rad * i4);     // b
    result[2] =  i5 * Math.sin(grad2rad * i3);     // c
    result[3] =  i6 * Math.cos(grad2rad * i4);     // d
    result[4] =  in1;                              // e
    result[5] =  in2;                              // f
    return result;
  }
  
  /**
   * Method parseInt.
   * @param value the String containing the representation to be parsed
   * @return the integer representation of the string.
   */
  public static int parseInt(String value) {
    value = value.trim();
    if(value.startsWith("0x")) {
      return Integer.parseInt(value.substring(2), 16);
    }
    else {
      return Integer.parseInt(value, 10);
    }
  }

  /**
   * Method parseLong.
   * @param value the String containing the representation to be parsed.
   * @return the long representation of the string.
   */
  public static long parseLong(String value) {
    value = value.trim();
    if(value.startsWith("0x")) {
      return Long.parseLong(value.substring(2), 16);
    }
    else {
      return Long.parseLong(value, 10);
    }
  }

  /**
   * Method parseDouble.
   * @param value the String containing the representation to be parsed.
   * @return the double representation of the string.
   */
  public static double parseDouble(String value) {
    value = value.trim();
    return Double.parseDouble(value);
  }

  /**
   * Method parseBoolean.
   * @param value the String containing the representation to be parsed.
   * @return the boolean representation of the string.
   */
  public static boolean parseBoolean(String value) {
    value = value.trim();
    if(value.startsWith("t"))
      return true;
    if(value.startsWith("T"))
      return true;
    if(value.startsWith("1"))
      return true;
    return false;
  }

  /**
   * convert int value to its hexadecimal representation.
   * @param value 
   * @param minLength 
   * @return the string.
   */
  public static String toHexString(int value, int minLength) {
    StringBuffer ret = new StringBuffer(Integer.toHexString(value));
    while(ret.length() < minLength) {
      ret.insert(0, '0') ;
    }
    return ret.toString();
  }

  /**
   * Method niceTypeName
   *
   * @param typeName
   *
   * @return
   */
  public static String getNiceTypeName(String typeName) {
    int pos = typeName.lastIndexOf('.');
    if (pos >= 0) {
      return typeName.substring(pos + 1);
    }
    return "";
  }

  public static double step(double p1, double p2) {
    if(p2 > p1)
      return 1.0;
    else
      return 0.0;
  }

  public static double clamp(double p1) {
    if(p1 < 0.0)
      return 0.0;
    else if(p1 > 1.0)
      return 1.0;
    else
      return p1;
  }

  public static double repeat_seamless(double p1) {
    double a1 = Math.abs(p1);
    int tile = (int) a1;
    if(tile % 2 == 0)
      return Math.abs(a1 % 1.0);
    else
      return Math.abs((1.0 - (a1 % 1.0)));
  }

  public static double bias(double p1, double p2) {
    return Math.pow(Math.abs(p2), Math.log(Math.abs(p1))/(-0.693147181));
  }

  /**
   * Write population to string.
   * @param model
    */
  public static String marshalToString(Object model, JAXBContext jc) {
    try {
      // create a Marshaller and marshal to a file
      CharArrayWriter arrayWriter = new CharArrayWriter();
      Marshaller m = jc.createMarshaller();
      m.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);
      m.marshal(model, arrayWriter);
      arrayWriter.close();
      return arrayWriter.toString();
    } catch (JAXBException je) {
      Debug.stackTrace(je);
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return "";
  }


}

