/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.util;

/**
 * Central random generator.
 * @author thomas jourdan
 */
public class CentralRandomizer {
  static java.util.Random rnd = new java.util.Random();

  /**
   * set the seed of this random number generator. Use this method only to achieve reproductible results for test cases.
   *
   * @param upperBound the upper bound on the random number to be returned.
   * Must be positive
   */
  public static void setSeed(long seed) {
    rnd.setSeed(seed);
  }

  /**
   * generates random boolean value.
   *
   * @param upperBound the upper bound on the random number to be returned.
   * Must be positive
   *
   * @return random number between 0 (inclusive) and upperBound (inclusive).
   */
  public static boolean getBoolean() {
    return rnd.nextBoolean();
  }

  /**
   * generates uniform distributed random number between 0 (inclusive) and the specified value (inclusive).
   *
   * @param upperBound the upper bound on the random number to be returned.
   * Must be positive
   *
   * @return random number between 0 (inclusive) and upperBound (inclusive).
   */
  public static int getInt(int upperBound) {
    int value  = rnd.nextInt(upperBound+1);
    if(Debug.enabled) assert (value >= 0) && (value <= upperBound): "random value is out of range";
    return value;
  }

  /**
   * generates uniform distributed random number between lowerBound (inclusive) and upperBound (inclusive).
   *
   * @param lowerBound the upper bound on the random number to be returned.
   * @param upperBound the upper bound on the random number to be returned.
   *
   * @return random number between lowerBound (inclusive) and upperBound (inclusive).
   */
  public static int getInt(int lowerBound, int upperBound) {
    if(lowerBound == upperBound)
      return lowerBound;
    if(Debug.enabled) assert lowerBound < upperBound: "lowerBound < upperBound";
    int modulo = upperBound - lowerBound + 1;
    int value  = lowerBound + rnd.nextInt(modulo);
    if(Debug.enabled) assert (value >= lowerBound) && (value <= upperBound): "random value is out of range";
    return value;
  }

  /**
   * generates uniform distributed random float number between 0.0f (inclusive) and the specified value (exclusive).
   *
   * @param upperBound the upper bound on the random number to be returned.
   * Must be positive
   *
   * @return random number between 0.0 (inclusive) and upperBound (exclusive).
   */
  public static float getFloat(float upperBound) {
    float value = upperBound * rnd.nextFloat();
    if(Debug.enabled) assert (value >= 0.0) && (value < upperBound): "random value is out of range";
    return value;
  }

  /**
   * generates uniform distributed random float number between lowerBound (inclusive) and upperBound (exclusive).
   *
   * @param lowerBound the upper bound on the random number to be returned.
   * @param upperBound the upper bound on the random number to be returned.
   *
   * @return random number between lowerBound (inclusive) and upperBound (exclusive).
   */
   public static float getFloat(float lowerBound, float upperBound) {
    if(Debug.enabled) assert lowerBound < upperBound: "lowerBound < upperBound";
    float modulo = upperBound - lowerBound;
    float value  = lowerBound + modulo * rnd.nextFloat();
    if(Debug.enabled) assert (value >= lowerBound) && (value < upperBound): "random value is out of range " + lowerBound + "-" + upperBound + " " + value;
    return value;
  }
  
  /**
   * generates uniform distributed double random number between 0.0 (inclusive) and the specified value (exclusive).
   *
   * @param upperBound the upper bound on the random number to be returned.
   * Must be positive
   *
   * @return random number between 0.0 (inclusive) and upperBound (exclusive).
   */
  public static double getDouble(double upperBound) {
    double value = upperBound * rnd.nextDouble();
    if(Debug.enabled) assert (value >= 0.0) && (value < upperBound): "random value is out of range";
    return value;
  }

  /**
   * generates uniform distributed double random number between lowerBound (inclusive) and upperBound (exclusive).
   *
   * @param lowerBound the upper bound on the random number to be returned.
   * @param upperBound the upper bound on the random number to be returned.
   *
   * @return random number between lowerBound (inclusive) and upperBound (exclusive).
   */
   public static double getDouble(double lowerBound, double upperBound) {
    if(Debug.enabled) assert lowerBound < upperBound: "lowerBound < upperBound";
    double modulo = upperBound - lowerBound;
    double value  = lowerBound + modulo * rnd.nextDouble();
    if(Debug.enabled) assert (value >= lowerBound) && (value < upperBound): "random value is out of range " + lowerBound + "-" + upperBound + " " + value;
    return value;
  }
  
  /**
   * generates uniform distributed random long value.
   *
   * @return random 64 bits value.
   */
  public static long getLong() {
    return rnd.nextLong();
  }
  
  /**
   * generates gaussian distributed random value.
   * @return
   */
  public static double getGaussian() {
    return rnd.nextGaussian();
  }

  /**
   * generates binomial distributed random value.
   * Code is from http://geovistastudio.sourceforge.net/
   * @param n
   * @param p probability
   * @return random number between 0 (inclusive) and n (inclusive).
   */
  public static int getBinomial(int n, double p) {
    int r = 0;
    for (int i = 0; i < n; i++) {
      if (rnd.nextDouble() < p) {
        r++;
      }
    }
    return r;
  }

  /**
   * generates boolean random value.
   * @param probability probability for returning true
   * @return true or false
   */
  public static boolean getBoolean(double probability) {
    return rnd.nextDouble() < probability;
  }
}
