/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.soup.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.StringReader;
import java.lang.reflect.Method;
import java.util.Iterator;
import java.util.List;

import javax.swing.JTree;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.transform.stream.StreamSource;

import kandid.Console;
import kandid.Environment;
import kandid.Fitness;
import kandid.colorator.ColoratorFactory;
import kandid.soup.ChromosomeType;
import kandid.soup.ColoratorType;
import kandid.soup.ContextGene;
import kandid.soup.EntityType;
import kandid.soup.GradientChromosome;
import kandid.soup.HsbFrequenceChromosome;
import kandid.soup.ImageType;
import kandid.soup.IntegerGene;
import kandid.soup.LookUpTableChromosome;
import kandid.soup.Lsys0LImage;
import kandid.soup.LsysD0LImage;
import kandid.soup.LsysProductionGene;
import kandid.soup.ObjectFactory;
import kandid.soup.PopulationType;
import kandid.soup.PovThingImage;
import kandid.soup.PovThingNFImage;
import kandid.soup.RatingBase;
import kandid.soup.ScalarExpressionChromosome;
import kandid.soup.ScalarExpressionImage;
import kandid.soup.TransparentLookUpTableChromosome;
import kandid.soup.VectorExpressionChromosome;
import kandid.soup.VectorExpressionImage;
import kandid.soup.VoronoiChromosome;
import kandid.soup.VoronoiImage;
import kandid.soup.VoronoiTransparentChromosome;
import kandid.soup.VoronoiTransparentImage;
import kandid.soup.map.CloneChromosome;
import kandid.soup.map.MergeChromosome;
import kandid.soup.map.MutateChromosome;
import kandid.soup.map.RandomizeChromosome;
import kandid.soup.map.SetEditorMapping;
import kandid.util.Debug;

/**
 * 
 * @author thomas jourdan
 */
public class SoupFactory {
  private static SoupFactory soupFactory;
  private ObjectFactory objectFactory;
  private CloneChromosome cloner;
  private MergeChromosome merger;
  private MutateChromosome mutator;
  private SetEditorMapping editorFiller;
  private RandomizeChromosome randomizer;
  private JAXBContext jc;

  /**
   * private constructor for SoupFactory.
   */
  private SoupFactory() {
    objectFactory = new ObjectFactory();
    cloner = new CloneChromosome();
    merger = new MergeChromosome();
    mutator = new MutateChromosome();
    editorFiller = new SetEditorMapping();
    randomizer = new RandomizeChromosome();
    try {
      jc = JAXBContext.newInstance("kandid.soup");
    } catch (JAXBException exc) {
      Debug.stackTrace(exc);
    }
  }

  /**
   * Get the soup factory.
   * @return SoupFactory
   */
  public static SoupFactory getSoupFactory() {
    if (soupFactory == null) {
      soupFactory = new SoupFactory();
    }
    return soupFactory;
  }

  /**
   * Find createXXX() method for a given class in the soups object factory 
   * and returns a new instance of this class.
   * @param qualifiedClassName
   * @return new instance of this class
   */
  public Object create(String qualifiedClassName) {
    if (Debug.enabled) assert (qualifiedClassName != null) && (qualifiedClassName.length() > 0) : "qualifiedClassName not set";
    try {
      int gpx = qualifiedClassName.lastIndexOf('.');
      String className = qualifiedClassName.substring(gpx + 1);
      Method[] of = ObjectFactory.class.getMethods();
      String creatorName = "create" + className;
      for (int mx = 0; mx < of.length; ++mx) {
        if (of[mx].getName().equals(creatorName)) {
          //!! if (Debug.enabled) System.out.println("found: ObjectFactory." + of[mx].getName());
          return of[mx].invoke(objectFactory, new Object[0]);
        }
      }
      throw new IllegalArgumentException("missing: ObjectFactory." + creatorName);      
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return null;
  }

  /**
   * Create population for images of an given type.
   * The population is empty. Insert images with createImage(...) method.
   * @param typeName
   * @return PopulationType
   */
  public PopulationType createPopulation(String typeName) {
    PopulationType population = null;
    try {
      population = objectFactory.createPopulationType();
    } catch (Exception exc) {
        Debug.stackTrace(exc);
    }
    return population;
  }

  /**
   * Create image and insert it at position nx to the population.
   * Chromsomes for calculation and coloration are not set.
   * @param population
   * @param nx
   * @param calculationName
   * @param coloratorName
   * @return ImageType
   */
  public ImageType createImage(PopulationType population, int nx, String typeName, String calculationName, String coloratorName) {
    String imageClassName = typeName + "Image";
    ImageType image = null;

    try {
      // create image type
      image = (ImageType) soupFactory.create(imageClassName);
      
      {
      // find setCalculationName method
      Class[] calculationTypes = new Class[1];
      calculationTypes[0] = calculationName.getClass();
      Method setCalculationName = image.getClass().getMethod("setCalculationName", calculationTypes);
      // add calculation name to image
      Object[] param = new Object[1];
      param[0] = calculationName;
      setCalculationName.invoke(image, param);
      }
      if(ColoratorFactory.isSimpleColorator(coloratorName)) {
        // find setColoratorName method
        Class[] coloratorTypes = new Class[1];
        coloratorTypes[0] = coloratorName.getClass();
        Method setColoratorName = image.getClass().getMethod("setColoratorName", coloratorTypes);
        // add colorator name to image
        Object[] param = new Object[1];
        param[0] = coloratorName;
        setColoratorName.invoke(image, param);
      }
      else {
        // create colorator
        ColoratorType newColorator = objectFactory.createColoratorType();
        newColorator.setColoratorName(coloratorName);
      
        // find setColorator method
        Class[] coloratorTypes = new Class[1];
        coloratorTypes[0] = ColoratorType.class;
        Method setColorator = image.getClass().getMethod("setColorator", coloratorTypes);
        // add colorator to image
        Object[] param = new Object[1];
        param[0] = newColorator;
        setColorator.invoke(image, param);
      }

      // add image to population
      Method getList = population.getClass().getMethod("get" + typeName, new Class[0]);
      java.util.List imageList = (java.util.List) getList.invoke(population, new Object[0]);
      imageList.add(image);
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }

    return image;
  }

  /**
   * Create random calculation chromosome and add it to image[nx].
   * @param population
   * @param nx
   * @param chromosomeStamp
   * @return ChromosomeType
   */
  public ChromosomeType createRandomChromosome(PopulationType population, int nx, ChromosomeStamp chromosomeStamp) {
    String chromosomeClassName = getTypeName(population) + "Chromosome";
    ChromosomeType newChromosome = null;

    try {
      // get image[nx]
      List list = getImageList(population);
      ImageType image = (ImageType)list.get(nx);
      
      if (image instanceof ScalarExpressionImage) {
        newChromosome = kandid.soup.genetic.scalarvm.Randomizer.create();
      }
      else if (image instanceof VectorExpressionImage) {
        newChromosome = kandid.soup.genetic.vectorvm.Randomizer.create();
      }
//      else if (image instanceof VoronoiImage) {
//        newChromosome = kandid.soup.genetic.voronoi.Randomizer.create();
//      }
//      else if (image instanceof VoronoiTransparentImage) {
//        newChromosome = kandid.soup.genetic.voronoi.RandomizerTransparent.create();
//      }
      else {
  
        // create chromosome
        newChromosome = (ChromosomeType) create(chromosomeClassName);
  
        // find randomize method
        Class[] signatureTypes = new Class[1];
        signatureTypes[0] = newChromosome.getClass();
        Method randomized = randomizer.getClass().getMethod("randomize", signatureTypes);
  
        // call randomize method
        Object[] param2 = new Object[1];
        param2[0] = newChromosome;
        randomized.invoke(randomizer, param2);
  
      }
      // add chromosome to image
      Object[] param = new Object[1];
      param[0] = newChromosome;
      // find setChromosome method
      Method[] of = image.getClass().getMethods();
      int mx = 0;
      for (mx = 0; mx < of.length; ++mx) {
        if (of[mx].getName().equals("setChromosome")) {
          // add chromosome to image
          of[mx].invoke(image, param);
          break;
        }
      }
      
      // set unique identification
      newChromosome.setIdent(chromosomeStamp.getNextStamp());
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }

    return newChromosome;
  }

  /**
   * Method getTypeName.
   * @param population
   * @return String
   */
  public static String getTypeName(PopulationType population) {
    try {
      // find get image list method
      Method[] of = population.getClass().getMethods();
      int mx = 0;
      for (mx = 0; mx < of.length; ++mx) {
        if (of[mx].getName().startsWith("get") && of[mx].getReturnType() == java.util.List.class) {
          // get image list
          java.util.List list;
          list = (List) of[mx].invoke(population, (Object[])null);
          if (list != null && list.size() > 0) {
            return of[mx].getName().substring(3);
          }
        }
      }
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return null;
  }


  /**
   * replace image in population
   * @param population
   * @param ix
   * @param image
   */
  public void replaceImage(PopulationType population, int ix, ImageType image) {
      // replace image in population
      java.util.List imageList = getImageList(population);
      imageList.set(ix, image);
  }

  /**
   * add image to population
   * @param population
   * @param ix
   * @param image
   */
  public void addImage(PopulationType population, int ix, ImageType image) {
      // add image to population
      java.util.List imageList = getImageList(population);
      imageList.add(ix, image);
  }

  /**
   * Create random colorator chromosome and add it to image[nx]
   * @param population
   * @param nx
   * @param coloratorName
   * @param chromosomeStamp
   * @return ColoratorType
   */
  public ColoratorType createRandomColorator(PopulationType population, int nx, String coloratorName, ChromosomeStamp chromosomeStamp) {
    String cloloratorChromosomeClassName = coloratorName;
    if(coloratorName.endsWith("Colorator")) {
      cloloratorChromosomeClassName = coloratorName.substring(0, coloratorName.length()-9);
    }
    cloloratorChromosomeClassName += "Chromosome";
    ColoratorType newColorator = null;
    try {
      // get image[nx]
      List list = getImageList(population);
      ImageType image = (ImageType)list.get(nx);

      // create colorator
      newColorator = objectFactory.createColoratorType();
      
      {
      if(ColoratorFactory.isSimpleColorator(coloratorName)) {
        // find setColoratorName method
        Class[] coloratorTypes = new Class[1];
        coloratorTypes[0] = coloratorName.getClass();
        Method setColoratorName = image.getClass().getMethod("setColoratorName", coloratorTypes);
        // add colorator name to image
        Object[] param = new Object[1];
        param[0] = coloratorName;
        setColoratorName.invoke(image, param);
      }
      else {
        // create colorator chromosome
        ChromosomeType newColoratorChromosome = (ChromosomeType) create(cloloratorChromosomeClassName);
  
        // set unique identification
        newColoratorChromosome.setIdent(chromosomeStamp.getNextStamp());
        newColorator.setColoratorName(coloratorName);
        
        // find randomize method
        Class[] signatureTypes1 = new Class[1];
        signatureTypes1[0] = newColoratorChromosome.getClass();
        Method randomized = randomizer.getClass().getMethod("randomize", signatureTypes1);
        // call randomize method
        Object[] param1 = new Object[1];
        param1[0] = newColoratorChromosome;
        randomized.invoke(randomizer, param1);
  
        // add color table to colorator
        setColorTable(newColorator, newColoratorChromosome);
          
        // find setColorator method
        Class[] coloratorTypes = new Class[1];
        coloratorTypes[0] = ColoratorType.class;
        Method setColorator = image.getClass().getMethod("setColorator", coloratorTypes);
        // add colorator to image
        Object[] param4 = new Object[1];
        param4[0] = newColorator;
        setColorator.invoke(image, param4);
      }
      }
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }

    return newColorator;
  }

  /**
   * Clone calculation chromosome.
   * 
   * @param chromosome
   * @param newIdent
   * @param populationIdent
   * @return ChromosomeType
   */
  public ChromosomeType clone(ChromosomeType chromosome, boolean newIdent, String populationIdent) {
    ChromosomeType newChromosome = null;

    try {
      if (chromosome instanceof ScalarExpressionChromosome) {
        newChromosome = kandid.soup.genetic.scalarvm.Cloner.deepclone((ScalarExpressionChromosome) chromosome);
      }
      else if (chromosome instanceof VectorExpressionChromosome) {
        newChromosome = kandid.soup.genetic.vectorvm.Cloner.deepclone((VectorExpressionChromosome) chromosome);
      }
//      else if (chromosome instanceof VoronoiChromosome) {
//        newChromosome = kandid.soup.genetic.voronoi.Cloner.deepclone((VoronoiChromosome) chromosome);
//      }
//      else if (chromosome instanceof VoronoiTransparentChromosome) {
//        newChromosome = kandid.soup.genetic.voronoi.ClonerTransparent.deepclone((VoronoiTransparentChromosome) chromosome);
//      }
      else {
        // find clone method
        Class[] signatureTypes = new Class[1];
        signatureTypes[0] = chromosome.getClass();
        Method clone = cloner.getClass().getMethod("clone", signatureTypes);

        // call clone method
        Object[] param = new Object[1];
        param[0] = chromosome;
        newChromosome = (ChromosomeType) clone.invoke(cloner, param);
      }

      // set unique identification
      if (newIdent) {
        newChromosome.setIdent(ChromosomeStamp.forceNextChromosomeId(chromosome.getIdent()));
      }
      else {
        newChromosome.setIdent(chromosome.getIdent());
      }
      if (populationIdent != null) {
        newChromosome.setIdent(ChromosomeStamp.patchPopulationIdent(newChromosome.getIdent(), populationIdent));
      }
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }

    return newChromosome;
  }

  /**
   * Clone calculation chromosome.
   * @param colorator
   * @param newIdent
   * @param populationIdent
   * @return ColoratorType
   */
  public ColoratorType clone(ColoratorType colorator, boolean newIdent, String populationIdent) {
    if (colorator != null) {
      try {
        ColoratorType newColorator = objectFactory.createColoratorType();
        newColorator.setColoratorName(new String(colorator.getColoratorName()));
        LookUpTableChromosome lut = colorator.getLookUpTable();
        if (lut != null) {
          newColorator.setLookUpTable((LookUpTableChromosome) clone(lut, newIdent, populationIdent));
        }
        else {
          TransparentLookUpTableChromosome transLut = colorator.getTransparentLookUpTable();
          if (transLut != null) {
            newColorator.setTransparentLookUpTable((TransparentLookUpTableChromosome) clone(transLut, newIdent, populationIdent));
          }
          else {
            GradientChromosome gradient = colorator.getGradient();
            if (gradient != null) {
              newColorator.setGradient((GradientChromosome) clone(gradient, newIdent, populationIdent));
            }
            else {
              HsbFrequenceChromosome hsbFrequence = colorator.getHsbFrequence();
              if (hsbFrequence != null) {
                newColorator.setHsbFrequence((HsbFrequenceChromosome) clone(hsbFrequence, newIdent, populationIdent));
              }
              else {
                if (Debug.enabled) assert false : "color table not set";
              }
            }
          }
        }
        return newColorator;
      } catch (Exception exc) {
        Debug.stackTrace(exc);
      }
    }
    return null;
  }
  
  /**
   * Merge two chromosomes with crossing over.
   * @param chromosome1
   * @param chromosome2
   * @param env
   * @return the new chromosome.
   */
  public ChromosomeType merge(ChromosomeType chromosome1, ChromosomeType chromosome2, Environment env) {
    ChromosomeType newChromosome = null;
    
    try {
      env.setSource1(true);
      
      if(chromosome1 instanceof ScalarExpressionChromosome) {
        assert chromosome2 instanceof ScalarExpressionChromosome;
        newChromosome = kandid.soup.genetic.scalarvm.Merger.merge(env, (ScalarExpressionChromosome)chromosome1, (ScalarExpressionChromosome)chromosome2);
      }
      else if (chromosome1 instanceof VectorExpressionChromosome) {
        assert chromosome2 instanceof VectorExpressionChromosome;
        newChromosome = kandid.soup.genetic.vectorvm.Merger.merge(env, (VectorExpressionChromosome)chromosome1, (VectorExpressionChromosome)chromosome2);
      }
//      else if (chromosome1 instanceof VoronoiChromosome) {
//        assert chromosome2 instanceof VoronoiChromosome;
//        newChromosome = kandid.soup.genetic.voronoi.Merger.merge(env, (VoronoiChromosome)chromosome1, (VoronoiChromosome)chromosome2);
//      }
//      else if (chromosome1 instanceof VoronoiTransparentChromosome) {
//        assert chromosome2 instanceof VoronoiTransparentChromosome;
//        newChromosome = kandid.soup.genetic.voronoi.MergerTransparent.merge(env, (VoronoiTransparentChromosome)chromosome1, (VoronoiTransparentChromosome)chromosome2);
//      }
      else {
        // find merge method
        Class[] signatureTypes = new Class[3];
        signatureTypes[0] = env.getClass();
        signatureTypes[1] = chromosome1.getClass();
        signatureTypes[2] = chromosome2.getClass();
        Method merge = merger.getClass().getMethod("merge", signatureTypes);
  
        // call merge(chromosome1, chromosome2) method
        Object[] param = new Object[3];
        param[0] = env;
        param[1] = chromosome1;
        param[2] = chromosome2;
        newChromosome = (ChromosomeType) merge.invoke(merger, param);
      }
      // set unique identification
      newChromosome.setIdent(ChromosomeStamp.forceNextChromosomeId(chromosome1.getIdent()));
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }

    return newChromosome;
  }

  /**
   * Mutate chromosome.
   * 
   * @param chromosome
   * @param env
   * @return int
   */
  public int mutate(ChromosomeType chromosome, Environment env) {
    int mutationCounter = 0;
    try {
      if (chromosome instanceof ScalarExpressionChromosome) {
        mutationCounter = kandid.soup.genetic.scalarvm.Mutator.mutate(env, (ScalarExpressionChromosome) chromosome);
      }
      else if (chromosome instanceof VectorExpressionChromosome) {
        mutationCounter = kandid.soup.genetic.vectorvm.Mutator.mutate(env, (VectorExpressionChromosome) chromosome);
      }
//      else if (chromosome instanceof VoronoiChromosome) {
//        mutationCounter = kandid.soup.genetic.voronoi.Mutator.mutate(env, (VoronoiChromosome) chromosome);
//      }
//      else if (chromosome instanceof VoronoiTransparentChromosome) {
//        mutationCounter = kandid.soup.genetic.voronoi.MutatorTransparent.mutate(env, (VoronoiTransparentChromosome) chromosome);
//      }
      else {
        // find mutate method
        Class[] signatureTypes = new Class[2];
        signatureTypes[0] = env.getClass();
        signatureTypes[1] = chromosome.getClass();
        Method mutate = mutator.getClass().getMethod("mutate", signatureTypes);

        // call mutate method
        Object[] param = new Object[2];
        param[0] = env;
        param[1] = chromosome;
        Integer ret = (Integer) mutate.invoke(mutator, param);
        mutationCounter = ret.intValue();
      }

      // set unique identification
      chromosome.setIdent(ChromosomeStamp.forceNextChromosomeId(chromosome.getIdent()));
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return mutationCounter;
  }

	public int mutate(ColoratorType colorator, Environment env) {
		if (colorator != null) {
			LookUpTableChromosome lut = colorator.getLookUpTable();
			if (lut != null) {
				return mutate(lut, env);
			}
      else {
        TransparentLookUpTableChromosome transLut = colorator.getTransparentLookUpTable();
        if (transLut != null) {
          return mutate(transLut, env);
        }
        else {
          GradientChromosome gradient = colorator.getGradient();
          if (gradient != null) {
            return mutate(gradient, env);
          }
  				else {
  					HsbFrequenceChromosome hsbFrequence = colorator.getHsbFrequence();
  					if (hsbFrequence != null) {
  						return mutate(hsbFrequence, env);
  					}
				  }
        }
		  }
    }
		//!! if (Debug.enabled) assert false : "color table not set";
		return 0;
	}
  
  /**
   * Fill editor with calculation and colorator chromosome.
   * @param chromosome
   * @param colorator
   * @param tree
   */
  public void fillEditor(ChromosomeType chromosome, ColoratorType colorator, JTree tree) {
    fillEditor(chromosome, tree);
    if(colorator != null) {
      ChromosomeType colorChromosome = soupFactory.getColorTable(colorator);
      if(colorChromosome != null) {
        soupFactory.fillEditor(colorChromosome, tree);
      }
    }
  }
  
  /**
   * Method fillEditor.
   * @param chromosome
   * @param tree
   */
  private void fillEditor(ChromosomeType chromosome, JTree tree) {
    try {
      // find fillEditor method
      Class[] signatureTypes = new Class[2];
      signatureTypes[0] = chromosome.getClass();
      signatureTypes[1] = tree.getClass();
      Method fillEditor = editorFiller.getClass().getMethod("fillEditor", signatureTypes);

      // call fillEditor method
      Object[] param = new Object[2];
      param[0] = chromosome;
      param[1] = tree;
      fillEditor.invoke(editorFiller, param);
      
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
  }

  /**
   * Write population to file.
   * @param model
   * @param outputFileName
   */
  public void marshalToFile(Object model, String outputFileName) {
    try {
      // create a Marshaller and marshal to a file
      FileOutputStream out = new FileOutputStream(outputFileName);
      Marshaller m = jc.createMarshaller();
      m.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);
      m.marshal(model, out);
      out.close();
      Console.append("store " +  (new File(outputFileName).getAbsoluteFile()));
    } catch (JAXBException je) {
      Debug.stackTrace(je);
    } catch (IOException ioe) {
      Debug.stackTrace(ioe);
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
  }

  /**
   * Read population from file.
   * @param inputFileName
   * @return Object
   */
  public Object unmarshalFromFile(String inputFileName) {
    Object input = null;
    try {
      // create an Unmarshaller
      Unmarshaller u = jc.createUnmarshaller();
      input = u.unmarshal(new FileInputStream(inputFileName));
      Console.append("read " + (new File(inputFileName).getAbsoluteFile()));
      upgrade(input);
    } catch (JAXBException je) {
      Debug.stackTrace(je);
    } catch (IOException ioe) {
      Debug.stackTrace(ioe);
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return input;
  }

  /**
   * Upgrades from older soup schema definitions.
   * @param input
   * @throws JAXBException
   */
  private void upgrade(Object input) throws JAXBException {
    if (input instanceof PopulationType) {
      PopulationType population = (PopulationType) input;
      // upgrade LsysD0L
      if (population.getLsysD0L() != null) {
        for (Iterator iter = population.getLsysD0L().iterator(); iter.hasNext();) {
          LsysD0LImage lsysD0LImage = (LsysD0LImage)iter.next();
          upgradeLsysD0L_0_2_7(lsysD0LImage);
          upgradeLsysD0L_0_2_9(lsysD0LImage);
        }
      }
      // upgrade Lsys0L
      if (population.getLsys0L() != null) {
        for (Iterator iter = population.getLsys0L().iterator(); iter.hasNext();) {
          Lsys0LImage lsys0LImage = (Lsys0LImage)iter.next();
          upgradeLsys0L_0_2_7(lsys0LImage);
          upgradeLsys0L_0_2_9(lsys0LImage);
        }
      }
      // upgrade ScalarExpression to Version 0.2.8
      if (population.getScalarExpression() != null) {
        for (Iterator iter = population.getScalarExpression().iterator(); iter.hasNext();) {
          upgradeScalarExpression_0_2_8((ScalarExpressionImage) iter.next());
        }
      }
      // upgrade PovThing to Version 0.3.3
      if (population.getPovThing() != null) {
        for (Iterator iter = population.getPovThing().iterator(); iter.hasNext();) {
          upgradePovThing_0_3_3((PovThingImage) iter.next());
        }
      }
      // upgrade PovThingNF to Version 0.3.3
      if (population.getPovThingNF() != null) {
        for (Iterator iter = population.getPovThingNF().iterator(); iter.hasNext();) {
          upgradePovThingNF_0_3_3((PovThingNFImage) iter.next());
        }
      }
    }
    else if (input instanceof EntityType) {
      EntityType entity = (EntityType) input;
      // upgrade LsysD0L
      if (entity.getLsysD0L() != null) {
        upgradeLsysD0L_0_2_7(entity.getLsysD0L());
        upgradeLsysD0L_0_2_9(entity.getLsysD0L());
      }
      // upgrade Lsys0L
      if (entity.getLsys0L() != null) {
        upgradeLsys0L_0_2_7(entity.getLsys0L());
        upgradeLsys0L_0_2_9(entity.getLsys0L());
      }
      // upgrade ScalarExpression to Version 0.2.8
      if (entity.getScalarExpression() != null) {
        upgradeScalarExpression_0_2_8(entity.getScalarExpression());
      }
      // upgrade PovThing to Version 0.3.3
      if (entity.getPovThing() != null) {
        upgradePovThing_0_3_3(entity.getPovThing());
      }
      // upgrade PovThing to Version 0.3.3
      if (entity.getPovThingNF() != null) {
        upgradePovThingNF_0_3_3(entity.getPovThingNF());
      }
    }
    else {
      assert false : "type not suported" + input.getClass().getName();
    }

  }

  /**
   * upgrade PovThing to Version 0.3.3
   * @param image
   */
  private void upgradePovThing_0_3_3(PovThingImage image) {
    if(image.getCalculationName().equals("kandid.calculation.bridge.povray.PovThingCalculation"))
      image.setCalculationName("kandid.calculation.bridge.povray.PovTextureCalculation");
  }

  /**
   * upgrade PovThing to Version 0.3.3
   * @param image
   */
  private void upgradePovThingNF_0_3_3(PovThingNFImage image) {
    if(image.getCalculationName().equals("kandid.calculation.bridge.povray.PovThingCalculation"))
      image.setCalculationName("kandid.calculation.bridge.povray.PovTextureCalculation");
  }

  /**
   * upgrade LsysProduction to Version 0.2.9
   * @param image
   */
  private void updateLsysProductionList_0_2_9(List productionList) throws JAXBException {
    for (Iterator iter = productionList.iterator(); iter.hasNext();) {
      LsysProductionGene production = (LsysProductionGene) iter.next();
      ContextGene leftContext = objectFactory.createContextGene();
      leftContext.setValue("");
      production.setLeftContext(leftContext);
      ContextGene rightContext = objectFactory.createContextGene();
      rightContext.setValue("");
      production.setRightContext(rightContext);
    }
  }

  /**
   * upgrade Lsys0L to Version 0.2.9
   * @param image
   */
  private void upgradeLsys0L_0_2_9(Lsys0LImage image) throws JAXBException {
    List productionList = image.getChromosome().getProduction();
    updateLsysProductionList_0_2_9(productionList);
  }

  /**
   * upgrade LsysD0L to Version 0.2.9
   * @param image
   */
  private void upgradeLsysD0L_0_2_9(LsysD0LImage image) throws JAXBException {
    List productionList = image.getChromosome().getProduction();
    updateLsysProductionList_0_2_9(productionList);
  }

  /**
   * @param image
   */
  private void upgradeScalarExpression_0_2_8(ScalarExpressionImage image) {
    String calculationName = image.getCalculationName();
    if(calculationName.equals("kandid.calculation.vm.ScalarExpressionCalculation")) {
      image.setCalculationName("kandid.calculation.vm.scalar.ScalarExpressionCalculation");
    }
  }

  /**
   * upgrade LsysD0L to Version 0.2.7
   * @param image
   * @throws JAXBException
   */
  private void upgradeLsysD0L_0_2_7(LsysD0LImage image) throws JAXBException {
    if(image.getChromosome().getBaseIndex() == null) {
      IntegerGene baseIndex = objectFactory.createIntegerGene();
      baseIndex.setValue(0);
      image.getChromosome().setBaseIndex(baseIndex);
    }
  }

  /**
   * upgrade Lsys0L to Version 0.2.7
   * @param image
   * @throws JAXBException
   */
  private void upgradeLsys0L_0_2_7(Lsys0LImage image) throws JAXBException {
    if(image.getChromosome().getBaseIndex() == null) {
      IntegerGene baseIndex = objectFactory.createIntegerGene();
      baseIndex.setValue(0);
      image.getChromosome().setBaseIndex(baseIndex);
    }
  }

  /**
   * Write population to string.
   * @param model
    */
  public String marshalToString(Object model) {
    return kandid.util.Util.marshalToString(model, jc);
  }


  /**
   * Read population from string.
   * @param stringRepresentation
   * @return Object
   */
  public Object unmarshalFromString(String stringRepresentation) {
//    javax.xml.bind.JAXBElement input = null;
    Object input = null;
    try {
      // create an Unmarshaller
      Unmarshaller u = jc.createUnmarshaller();
      Object unkown = u.unmarshal(new StreamSource(new StringReader(stringRepresentation))); //TODO
      if(unkown instanceof javax.xml.bind.JAXBElement) {
        input = (JAXBElement) unkown;
        upgrade(((JAXBElement)input).getValue());
      }
      else {
        input = unkown;
        upgrade(input);
      }
//      input = (JAXBElement) u.unmarshal(new StreamSource(new StringReader(stringRepresentation)));
//      input = u.unmarshal(new StreamSource(new StringReader(stringRepresentation)));
//      upgrade(input.getValue());
//      upgrade(input);
//!!      System.out.println(stringRepresentation + " " + input.getClass().getName());
    } catch (JAXBException je) {
      Debug.stackTrace(je);
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return input;
  }

  /**
   * Method validate.
   * @param model
   * @return boolean
   */
  public boolean validate(Object model) {
	  return true;
//xx    boolean valid = false;
//xx    try {
//xx      System.out.println("validate " + model.getClass());
//xx      Validator v = jc.createValidator();
//xx      valid = v.validateRoot(model);  //TODO
//xx    } catch (JAXBException je) {
//xx      Debug.stackTrace(je);
//xx    } catch (Exception exc) {
//xx      Debug.stackTrace(exc);
//xx    }
//xx    System.out.println("valid = " + valid);
//xx    return valid;
  }

  /**
   * Get the population's list of images.
   * @param population
   */
  public List getImageList(PopulationType population) {
    try {
      // find get image list method
        Method[] of = population.getClass().getMethods();
        int mx = 0;
        for (mx = 0; mx < of.length; ++mx) {
          if (of[mx].getName().startsWith("get") && of[mx].getReturnType() == java.util.List.class ) {
            // get image list
            java.util.List list = (List)of[mx].invoke(population, (Object[])null);
            if(list.size() > 0) {
              return list;
            }
          }
        }
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return null;
  }

  /**
   * Method getImage.
   * @param id
   */
  public ImageType getImage(PopulationType population, int id) {
    return (ImageType)soupFactory.getImageList(population).get(id);
  }

  /**
   * Get size of population.
   * @param population
   */
  public int getSize(PopulationType population) {
    List list = getImageList(population);
    return list == null ? 0 : list.size();
  }

  /**
   * Get calculation chromosome from image[nx].
   * @param population
   * @param nx
   * @return ChromosomeType
   */
  public ChromosomeType getChromosome(PopulationType population, int nx) {
    ChromosomeType chromosomeType = null;
    try {
      List list = getImageList(population);
      ImageType image = (ImageType)list.get(nx);
      // find getChromosome method
      Method getChromosome = image.getClass().getMethod("getChromosome", (Class<?>[])null);
      // get image list
      chromosomeType = (ChromosomeType)getChromosome.invoke(image, (Object[])null);
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return chromosomeType;
  }

  /**
   * @param population
   * @param nx
   * @return coloratorName
   */
  public String getColoratorName(PopulationType population, int nx) {
    String coloratorName = null;
    try {
      List list = getImageList(population);
      ImageType image = (ImageType)list.get(nx);
      // find getColoratorName method
      Method getColoratorName = image.getClass().getMethod("getColoratorName", (Class<?>[])null);
      // get colorator name
      coloratorName = (String)getColoratorName.invoke(image, (Object[])null);
      if(coloratorName == null) {
        ColoratorType coloratorType = getColorator(population, nx);
        coloratorName = coloratorType.getColoratorName();
      }
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return coloratorName;
  }

  /**
   * Get colorator from image[nx].
   * @param population
   * @param nx
   * @return ColoratorType
   */
  public ColoratorType getColorator(PopulationType population, int nx) {
    ColoratorType colorator = null;
    try {
      List list = getImageList(population);
      ImageType image = (ImageType)list.get(nx);
      // find getColorator method
      Method getColorator = image.getClass().getMethod("getColorator", (Class<?>[])null);
      // get colorator name
      colorator = (ColoratorType)getColorator.invoke(image, (Object[])null);
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return colorator;
  }

  /**
   * Method getCalculationName.
   * @param population
   * @param typeName
   * @param nx
   * @return calculation class name
   */
  public String getCalculationName(PopulationType population, int nx) {
    String calculationName = null;
    try {
      List list = getImageList(population);
      ImageType image = (ImageType)list.get(nx);
      // find getCalculationName method
      Method getCalculationNameMethod = image.getClass().getMethod("getCalculationName", (Class<?>[])null);
      // get colorator name
      calculationName = (String)getCalculationNameMethod.invoke(image, (Object[])null);
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return calculationName;
  }

  /**
   * Method setRating.
   * @param population
   * @param typeName
   * @param nx
   * @param fitness
   */
  public void setRating(PopulationType population, int nx, Fitness fitness) {
    try {
      List list = getImageList(population);
      ImageType image = (ImageType)list.get(nx);
      if(fitness.abort) {
        image.setRating(RatingBase.ABORT);
      }
      else if(fitness.best) {
        image.setRating(RatingBase.BEST);
      }
      else if(fitness.parent1) {
        image.setRating(RatingBase.PARENT_1);
      }
      else if(fitness.parent2) {
        image.setRating(RatingBase.PARENT_2);
      }
      else {
        image.setRating(RatingBase.NONE);
      }
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
  }

  /**
   * Method mapRating.
   * @param population
   * @param nx
   * @return Fitness
   */
  public Fitness mapRating(PopulationType population, int nx) {
    Fitness fitness = new Fitness();
    try {
      List list = getImageList(population);
      ImageType image = (ImageType)list.get(nx);
      RatingBase rating = image.getRating();
      fitness.setRating(rating);
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return fitness;
  }

  /**
   * Method setChromosome.
   * @param population
   * @param nx
   * @param chromosome
   */
  public void setChromosome(PopulationType population, int nx, ChromosomeType chromosome) {
    try {
      List list = getImageList(population);
      ImageType image = (ImageType)list.get(nx);
      // add chromosome to image
      Object[] param = new Object[1];
      param[0] = chromosome;
      // find setChromosome method
      Method[] of = image.getClass().getMethods();
      int mx = 0;
      for (mx = 0; mx < of.length; ++mx) {
        if (of[mx].getName().equals("setChromosome")) {
          // add chromosome to image
          of[mx].invoke(image, param);
          break;
        }
      }
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
  }

  /**
   * Method setColorator.
   * @param imageType
   * @param newColorator
   */
  public void setColorator(PopulationType population, int nx, ColoratorType colorator) {
    List list = getImageList(population);
    ImageType image = (ImageType)list.get(nx);
    image.setColorator(colorator);
  }

  public void setColorTable(ColoratorType colorator, ChromosomeType coloratorChromosome) throws Exception {
    switch(coloratorChromosome.getClass().getSimpleName()) {
    case "HsbFrequenceChromosome":
      colorator.setHsbFrequence((HsbFrequenceChromosome) coloratorChromosome);
      break;
    case "GradientChromosome":
      colorator.setGradient((GradientChromosome) coloratorChromosome);
      break;
    case "LookUpTableChromosome":
      colorator.setLookUpTable((LookUpTableChromosome) coloratorChromosome);
      break;
    case "TransparentLookUpTableChromosome":
      colorator.setTransparentLookUpTable((TransparentLookUpTableChromosome) coloratorChromosome);
      break;
    default:
      System.err.println("chromosome = " + coloratorChromosome.getClass().getSimpleName());
      System.err.println("colorator  = " + colorator.getColoratorName() + " " + colorator.getClass().getName());
      throw new Exception("colorator / chromosome mapping not implemented");
    }
  }

  public ChromosomeType getColorTable(ColoratorType colorator) {
    try {
      ChromosomeType colorTable = colorator.getLookUpTable();
      if(colorTable != null) {
        return colorTable;
      }
      colorTable = colorator.getTransparentLookUpTable();
      if(colorTable != null) {
        return colorTable;
      }
      colorTable = colorator.getGradient();
      if(colorTable != null) {
        return colorTable;
      }
      colorTable = colorator.getHsbFrequence();
      if(colorTable != null) {
        return colorTable;
      }
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return null;
  }

  /**
   * Method createEntity.
   * @param typeName
   * @param image
   * @return EntityType
   */
  public EntityType createEntity(String typeName, ImageType image) {
    EntityType entity = null;

    try {
      // create entity
      entity = objectFactory.createEntityType();
      
      Object[] param = new Object[1];
      param[0] = image;
      Method[] of = entity.getClass().getMethods();
      int mx = 0;
      for (mx = 0; mx < of.length; ++mx) {
        if (of[mx].getName().equals("set" + typeName)) {
          // add chromosome to image
          of[mx].invoke(entity, param);
          break;
        }
      }
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }

    return entity;
  }

  public EntityType createEntity(ImageType image) {
    EntityType entity = null;

    try {
      // create entity
      entity = objectFactory.createEntityType();
      
      Object[] param = new Object[1];
      param[0] = image;
      Method[] of = entity.getClass().getMethods();
      int mx = 0;
      for (mx = 0; mx < of.length; ++mx) {
        if (of[mx].getName().startsWith("set") 
            && of[mx].getParameterTypes().length == 1
            && of[mx].getParameterTypes()[0] == image.getClass().getInterfaces()[0]) {
          // add image to entity
          of[mx].invoke(entity, param);
          break;
        }
      }
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }

    return entity;
  }

  /**
   * get image from entity.
   * @param entity
   * @return ImageType
   */
  public ImageType getImage(EntityType entity) {
    try {
      // find get image method
        Method[] of = entity.getClass().getMethods();
        int mx = 0;
        for (mx = 0; mx < of.length; ++mx) {
          if (of[mx].getName().startsWith("get") && of[mx].getParameterTypes().length == 0) {
            // get image
            Object image = of[mx].invoke(entity, (Object[])null);
            if(image != null && image instanceof ImageType) {
              return (ImageType)image;
            }
          }
        }
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return null;
  }

  /**
   * Method getTypeName.
   * @param entity
   * @return String
   */
  public static String getTypeName(EntityType entity) {
     try {
      // find get image list method
      Method[] of = entity.getClass().getMethods();
      int mx = 0;
      for (mx = 0; mx < of.length; ++mx) {
        if (of[mx].getName().startsWith("get") && of[mx].getParameterTypes().length == 0) {
          // get image
          Object image = of[mx].invoke(entity, (Object[])null);
          if(image != null && image instanceof ImageType) {
            return of[mx].getName().substring(3);
          }
        }
      }
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
   return null;
  }

  /**
   * Method getChromosome.
   * @param entity
   * @param px
   * @return ChromosomeType
   */
  public ChromosomeType getChromosome(EntityType entity) {
    ChromosomeType chromosomeType = null;
    try {
      ImageType image = getImage(entity);
      if(image != null) {
      // find getChromosome method
      Method getChromosome = image.getClass().getMethod("getChromosome", (Class<?>[])null);
      // get chromosome
      chromosomeType = (ChromosomeType)getChromosome.invoke(image, (Object[])null);
      }
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return chromosomeType;
  }

  /**
   * Method getCalculationName.
   * @param entity
   * @param px
   * @return String
   */
  public String getCalculationName(EntityType entity) {
    String calculationName = null;
    try {
      ImageType image = getImage(entity);
      if(image != null) {
        // find getCalculationName method
        Method getCalculationNameMethod = image.getClass().getMethod("getCalculationName", (Class<?>[])null);
        // get colorator name
        calculationName = (String)getCalculationNameMethod.invoke(image, (Object[])null);
      }
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return calculationName;
  }

  /**
   * Method getColorator.
   * @param entity
   * @param px
   * @return ColoratorType
   */
  public ColoratorType getColorator(EntityType entity) {
    ColoratorType colorator = null;
    try {
      ImageType image = getImage(entity);
      // find getColorator method
      Method getColorator = image.getClass().getMethod("getColorator", (Class<?>[])null);
      // get colorator name
      colorator = (ColoratorType)getColorator.invoke(image, (Object[])null);
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return colorator;
  }

  /**
   * Method getColoratorName.
   * @param entity
   * @param px
   * @return String
   */
  public String getColoratorName(EntityType entity) {
    String coloratorName = null;
    try {
      ImageType image = getImage(entity);
      // find getColoratorName method
      Method getColoratorName = image.getClass().getMethod("getColoratorName", (Class<?>[])null);
      // get colorator name
      coloratorName = (String)getColoratorName.invoke(image, (Object[])null);
      if(coloratorName == null) {
        ColoratorType coloratorType = getColorator(entity);
        coloratorName = coloratorType.getColoratorName();
      }
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return coloratorName;
  }

}
