/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.soup.test;

import java.io.File;
import java.io.FileWriter;
import java.lang.reflect.Field;
import java.lang.reflect.Method;

import javax.swing.JTree;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;

import junit.framework.TestCase;
import kandid.Environment;
import kandid.soup.ChromosomeType;
import kandid.soup.ColoratorType;
import kandid.soup.ImageType;
import kandid.soup.PopulationType;
import kandid.soup.map.SetEditorMapping;
import kandid.soup.util.ChromosomeStamp;
import kandid.soup.util.SoupFactory;
import kandid.util.CentralRandomizer;
import kandid.util.Debug;

public class SoupFactoryTest extends TestCase {
  private FileDiff fileDiff = new FileDiff();
  private SoupFactory soupFactory = null;
  public static String outputPath = "src/kandid/soup/test/";

  /**
   * Constructor for SoupFactoryTest.
   * @param arg0
   */
  public SoupFactoryTest(String arg0) {
    super(arg0);
  }

  public void ignore_testPovIsoSurface() {  //TODO testPovIsoSurface
      // create a JAXBContext capable of handling classes generated into kandid.soup package
      soupFactory = SoupFactory.getSoupFactory();
      populationTest("PovIsoSurface", "kandid.colorator.RGBColorator", 1512, 0.95);
  }

  public void testPovNFThing() {
      // create a JAXBContext capable of handling classes generated into kandid.soup package
      soupFactory = SoupFactory.getSoupFactory();
      populationTest("PovThingNF", "kandid.colorator.RGBColorator", 1512, 0.95);
  }

  public void testPovThing() {
      // create a JAXBContext capable of handling classes generated into kandid.soup package
      soupFactory = SoupFactory.getSoupFactory();
      populationTest("PovThing", "kandid.colorator.RGBColorator", 1512, 0.95);
  }

  public void testFlameIfs() {
      // create a JAXBContext capable of handling classes generated into kandid.soup package
      soupFactory = SoupFactory.getSoupFactory();
      populationTest("FlameIfs", "kandid.colorator.RGBColorator", 1512, 0.95);
  }

  public void testLsysD0L() {
      // create a JAXBContext capable of handling classes generated into kandid.soup package
      soupFactory = SoupFactory.getSoupFactory();
      populationTest("LsysD0L", "kandid.colorator.BlackWhiteColorator", 31461, 0.99);
  }

  public void testLsys0L() {
      // create a JAXBContext capable of handling classes generated into kandid.soup package
      soupFactory = SoupFactory.getSoupFactory();
      populationTest("Lsys0L", "kandid.colorator.GrayColorator", 1512, 0.99);
  }

  public void testLsysIL() {
      // create a JAXBContext capable of handling classes generated into kandid.soup package
      soupFactory = SoupFactory.getSoupFactory();
      populationTest("LsysIL", "kandid.colorator.TransparentLookUpTableColorator", 31461, 0.99);
  }

  public void ignore_testScalarExpression() {  //TODO testScalarExpression
      // create a JAXBContext capable of handling classes generated into kandid.soup package
      soupFactory = SoupFactory.getSoupFactory();
      populationTest("ScalarExpression", "kandid.colorator.GradientColorator", 31462, 0.99);
  }

  public void ignore_testVectorExpression() {  //TODO testVectorExpression
      // create a JAXBContext capable of handling classes generated into kandid.soup package
      soupFactory = SoupFactory.getSoupFactory();
      populationTest("VectorExpression", "kandid.colorator.HsbFrequenceColorator", 31466, 0.99);
  }

  public void testDirectLca() {
      // create a JAXBContext capable of handling classes generated into kandid.soup package
      soupFactory = SoupFactory.getSoupFactory();
      populationTest("DirectLca", "kandid.colorator.RGBColorator", 1512, 0.05);
  }

  public void testTotalLca() {
      // create a JAXBContext capable of handling classes generated into kandid.soup package
      soupFactory = SoupFactory.getSoupFactory();
      populationTest("TotalLca", "kandid.colorator.RGBColorator", 1512, 0.05);
  }

  public void testAffineIfs() {
      // create a JAXBContext capable of handling classes generated into kandid.soup package
      soupFactory = SoupFactory.getSoupFactory();
      populationTest("AffineIfs", "kandid.colorator.RGBColorator", 1512, 0.05);
  }

  public void testAffineIfsSymm() {
      // create a JAXBContext capable of handling classes generated into kandid.soup package
      soupFactory = SoupFactory.getSoupFactory();
      populationTest("AffineIfsSymm", "kandid.colorator.GradientColorator", 1512, 0.05);
  }

  public void testVoronoi() {
      // create a JAXBContext capable of handling classes generated into kandid.soup package
      soupFactory = SoupFactory.getSoupFactory();
//!!      populationTest("Voronoi", "kandid.colorator.RGBColorator");
      populationTest("Voronoi", "kandid.colorator.LookUpTableColorator", 1512, 0.05);
  }

  public void testVoronoiTransparent() {
      // create a JAXBContext capable of handling classes generated into kandid.soup package
      soupFactory = SoupFactory.getSoupFactory();
      populationTest("VoronoiTransparent", "kandid.colorator.RGBColorator", 1512, 0.05);
  }

//  public void testColorator() {
//      // create a JAXBContext capable of handling classes generated into kandid.soup package
//      soupFactory = SoupFactory.getSoupFactory();
//      coloratorTest("LookUpTable");
//  }

//  private void coloratorTest(String typeName) {
//    try {
//      System.out.println("--- coloratorTest " + typeName + " ---");
//      Environment env = new Environment();
//      String originalFileName    = outputPath + typeName + ".test.xml";
//      String originalRefFileName = outputPath + typeName + ".ref.xml";
//      
//      // test random creation
//      CentralRandomizer.setSeed(1512); 
//
//      ChromosomeStamp chromosomeStamp = new ChromosomeStamp(typeName);
//      chromosomeStamp.wipeIdent();
//      // create color chromosome
//      ColoratorType chromosome = soupFactory.createRandomColorator(population, 0, typeName, chromosomeStamp);
//      
//      soupFactory.marshal(chromosome, originalFileName);
//      // test against reference file
//      assertTrue(fileDiff.isEquals(originalFileName, originalRefFileName, false));
//    } catch (Exception exc) {
//      Debug.stackTrace(exc);
//      assertTrue(false);
//    }
//  }
//  
  private void populationTest(String imageType, String coloratorName, long seed, double mutationRate) {
    try {
      System.out.println("--- populationTest " + imageType + " ---");
      Environment env = new Environment();
      env.setMutationRate(mutationRate);
      
      // test random creation
      CentralRandomizer.setSeed(seed);
      String originalFileName    = outputPath + imageType + ".test.xml";
      String originalRefFileName = outputPath + imageType + ".ref.xml";
      PopulationType population = createRandomPopulation(soupFactory, imageType, imageType, coloratorName);
      soupFactory.marshalToFile(population, originalFileName);
      assertTrue(soupFactory.validate(population));
      kandid.soup.PopulationType population1 = (kandid.soup.PopulationType) soupFactory.unmarshalFromFile(outputPath + imageType + ".test.xml");
      assertTrue(soupFactory.validate(population1));
      // test against reference file
      assertTrue(fileDiff.isEquals(originalFileName, originalRefFileName, true));
      
//      List imageList = soupFactory.getImageList(population);
      for (int ix = 0; ix <soupFactory.getSize(population); ++ix) {
        ChromosomeType aChromosome = soupFactory.getChromosome(population, ix);
        
        // does not compare with a reference
        // only for testing that no exception was thrown
        JTree tree = new JTree();
        tree.setEditable(true);
        tree.setModel(new DefaultTreeModel(new DefaultMutableTreeNode("test")));
        soupFactory.fillEditor(aChromosome, soupFactory.getColorator(population, ix), tree);
        StringBuffer buffer = SetEditorMapping.dump(tree);
        FileWriter fileWriter = new FileWriter(new File(outputPath + imageType + ".test.txt"));
        fileWriter.write(buffer.toString());
        fileWriter.close();
        if(imageType.equals("Voronoi")) {
          assertTrue(fileDiff.isEquals(outputPath + imageType + ".test.txt",
                                       outputPath + imageType + ".ref.txt", false));
        }
        
        // clone original calculation chromosome
        ChromosomeType aClone = soupFactory.clone(aChromosome, false, "a");
        // clone original color chromosome
        ColoratorType aColorClone = soupFactory.clone(soupFactory.getColorator(population, ix), false, "a");
        // marshall clone
//        String cloneFileName0 = outputPath + imageType + "clone.test.xml";
//        soupFactory.setColorator(population, ix, aColorClone);
//        soupFactory.setChromosome(population, ix, aClone);
//        soupFactory.marshalToFile(population, cloneFileName0); 
//        assertTrue(soupFactory.validate(aClone));
//        assertTrue(soupFactory.validate(aColorClone));
//        assertTrue(soupFactory.validate(population));
               

        // merge original and clone
        ChromosomeType aMerge = soupFactory.merge(aClone, aChromosome, env);
        soupFactory.setChromosome(population, ix, aMerge);
        System.out.println("number of switch during merge " + env.getSwitchCounter());
        String mergeFileName = outputPath + imageType + ix + "merge.test.xml";
        soupFactory.marshalToFile(population, mergeFileName);
        // merged chromosome should be equal to original chromosome
        assertTrue(soupFactory.validate(population));
//        assertTrue(fileDiff.isEquals(mergeFileName, originalRefFileName, true));
        
        // mutate original calculation chromosome
        soupFactory.mutate(aChromosome, env);
        // mutate original color chromosome
        ColoratorType aMutatedColorator = soupFactory.getColorator(population, ix);
        soupFactory.mutate(aMutatedColorator, env);
 
        // marshall clone
        String cloneFileName = outputPath + imageType + ix + "clone.test.xml";
        soupFactory.setColorator(population, ix, aColorClone);
        soupFactory.setChromosome(population, ix, aClone);
        soupFactory.marshalToFile(population, cloneFileName); 
               
        // marshall mutation
        String mutateFileName    = outputPath + imageType + ix + "mutate.test.xml";
        String mutateRefFileName = outputPath + imageType + ix + "mutate.ref.xml";
        soupFactory.setChromosome(population, ix, aChromosome);
        soupFactory.setColorator(population, ix, aMutatedColorator);
        soupFactory.marshalToFile(population, mutateFileName);

        // no differences between original and clone
        assertTrue(fileDiff.isEquals(cloneFileName, originalFileName, true));

        // differences between original and mutation
        assertTrue(!fileDiff.isEquals(mutateFileName, originalFileName, false));

        // test mutation against reference file
        assertTrue(fileDiff.isEquals(mutateFileName, mutateRefFileName, true));
        
//        Test_02 view = new Test_02();
//        soupFactory.fillEditor(soupFactory.getChromosome(image), view.getTree());
//        view.show();
        
      }

      assertTrue(soupFactory.validate(population));
      soupFactory.marshalToFile(population, outputPath + imageType + "2.test.xml");

      kandid.soup.PopulationType population2 = (kandid.soup.PopulationType) soupFactory.unmarshalFromFile(outputPath + imageType + ".test.xml");
      assertTrue(soupFactory.validate(population2));

    } catch (Exception exc) {
      Debug.stackTrace(exc);
      assertTrue(false);
    }
  }

  public static PopulationType createRandomPopulation(SoupFactory setSoupFactory, String typeName, String calculationClassName, String coloratorClassName) {
//    String imageClassName = typeName + "Image";
    PopulationType population = null;

    try {
      // create population type
      population = setSoupFactory.createPopulation(typeName);
      
      // create image type
      ImageType image = setSoupFactory.createImage(population, 0, typeName, calculationClassName, coloratorClassName);

      ChromosomeStamp chromosomeStamp = new ChromosomeStamp(typeName);
      chromosomeStamp.wipeIdent();
      // create chromosome
      ChromosomeType chromosome = setSoupFactory.createRandomChromosome(population, 0, chromosomeStamp);
      ColoratorType colorator = setSoupFactory.createRandomColorator(population, 0, coloratorClassName, chromosomeStamp);
      
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }

    return population;
  }

  public static void listMethods(Class aClass) {
    Method[] me = aClass.getMethods();
    for (int mx = 0; mx < me.length; ++mx) {
      //!!      System.out.println("method: " + aClass.getName() + "." + me[mx]);
      System.out.print("method: " + aClass.getName() + "." + me[mx].getName() + "(");
      Class[] params = me[mx].getParameterTypes();
      for (int px = 0; px < params.length; ++px) {
        System.out.print("  " + params[px].getName());
      }
      System.out.println(")");
    }
  }


  public static void listFields(Class aClass) {
    Field[] fl = aClass.getFields();
    for (int mx = 0; mx < fl.length; ++mx) {
      System.out.println("field: " + aClass.getName() + "." + fl[mx].getName());
    }
  }

}
