<!--
  Copyright (C) 2002 - 2025 Thomas Jourdan

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
-->

<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" 
                xmlns:xsd="http://www.w3.org/2001/XMLSchema"
                xmlns:jxb="http://java.sun.com/xml/ns/jaxb"
                xmlns:kandid="http://kandid.sourceforge.net/2003/XMLSchema"
                version="1.0">

  <xsl:output method="text"/>

  <xsl:template match="xsd:schema">
    <xsl:variable name="chromosomeType" select="descendant::xsd:element[@name='chromosome']/@type"/>
    // This file was generated by the vectorvm.xsl style sheet. 
    // Any modifications to this file will be lost upon recompilation of the style sheet or the source schema. 
    
    package kandid.calculation.vm.vector;
      import java.util.Random;
      import kandid.soup.ExprVectorArgsListGene;
      import kandid.soup.ExprVectorListGene;
      import kandid.soup.ExprVectorOprBase;
      import kandid.soup.ExprVectorOprGene;
      import kandid.soup.ExprVectorVarBase;
      import kandid.soup.ExprVectorVarGene;
      import kandid.soup.ExprVectorConstGene;
      import kandid.soup.GeneType;
      import kandid.soup.VectorExprGene;
      import kandid.calculation.vm.scalar.ScalarVM;
      import kandid.util.Complex;
      import kandid.util.PovNoise;
      import javax.vecmath.Vector3d;

    public class VectorVM {
      private static final int maxTapeLen = 16 * 1024;
      protected int codeTape[];
      protected int iOper[];
      protected double xOper[];
      protected double yOper[];
      protected double zOper[];
      protected int tapeLength;
      protected double xStack[] = new double[128];
      protected double yStack[] = new double[128];
      protected double zStack[] = new double[128];
      protected PovNoise povNoise;
      protected double[] aList = new double[6];
      protected Random aListRandomAccess = new Random();

      <xsl:call-template name="operator"/>

      public VectorVM(long seed) {
        povNoise = new PovNoise(seed);
      }
			
			/**
			 * Method compile.
			 * @param vectorExprGene
			 */
			public void compile(VectorExprGene vectorExprGene) {
				codeTape = new int[maxTapeLen];
				iOper = new int[maxTapeLen];
				xOper = new double[maxTapeLen];
				yOper = new double[maxTapeLen];
				zOper = new double[maxTapeLen];
				tapeLength = 0;
				compile0(vectorExprGene);
				codeTape[tapeLength] = oprStop;
				++tapeLength;
			}
		
      <xsl:call-template name="compile0"/>
			
      /**
       * @param obj
       */
      private void compileElement(GeneType obj) {
        if (obj instanceof ExprVectorVarGene) {
          ExprVectorVarGene var = (ExprVectorVarGene) obj;
          if (var.getVectorVar().equals(ExprVectorVarBase.XYT)) {
            codeTape[tapeLength++] = oprPushXYT;
          }
          else if (var.getVectorVar().equals(ExprVectorVarBase.XYR)) {
            codeTape[tapeLength++] = oprPushXYR;
          }
          else if (var.getVectorVar().equals(ExprVectorVarBase.XYM)) {
            codeTape[tapeLength++] = oprPushXYM;
          }
          else if (var.getVectorVar().equals(ExprVectorVarBase.CMPOW)) {
            codeTape[tapeLength++] = oprPushCMPOW;
          }
          else if (var.getVectorVar().equals(ExprVectorVarBase.CMSINH)) {
            codeTape[tapeLength++] = oprPushCMSINH;
          }
          else if (var.getVectorVar().equals(ExprVectorVarBase.CMIM)) {
            codeTape[tapeLength++] = oprPushCMIM;
          }
          else if (var.getVectorVar().equals(ExprVectorVarBase.CMRE)) {
            codeTape[tapeLength++] = oprPushCMRE;
          }
          else {
            assert false : "invalid var compiling vector expression" + var.getVectorVar();
          }
        }
        else if (obj instanceof ExprVectorConstGene) {
          ExprVectorConstGene aConst = (ExprVectorConstGene) obj;
          codeTape[tapeLength] = oprPushConst;
          xOper[tapeLength] = aConst.getCx().getValue();
          yOper[tapeLength] = aConst.getCy().getValue();
          zOper[tapeLength] = aConst.getCz().getValue();
          ++tapeLength;
        }
        else if (obj instanceof VectorExprGene) {
          VectorExprGene vectorExpressionArg = (VectorExprGene) obj;
          compile0(vectorExpressionArg);
          codeTape[tapeLength++] = oprPushAccu;
        }
        else {
          assert false : "invalid object compiling vector expression" + obj.getClass().getName();
        }
      }
    
      <xsl:call-template name="execute"/>
			
      <xsl:call-template name="toString"/>
    }
  </xsl:template>


  <xsl:template name="operator">
      protected static final int oprStop       = 0;
      protected static final int oprPop        = 1;
      protected static final int oprPushConst  = 2;
      protected static final int oprPushAccu   = 3;
      protected static final int oprPushXYT    = 4;
      protected static final int oprPushXYR    = 5;
      protected static final int oprPushXYM    = 6;
      protected static final int oprPushCMPOW  = 7;
      protected static final int oprPushCMSINH = 8;
      protected static final int oprPushCMIM   = 9;
      protected static final int oprPushCMRE   =10;
      protected static final int oprIf         =11;
      protected static final int oprJmp        =12;
      <xsl:for-each select="descendant::xsd:simpleType[@name = 'exprVectorOprBase']">
        <xsl:for-each select="descendant::xsd:restriction/xsd:enumeration/xsd:annotation/xsd:appinfo/kandid:execute">
            protected static final int opr^<xsl:value-of select="../../../@value"/> = (oprJmp + <xsl:value-of select="position()"/>);
        </xsl:for-each>
      </xsl:for-each>
  </xsl:template>


  <xsl:template name="compile0">
    /**
     * Method compile0.
     * @param vectorExprGene
     */
    private void compile0(VectorExprGene vectorExprGene) {
      ExprVectorListGene exprListGene = vectorExprGene.getList();
      ExprVectorOprGene opr = exprListGene.getOpr();
      ExprVectorArgsListGene args = exprListGene.getArgs();
      java.util.List&lt;GeneType&gt; argsList = args.getVarOrConstOrVectorExpression();
      int argsLen = argsList.size();
      assert argsLen > 0 : "argument list to short " + argsLen;
      if (false) {
      }
      <xsl:for-each select="descendant::xsd:simpleType[@name = 'exprVectorOprBase']">
        <xsl:for-each select="descendant::xsd:restriction/xsd:enumeration/xsd:annotation/xsd:appinfo/kandid:execute">
          else if(opr.getVectorOpr().equals(ExprVectorOprBase.<xsl:value-of select="translate(../../../@value, 'abcdefghijklmnopqrstuvwxyz', 'ABCDEFGHIJKLMNOPQRSTUVWXYZ')"/>)) {
          <xsl:if test="../kandid:args">
            int ax = <xsl:value-of select="../kandid:args"/>;
            while (ax > 0) {
              --ax;
              compileElement(argsList.get(ax % argsLen));
            }
            codeTape[tapeLength] = opr^<xsl:value-of select="../../../@value"/>;
            iOper[tapeLength] = <xsl:value-of select="../kandid:args"/>;
            ++tapeLength;
          </xsl:if>
          <xsl:if test="not(../kandid:args)">
            while (argsLen > 0) {
              --argsLen;
              compileElement(argsList.get(argsLen));
            }
            codeTape[tapeLength] = opr^<xsl:value-of select="../../../@value"/>;
            iOper[tapeLength] = argsList.size();
            ++tapeLength;
          </xsl:if>
          }
        </xsl:for-each>
      </xsl:for-each>
      else {
        assert false : "invalid operator " + opr.getVectorOpr();
      }
    }
  </xsl:template>


  <xsl:template name="execute">
  /**
   * Executes code tape and returns the result from accumulator.
   * @param x
   * @param y
   * @param t
   * @param result
   */
    public void execute(double x, double y, double t, double xscale, double yscale, Vector3d result) {
    x *= Math.exp(xscale);
    y *= Math.exp(yscale);
    double r = Math.sqrt(x * x + y * y);
    double m = Math.abs(x) + Math.abs(y);
    
    Complex cm = new Complex(x, y);
    Complex cm_pow2 = Complex.multiply(cm, cm);
    Complex cm_pow3 = Complex.multiply(cm, cm_pow2);
    
    Complex cm_cos = Complex.cos(cm);

    Complex cm_sinh = Complex.sinh(cm);
    
    double xAccu = 0.0;
    double yAccu = 0.0;
    double zAccu = 0.0;
    int tos = 0;
    int ip = 0;
    int argLength = 0;
    while (ip &lt; tapeLength) {
      switch (codeTape[ip]) {
        case oprStop :
          result.x = xAccu;
          result.y = yAccu;
          result.z = zAccu;
          return;
        case oprPop :
          --tos;
          xAccu = xStack[tos];
          yAccu = yStack[tos];
          zAccu = zStack[tos];
          break;
        case oprPushConst :
          xStack[tos] = xOper[ip];
          yStack[tos] = yOper[ip];
          zStack[tos] = zOper[ip];
          tos++;
          break;
        case oprPushAccu :
          xStack[tos] = xAccu;
          yStack[tos] = yAccu;
          zStack[tos] = zAccu;
          tos++;
          break;
        case oprPushXYT :
          xStack[tos] = x;
          yStack[tos] = y;
          zStack[tos] = t;
          tos++;
          break;
        case oprPushXYR :
          xStack[tos] = x;
          yStack[tos] = y;
          zStack[tos] = r;
          tos++;
          break;
        case oprPushXYM :
          xStack[tos] = x;
          yStack[tos] = y;
          zStack[tos] = m;
          tos++;
          break;
        case oprPushCMPOW :
          xStack[tos] = cm_pow3.re;
          yStack[tos] = cm_pow3.im;
          zStack[tos] = cm_sinh.re;
          tos++;
          break;
        case oprPushCMSINH :
          xStack[tos] = cm_sinh.re;
          yStack[tos] = cm_sinh.im;
          zStack[tos] = cm_cos.im;
          tos++;
          break;
        case oprPushCMIM :
          xStack[tos] = cm_pow2.im;
          yStack[tos] = cm_cos.im;
          zStack[tos] = cm_sinh.im;
          tos++;
          break;
        case oprPushCMRE :
          xStack[tos] = cm_pow2.re;
          yStack[tos] = cm_cos.re;
          zStack[tos] = cm_sinh.re;
          tos++;
          break;
        case oprJmp :
          ip = iOper[ip]-1;
          break;
      <xsl:for-each select="descendant::xsd:simpleType[@name = 'exprVectorOprBase']">
        <xsl:for-each select="descendant::xsd:restriction/xsd:enumeration/xsd:annotation/xsd:appinfo/kandid:execute">
        case opr^<xsl:value-of select="../../../@value"/> :
          argLength = iOper[ip];
          assert argLength == 2 : "argument length of operator '<xsl:value-of select="../../../@value"/>' should be 2.";
          {
            <xsl:value-of select="."/>
          }
          tos -= 2;
          break;
        </xsl:for-each>
      </xsl:for-each>
        default :
          assert false : "invalid code " + codeTape[ip] + " on tape position " + ip;
          break;
      }
      ++ip;
    }
    result.x = xAccu;
    result.y = yAccu;
    result.z = zAccu;
    return;
    }
  </xsl:template>

  <xsl:template name="toString">
  /**
   * @see java.lang.Object#toString()
   */
  public String toString() {
    StringBuffer asm = new StringBuffer();
    int ip = 0;
    while (ip &lt; tapeLength) {
      asm.append(ip + "\t");
      switch (codeTape[ip]) {
        case oprStop :
          asm.append("stop");
          break;
        case oprPop :
          asm.append("pop");
          break;
        case oprPushConst :
          asm.append("pushConst\t");
          asm.append(xOper[ip]);
          asm.append('\t');
          asm.append(yOper[ip]);
          asm.append('\t');
          asm.append(zOper[ip]);
          break;
        case oprPushAccu :
          asm.append("pushAccu");
          break;
        case oprPushXYT :
          asm.append("pushXYT");
          break;
        case oprPushXYR :
          asm.append("pushXYR");
          break;
        case oprPushXYM :
          asm.append("pushXYM");
          break;
        case oprPushCMPOW :
          asm.append("pushCMPOW");
          break;
        case oprPushCMSINH :
          asm.append("pushCMSINH");
          break;
        case oprPushCMIM :
          asm.append("pushCMIM");
          break;
        case oprPushCMRE :
          asm.append("pushCMRE");
          break;
    <xsl:for-each select="descendant::xsd:simpleType[@name = 'exprVectorOprBase']">
      <xsl:for-each select="descendant::xsd:restriction/xsd:enumeration/xsd:annotation/xsd:appinfo/kandid:execute">
        case opr^<xsl:value-of select="../../../@value"/> :
          asm.append("<xsl:value-of select="../../../@value"/>\t");
          asm.append(iOper[ip]);
          break;
      </xsl:for-each>
    </xsl:for-each>
        default :
          asm.append("invalid code\t");
          asm.append(iOper[ip]);
          assert false : "invalid code " + codeTape[ip] + " on tape position " + ip;
          break;
      }
      asm.append('\n');
      ++ip;
    }
    return asm.toString();
  }
  </xsl:template>

  <xsl:template match="*">
  </xsl:template>

</xsl:stylesheet>

