/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.preference.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;

import kandid.Kandid;
import kandid.Console;
import kandid.preference.GenomeDBType;
import kandid.preference.HttpProxyType;
import kandid.preference.KandidType;
import kandid.preference.ObjectFactory;
import kandid.preference.OutgoingMailType;
import kandid.preference.PreferenceType;
import kandid.preference.UserType;
import kandid.util.Debug;

public class PreferenceModel {
  private static PreferenceModel preferencModel;
  
  private PreferenceType preference; 
  
  private JAXBContext jc;
  private ObjectFactory objectFactory;

  /**
   * private constructor for CatalogFactory.
   */
  private PreferenceModel() {
    objectFactory = new ObjectFactory();
    try {
      jc = JAXBContext.newInstance("kandid.preference");
    } catch (JAXBException exc) {
      Debug.stackTrace(exc);
    }
    unmarshalFromFile();
  }

  public static PreferenceModel getInstance() {
    if (preferencModel == null) {
      preferencModel = new PreferenceModel();
    }
    return preferencModel;
  }

  /**
   * Write preferences to file.
   */
  public void marshalToFile() {
    try {
      // create a Marshaller and marshal to a file
      FileOutputStream out = new FileOutputStream(Kandid.preferencesFileName);
      Marshaller m = jc.createMarshaller();
      m.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);
      m.marshal(preference, out);
      out.close();
      Console.append("store preferences " + (new File(Kandid.preferencesFileName).getAbsoluteFile()));
    } catch (JAXBException je) {
      Debug.stackTrace(je);
    } catch (IOException ioe) {
      Debug.stackTrace(ioe);
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
  }

  /**
   * Read preferences from file.
   * 
   * @param inputFileName
   * @return Object
   */
  public Object unmarshalFromFile() {
    preference = null;
    try {
      // create an Unmarshaller
      Unmarshaller u = jc.createUnmarshaller();
      try {
        preference = (PreferenceType) u.unmarshal(new FileInputStream(Kandid.preferencesFileName));
        Console.append("reading preferences " +  (new File(Kandid.preferencesFileName).getAbsoluteFile()));
      } catch (FileNotFoundException exc) {
        PreferenceType defaultPreference = null;
        defaultPreference = objectFactory.createPreferenceType();

        Console.append("missing preferences " +  (new File(Kandid.preferencesFileName).getAbsoluteFile()));
        preference = defaultPreference;
      }

      addDefaults();
    } catch (JAXBException je) {
      Debug.stackTrace(je);
    }

    return preference;
  }

  private void addDefaults() throws JAXBException {
    if(preference == null) {
      preference = objectFactory.createPreferenceType();
    }
    
    if (preference.getKandid() == null) {
      KandidType kandid = objectFactory.createKandidType();
      kandid.setHome("http://");
      
      preference.setKandid(kandid);
    }
    
    if (preference.getUser() == null) {
      UserType user = objectFactory.createUserType();
      user.setNickname("");
      user.setEmail("");
      
      preference.setUser(user);
    }
    
    if (preference.getOutgoingMail() == null) {
      OutgoingMailType outgoingMail = objectFactory.createOutgoingMailType();
      outgoingMail.setSmptHost("");
      
      preference.setOutgoingMail(outgoingMail);
    }
    
    if (preference.getHttpProxy() == null) {
      HttpProxyType httpProxy = objectFactory.createHttpProxyType();
      httpProxy.setEnabled(false);
      httpProxy.setHost("");
      httpProxy.setPort(80);
      httpProxy.setNonProxyHosts("");
      httpProxy.setUsername("");
      httpProxy.setPassword("****");
      
      preference.setHttpProxy(httpProxy);
    }

    if (preference.getGenomeDB() == null) {
      GenomeDBType genomeDBType = objectFactory.createGenomeDBType();
      genomeDBType.setURL("http://www.???.org/KandidGenomeDBv1/KandidConnector_v1.php");
      genomeDBType.setInstallationId(0);
      genomeDBType.setTimeout(50000);

      preference.setGenomeDB(genomeDBType);
    }
  }

  public boolean checkOutgoingMailCompleted() {
    boolean completed = false;
    if (preference != null
        && preference.getUser() != null
        && preference.getOutgoingMail() != null
        && preference.getHttpProxy() != null) {
      completed = preference.getUser().getNickname().length() > 0
                           && preference.getOutgoingMail().getSmptHost().length() > 0
                           && (preference.getHttpProxy().isEnabled() == false || preference.getHttpProxy() .getHost().length() > 0);
    }
    return completed;
  }
  
  public boolean checkLoadGenomeDBCompleted() {
    boolean completed = false;
    if (preference != null
        && preference.getHttpProxy() != null
        && preference.getGenomeDB() != null) {
      completed = preference.getHttpProxy().isEnabled() == false || preference.getHttpProxy() .getHost().length() > 0;
    }
    return completed;
  }
  
  public boolean checkExportGenomeDBCompleted() {
    boolean completed = false;
    if (preference != null
        && preference.getUser() != null
        && preference.getHttpProxy() != null
        && preference.getGenomeDB() != null) {
      completed = preference.getUser().getNickname().length() > 0
                           && (preference.getHttpProxy().isEnabled() == false || preference.getHttpProxy() .getHost().length() > 0);
    }
    return completed;
  }
  
  /**
   * @return
   */
  public boolean fileExists() {
    return (new File(Kandid.preferencesFileName)).exists();
  }

  /**
   * @return Returns the objectFactory.
   */
  public ObjectFactory getObjectFactory() {
    return objectFactory;
  }
  /**
   * @return Returns the preference.
   */
  public PreferenceType getPreference() {
    return preference;
  }
}
