/*
 * Copyright 2025, Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package kandid.noise;

import java.awt.image.BufferedImage;
import java.awt.image.Raster;
import java.io.File;
import java.io.FilenameFilter;
import java.util.Arrays;

import javax.imageio.ImageIO;

import kandid.util.Util;

public class ImageNoise extends AbstractNoise {

  static int[][] argbArray = null;
  static File[] in_files = null;
  static int in_width[];
  static int in_height[];
  int where;

  public ImageNoise(long seed) {
    initializeNoise(seed);
  }

  public void initializeNoise(long seed) {
    try {
      if (in_files == null) {
        in_files = listFiles(new File("./incoming/contrast/")); // TODO
        if(in_files.length > 0) {
          argbArray = new int[in_files.length][];
          in_width  = new int[in_files.length];
          in_height = new int[in_files.length];
          
          for(int in=0; in<in_files.length; ++in) {
            BufferedImage in_image = ImageIO.read(in_files[(int) (Math.abs(seed) % in_files.length)]);
            in_width[in] = in_image.getWidth();
            in_height[in] = in_image.getHeight();
            if(in_width[in] > 0 && in_height[in] > 0 ) {
              argbArray[in] = new int[4 * in_width[in] * in_height[in]];
              Raster raster = in_image.getData();
              raster.getPixels(0, 0, in_width[in], in_height[in], argbArray[in]);
            }
          }

          where = (int) (Math.abs(seed) % in_files.length);
        }
      }
    } catch (Exception e) {
      e.printStackTrace();
      argbArray = null;
    }
  }

  /**
   * Search for files none recursively.
   * 
   * @param directory Starting folder.
   * @return A list of file names with the specified pattern.
   */
  private static File[] listFiles(File directory) {
    File[] entries = directory.listFiles(new FilenameFilter() {
      @Override
      public boolean accept(File entry, String name) {
        return true;
      }
    });
    Arrays.sort(entries);
    return entries;
  }

  private double noiseImage(double x, double y) {
    if(argbArray == null)
      return 0;

    final int ix = (int)(Util.repeat_seamless(x) * in_width[where])  % in_width[where];
    final int iy = (int)(Util.repeat_seamless(y) * in_height[where]) % in_height[where];
    final int index = 4 * (iy * in_width[where] + ix);
    final double gray  = 0.30 * argbArray[where][index] + 0.59 * argbArray[where][index+1] + 0.11 * argbArray[where][index+2];
    return gray / 256.0;
  }

  public double unoise(double x, double y, double z) {
    double dist = noiseImage(x, y);
    return dist;
  }

  public double snoise(double x, double y, double z) {
    double dist = noiseImage(x, y);
    return 2.0 * dist - 1.0;
  }

}
