/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.extensions;

import java.awt.Dimension;
import java.awt.image.BufferedImage;

import junit.framework.TestCase;
import kandid.calculation.Calculation;
import kandid.calculation.bridge.povray.PovTextureCalculation;
import kandid.util.Debug;

/**
 * @author thomas jourdan
 *
 */
public class LRUImageCacheTest extends TestCase {

  private static final int yDim = 100;
  private static final int xDim = 200;
  private Calculation calc;
  private String id1;
  private BufferedImage img1;
  private String id2;
  private BufferedImage img2;
  private String id3;
  private BufferedImage img3;
  /**
   * Constructor for LRUImageCacheTest.
   * @param name
   */
  public LRUImageCacheTest(String name) {
    super(name);
  }

  /* (non-Javadoc)
   * @see junit.framework.TestCase#setUp()
   */
  protected void setUp() throws Exception {
    super.setUp();
    calc = new PovTextureCalculation();
    id1 = "id1";
    img1 = new BufferedImage(xDim, yDim, BufferedImage.TYPE_INT_RGB);
    id2 = "id2";
    img2 = new BufferedImage(xDim, yDim, BufferedImage.TYPE_INT_RGB);
    id3 = "id3";
    img3 = new BufferedImage(xDim, yDim, BufferedImage.TYPE_INT_RGB);

    LRUImageCache lruImageCache = LRUImageCache.getLRUImageCache();
    lruImageCache.put(calc, id1, img1);
    lruImageCache.put(calc, id2, img2);
    for (int i = 2; i < LRUImageCache.maxEntries; i++) {
      lruImageCache.put(calc, "id-filler"+i, img1);      
    }
  }

  public void testLRU() {
    addSequence();
    forgetSequence();
    removeAllSequence();
  }
  
  private void addSequence() {
    LRUImageCache lruImageCache = LRUImageCache.getLRUImageCache();
    assertEquals(lruImageCache.get(id2, new Dimension(xDim, yDim)).getWidth(), img2.getWidth());
    assertEquals(lruImageCache.get(id1, new Dimension(xDim, yDim)).getWidth(), img1.getWidth());
    assertNull(lruImageCache.get(id2, new Dimension(xDim+1, yDim+1)));
    assertNull(lruImageCache.get("id0", new Dimension(xDim, yDim)));    
  }

  private void forgetSequence() {
    LRUImageCache lruImageCache = LRUImageCache.getLRUImageCache();
    lruImageCache.get(id1, new Dimension(xDim, yDim));
    try {
      Thread.sleep(1500);
    }
    catch (InterruptedException exc) {
      Debug.stackTrace(exc);
    }
    lruImageCache.get(id2, new Dimension(xDim, yDim));
    for (int i = 2; i < LRUImageCache.maxEntries; i++) {
      lruImageCache.get("id-filler"+i, new Dimension(xDim, yDim));      
    }
    
    assertEquals(LRUImageCache.maxEntries, lruImageCache.getNofEntries());
    lruImageCache.put(calc, id3, img3);
    assertEquals(LRUImageCache.maxEntries, lruImageCache.getNofEntries());
    
    assertEquals(lruImageCache.get(id2, new Dimension(xDim, yDim)).getWidth(), img2.getWidth());
    assertEquals(lruImageCache.get(id3, new Dimension(xDim, yDim)).getWidth(), img3.getWidth());
    assertNull(lruImageCache.get(id1, new Dimension(xDim, yDim)));
  }

  private void removeAllSequence() {
    LRUImageCache lruImageCache = LRUImageCache.getLRUImageCache();
    assertEquals(LRUImageCache.maxEntries, lruImageCache.getNofEntries());
    assertEquals(lruImageCache.get(id3, new Dimension(xDim, yDim)).getWidth(), img3.getWidth());
    lruImageCache.removeAll(id3);
    assertNull(lruImageCache.get(id3, new Dimension(xDim, yDim)));
    assertEquals(LRUImageCache.maxEntries-1, lruImageCache.getNofEntries());
  }

  public void testDuplicate() {
    LRUImageCache lruImageCache = LRUImageCache.getLRUImageCache();
    lruImageCache.put(calc, id1, img1);
    assertEquals(lruImageCache.get(id1, new Dimension(xDim, yDim)).getWidth(), img1.getWidth());
    lruImageCache.duplicate(id1, "dup1");
    BufferedImage imgDup1 = lruImageCache.get("dup1", new Dimension(xDim, yDim));
    assertNotNull(imgDup1);
    assertEquals(img1.getWidth(), imgDup1.getWidth());
    assertEquals(lruImageCache.get(id1, new Dimension(xDim, yDim)).getWidth(), img1.getWidth());
    assertEquals(lruImageCache.get("dup1", new Dimension(xDim, yDim)).getWidth(), imgDup1.getWidth());
    assertTrue(img1 != imgDup1);
  }

  public static void main(String[] args) {
    junit.textui.TestRunner.run(LRUImageCacheTest.class);
  }


}
