/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.extensions;

import java.awt.Dimension;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import kandid.Kandid;
import kandid.calculation.Calculation;
import kandid.calculation.lca.LcaCalculation;
import kandid.util.Debug;

/**
 * @author thomas jourdan
 * 
 */
public class LRUImageCache {
  private static final String imagetype = "png";
  private static final String fileExtension = ".kcache." + imagetype;
  public final static int maxEntries = 1000;
  private static HashMap<String, CachedImageInfo> lru;
  private static LRUImageCache lruImageCache;

  /**
   * Hidden constructor.
   * @param maxEntries
   */
  private LRUImageCache() {
    lru = new HashMap<>();
  }
 
  /**
   * Factory method returning the image cache.
   * @param maxEntries
   * @return
   */
  public static LRUImageCache getLRUImageCache() {
    if(lruImageCache == null) {
      lruImageCache = new LRUImageCache();
      File cacheDir = new File(Kandid.cacheFolder);   
      if(cacheDir.exists()) {
        try {
          File[] dirList = cacheDir.listFiles();
          for (int fx = 0; fx < dirList.length; fx++) {
            CachedImageInfo cachedImageInfo = new CachedImageInfo();
            String fname = dirList[fx].getName();
            if(dirList[fx].isFile() && dirList[fx].canRead() && fname.toLowerCase().endsWith(fileExtension)) {
              int pos_x = fname.indexOf("_x");
              int pos_y = fname.indexOf("_y");
              int pos_png = fname.indexOf(fileExtension);
              if(pos_x > 1 && pos_y > pos_x && pos_png > pos_y) {
                String sx = fname.substring(pos_x+2, pos_y);
                String sy = fname.substring(pos_y+2, pos_png);
                cachedImageInfo.dim = new Dimension(Integer.parseInt(sx), Integer.parseInt(sy));
          
                cachedImageInfo.id = getQualifiedId(fname.substring(0, pos_x), cachedImageInfo.dim);
                cachedImageInfo.stamp = dirList[fx].lastModified();
                // update hash table
                lru.put(cachedImageInfo.id, cachedImageInfo);
                if (Debug.enabled)
                  System.out.println("cache sync " + cachedImageInfo.id);
              }
            }
          }
        }
        catch (Exception exc) {
          Debug.stackTrace(exc);
        }
      }
      else {
        cacheDir.mkdirs();
      } 
    }
    return lruImageCache;
  }
 
  /**
   * Adds image to cache.
   * @param id
   * @param img
   */
  public synchronized void put(Calculation calculation, String id, BufferedImage img) {
    if(calculation instanceof LcaCalculation) {
      // no speed up, only problems with image size
      return;
    }
    Dimension dim = new Dimension(img.getWidth(), img.getHeight());
    CachedImageInfo cachedImageInfo = lru.get(getQualifiedId(id, dim));
    if (cachedImageInfo == null) {
      // remove from cache
      while (lru.size() >= maxEntries) {
        forget();
      }
      
      cachedImageInfo = new CachedImageInfo();
      cachedImageInfo.id = getQualifiedId(id, dim);
      cachedImageInfo.dim = dim;
      cachedImageInfo.stamp = System.currentTimeMillis();

      try {
        // store image to file system
        File imageOutFile = new File(getQualifiedFilename(cachedImageInfo));
        javax.imageio.ImageIO.write(img, imagetype, imageOutFile);
        imageOutFile.setLastModified(cachedImageInfo.stamp);
        // update hash table
        lru.put(cachedImageInfo.id, cachedImageInfo);
        if (Debug.enabled)
          System.out.println("cache add " + getQualifiedId(id, dim));
      }
      catch (IOException exc) {
        Debug.stackTrace(exc);
      }
    }
  }

  /**
   * Remove one image from cache.
   */
  private void forget() {
    int ne = lru.size();
    long oldest = Long.MAX_VALUE;
    if(Debug.enabled) assert(lru.values() != null);
    // find oldest image
    for (Iterator<CachedImageInfo> iter = lru.values().iterator(); iter.hasNext();) {
      CachedImageInfo cachedImageInfo = iter.next();
      if(cachedImageInfo.stamp < oldest) {
        oldest = cachedImageInfo.stamp;
      }
    }
    // remove one image from cache
    for (Iterator<CachedImageInfo> iter = lru.values().iterator(); iter.hasNext();) {
      CachedImageInfo cachedImageInfo = iter.next();
      if(cachedImageInfo.stamp == oldest) {
        // remove from files system
        (new File(getQualifiedFilename(cachedImageInfo))).delete();
        // remove from hash table
        iter.remove();
        if(Debug.enabled)
          System.out.println("cache forget " + cachedImageInfo.id);
        break;
      }
    }
    if(Debug.enabled) assert(lru.size() == ne-1);
  }

  /**
   * Load image from cache.
   * @param id chromosome id
   * @param dim dimension of the image
   * @return
   */
  public synchronized BufferedImage get(String id, Dimension dim) {
    BufferedImage img = null;
    CachedImageInfo cachedImageInfo = lru.get(getQualifiedId(id, dim));
    if(cachedImageInfo != null) {
      File imageInFile = new File(getQualifiedFilename(cachedImageInfo));
      if(imageInFile.exists())
        try {
          img = javax.imageio.ImageIO.read(imageInFile);
          cachedImageInfo.stamp = System.currentTimeMillis();
          imageInFile.setLastModified(cachedImageInfo.stamp);
          if(Debug.enabled)
            System.out.println("cache hit " + getQualifiedId(id, dim));
        }
        catch (IOException exc) {
          Debug.stackTrace(exc);
        }
      return img;
    }
    if(Debug.enabled)
      System.out.println("cache missing " + getQualifiedId(id, dim));
    return null;
  }

  /**
   * @return number of entries stored in the cache.
   */
  public int getNofEntries() {
    return lru.size();
  }

  /**
   * Removes all cached images with an given id, without respect to the size.
   * @param id
   */
  public synchronized void removeAll(String id) {
    for (Iterator<CachedImageInfo> iter = lru.values().iterator(); iter.hasNext();) {
      CachedImageInfo cachedImageInfo = iter.next();
      if(cachedImageInfo.id.startsWith(id + "_x")) {
        (new File(getQualifiedFilename(cachedImageInfo))).delete();
        iter.remove();
        if(Debug.enabled)
          System.out.println("cache remove " + cachedImageInfo.id);
      }
    }
  }

  /**
   * Same image, but different id.
   * @param oldId
   * @param newId
   */
  public synchronized void duplicate(String oldId, String newId) {
    List<CachedImageInfo> dupList = new ArrayList<>();
    if(Debug.enabled) assert(!oldId.equals(newId));
    // search for images that should be duplicated
    for (Iterator<CachedImageInfo> iter = lru.values().iterator(); iter.hasNext();) {
      CachedImageInfo cachedImageInfo = iter.next();
      if(cachedImageInfo.id.startsWith(oldId + "_x")) {
//!!        cachedImageInfo.stamp = System.currentTimeMillis();
        dupList.add(cachedImageInfo);
      }
    }
    // duplicate
    for (Iterator<CachedImageInfo> dupIter = dupList.iterator(); dupIter.hasNext();) {
      CachedImageInfo cachedImageInfo = dupIter.next();      
      try {
        // read original image and update lru stamp
        File imageInFile = new File(getQualifiedFilename(cachedImageInfo));
        BufferedImage img = javax.imageio.ImageIO.read(imageInFile);
        cachedImageInfo.stamp = System.currentTimeMillis();
        imageInFile.setLastModified(cachedImageInfo.stamp);
        int pos = cachedImageInfo.id.indexOf("_x");
      
        // duplicate image info
        CachedImageInfo dupImageInfo = (CachedImageInfo)cachedImageInfo.clone();
//!!        String fullId = newId + cachedImageInfo.id.substring(pos);
        dupImageInfo.id = newId + cachedImageInfo.id.substring(pos);
//  !!      cachedImageInfo.stamp = System.currentTimeMillis();
        dupImageInfo.stamp = System.currentTimeMillis();
        
        // duplicate image file
        File imageOutFile = new File(getQualifiedFilename(dupImageInfo));
        javax.imageio.ImageIO.write(img, imagetype, imageOutFile);
        imageOutFile.setLastModified(dupImageInfo.stamp);
        
        // update hash table
        lru.put(dupImageInfo.id, dupImageInfo);      
        if(Debug.enabled) System.out.println("cache duplicate " + oldId + " " + newId);
      }
      catch (IOException exc) {
        Debug.stackTrace(exc);
      }
    }
    // remove from cache
    while(lru.size() >= maxEntries) {
      forget();
    }
  }

  /**
   * Update timstamp of cached image.
   * @param id
   * @param dim
   */
  public void touch(String id, Dimension dim) {
    CachedImageInfo cachedImageInfo = lru.get(getQualifiedId(id, dim));
    if(cachedImageInfo != null) {
      cachedImageInfo.stamp = System.currentTimeMillis();
      File imageInFile = new File(getQualifiedFilename(cachedImageInfo));
      imageInFile.setLastModified(cachedImageInfo.stamp);
      if(Debug.enabled)
        System.out.println("cache touch " + getQualifiedId(id, dim));
    }
  }

  /**
   * Concatenates chromosome id and image dimension.
   * @param id chromosome id
   * @param dim dimension of the image
   * @return
   */
  private static String getQualifiedId(String id, Dimension dim) {
    return id + "_x" + dim.width + "_y" + dim.height;
  }

  /**
   * Concatenates path, filename and extension.
   * @param id chromosome id
   * @param dim dimension of the image
   * @return
   */
  private static String getQualifiedFilename(CachedImageInfo cachedImageInfo) {
    return Kandid.cacheFolder + "/" + cachedImageInfo.id + fileExtension;
  }

}
