/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.extensions;

import java.io.File;

import kandid.*;
import kandid.util.*;

/**
 * Helper for exporting images to different file formats and printer.
 * @author thomas jourdan
 */
public class ExportState {
  public static final int FILE_EXPORT = 1;

  public int    exportDestination = FILE_EXPORT;
  public int    exportType    = Gateway.BITMAP_FILE;
  public String fileExtension = "png";
  public String fileName      = "out";
  public int    xDim          = 2048;
  public int    yDim          = 2048;
  public String to            = "genepool@???.org";

  /**
   * get path to subdirectory
   * @param baseFolder
   *
   * @return
   */
  public String getPath(String baseFolder) {
    return getIdPath(baseFolder, -1);
  }

  /**
   * get path to subdirectory
   * @param baseFolder
   *
   * @return
   */
  public String getIdPath(String baseFolder, int id) {
    if(fileName.length() > 3) {
      int pos = fileName.indexOf('_');
      pos = fileName.indexOf('_', pos+1);
      pos = fileName.indexOf('_', pos+1);
      String prefix = fileName.substring(0, pos);
      String path = baseFolder + "/" ;
      if(id >= 0)
        path += id + "/";
      return path + prefix.replace('_', '/') + "/";
    }
    return "";
  }

  /**
   * get filename and extension
   *
   * @return
   */
  public String getFullname() {
    return fileName + "." + fileExtension;
  }

  /**
   * get filename
   *
   * @return
   */
  public String getFilename() {
    return fileName;
  }

  /**
   * get file extension
   *
   * @return
   */
  public String getExtension() {
    return fileExtension;
  }

  /**
   * set filename and extension
   *
   * @param fn file name and extension 
   */
  public void setFullname(String name) {
    int lastFileSeperator = name.lastIndexOf("/");
    if(lastFileSeperator > -1) {
      name = name.substring(lastFileSeperator+1);
    }
    int pos = name.lastIndexOf('.');
    if (pos < 0) {
      this.fileName = name;
    }
    else {
      fileName      = name.substring(0, pos);
      fileExtension = name.substring(pos + 1);
    }
    if(Debug.enabled) System.out.println(this.toString());
  }

  /**
   * @param chromosomeId
   * @param coloratorShortName
   */
  public void setFullname(String chromosomeId, String coloratorShortName) {
    int firstSeperator = chromosomeId.indexOf("_");
    if(firstSeperator > -1) {
      String part1 = chromosomeId.substring(0, firstSeperator);
      String part2 = chromosomeId.substring(firstSeperator+1);
      setFullname(part1 + "_" + coloratorShortName + "_" +  part2);
    }
  }

  /**
   * builds nice name
   *
   * @returns export type, dimension and file name as string.
   */
  public String toString() {
    return exportType + ", " + xDim + "x" + yDim + ", " + getFullname();
  }

  /**
   * @param baseFolder
   * @return
   */
  public String getAbsolutPath(String baseFolder) {
    File folder = new File(getPath(baseFolder));
    return folder.getAbsolutePath();
  }

}