/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.extensions;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;

import kandid.Console;
import kandid.Gateway;
import kandid.Kandid;
import kandid.calculation.Calculation;
import kandid.calculation.Deviation;
import kandid.calculation.PixelCalculation;
import kandid.calculation.bridge.PixelBridgeCalculation;
import kandid.colorator.Colorator;
import kandid.soup.ChromosomeType;

/**
 * 
 * @author thomas jourdan
 */
public class BitmapFileEngine extends BitmapEngine {

  /**
   * Constructor BitmapFileEngine
   *
   * @param chromosome chromosome of this image
   * @param calculationName name of the calculation for this image
   * @param colorator color strategy for this image
   * @param exportState detail information like file name and size
   */
  public BitmapFileEngine(ChromosomeType chromosome, String calculationName, Colorator colorator, ExportState exportState) {
    super(chromosome, calculationName, colorator, exportState);
  }

  /**
   * @see kandid.extensions.BitmapEngine#generateOutput()
   */
  protected void generateOutput(Gateway gw, Deviation dev, String timestamp) throws IOException {
    Calculation calc = gw.getCalculation();
    if(!(calc instanceof PixelBridgeCalculation)) {
      File imageOutFile;
      if(dev == null) {
        imageOutFile = new File(exportState.getPath(Kandid.localImageFolder) + exportState.getFullname());
      }
      else {
        if(timestamp == null) {
          imageOutFile = new File(exportState.getPath(Kandid.localImageFolder) + "kandid_" + String.format("%06d", dev.frame) + "_" + exportState.getFullname());
        }
        else {
          File imageOutPath = new File(exportState.getPath(Kandid.localImageFolder) + timestamp);
          imageOutPath.mkdirs();
          imageOutFile = new File(imageOutPath + "/kandid_" + String.format("%06d", dev.frame) + "_" + exportState.getFullname());        }
      }
      
      String formatName = exportState.fileExtension.toLowerCase();
      if(calc instanceof PixelCalculation
          && (formatName.equals("png") || formatName.equals("pfm") || formatName.equals("jpg"))) {
        switch(formatName) {
        case "png":
          ((PixelCalculation)calc).writeRGB_PNG16(exportState, imageOutFile);
          break;          
        case "pfm":
          ((PixelCalculation)calc).writeRGB_PFM(exportState, imageOutFile);
          break;          
        case "jpg":
          ((PixelCalculation)calc).writeRGB_JPG(imageOutFile);
          break;          
        }
        Console.append(exportState.getAbsolutPath(Kandid.localImageFolder) + "/" + exportState.getFullname() + " is ready");
      }
      else { // calc instanceof VectorCalculation
        if(formatName.equals("pfm")) {
          formatName = "png";
          Console.append("Portable float map PFM is not supported for calculation based on vector graphics, yet. Will use " + formatName.toUpperCase() + " format instead.");
          File patched_imageOutFile = new File(exportState.getPath(Kandid.localImageFolder) + exportState.getFilename() + "." + formatName);
          javax.imageio.ImageIO.write((BufferedImage) gw.getImage(), formatName, patched_imageOutFile);
//          Console.append(exportState.getAbsolutPath(Kandid.localImageFolder) + "/" + patched_imageOutFile + " is ready");
          Console.append(patched_imageOutFile.toString() + " is ready");
        }
        else {
          javax.imageio.ImageIO.write((BufferedImage) gw.getImage(), formatName, imageOutFile);
          Console.append(exportState.getAbsolutPath(Kandid.localImageFolder) + "/" + exportState.getFullname() + " is ready");
        }
      }
    }
  }

}
