/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.extensions;

import java.awt.Dimension;
import java.io.File;
import java.io.IOException;
import java.text.FieldPosition;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import kandid.Gateway;
import kandid.Kandid;
import kandid.calculation.Calculation;
import kandid.calculation.Deviation;
import kandid.colorator.Colorator;
import kandid.soup.ChromosomeType;
import kandid.util.Debug;

/**
 * Exports current image to bit map file format.
 * This class uses javax.imageio for exporting.
 * Supported file formats are PNG and JPEG. See javax.imageio API documentation.
 * @author thomas jourdan
 */
public abstract class BitmapEngine extends ExportEngine implements Runnable {
  boolean with_deviations;
  int frames = 500;

  /**
   * Constructor BitmapEngine
   *
   * @param chromosome chromosome of this image
   * @param calculationName name of the calculation for this image
   * @param colorator color strategy for this image
   * @param exportState detail information like file name and size
   */
  public BitmapEngine(ChromosomeType chromosome, String calculationName, Colorator colorator, ExportState exportState) {
    super(chromosome, calculationName, colorator, exportState);
  }
  
  protected abstract void generateOutput(Gateway gw, Deviation dev, String timestamp) throws IOException;

  /**
   * start export engine
   */
  public void start(boolean with_deviations) {
    this.with_deviations = with_deviations;
    if (engine == null) {
      engine = new Thread(this, "engine");
      engine.start();
    }
  }

  /**
   * stop export engine
   */
  public void stop() {
    engine = null;
  }

  /**
   * calculate one or more bitmaps and write them to the file system
   */
  public void run() {
    (new File(exportState.getPath(Kandid.localImageFolder))).mkdirs();
    if(with_deviations) {
      run_parallel();
    }
    else {
      run_single();
    }
  }
 
  /**
   * calculate a single bitmap and write it to the file system
   */
  private void run_single() {
    Thread me = Thread.currentThread();
    me.setPriority(Thread.MIN_PRIORITY);
    Gateway gw = new Gateway();
    try {
      long startTime = System.currentTimeMillis();
      calculate_image(gw);
      while (!gw.getReady() && (me == engine)) {
        gw.calculate(null, false, exportState.getPath(Kandid.localImageFolder) + exportState.getFilename());
      }
      generateOutput(gw, null, "");
      if(Debug.enabled) System.out.println("engine ready " + (System.currentTimeMillis()-startTime) + "ms");
    } catch (Throwable exc) {
      kandid.Console.append("error while single image exporting ");
      kandid.Console.append(exc.toString());
      Debug.stackTrace(exc);
    }
  }

  /**
   * parallel calculate a a lot of bitmaps and write all to the file system
   */
  static final int AP = Runtime.getRuntime().availableProcessors();
  static final int FRAME_QUEUE_LEN = 2 * AP;
  static final int FRAME_QUEUE_CONSUMERS = AP + AP / 2;
  
  private class TaskProducer implements Runnable {
    private final BlockingQueue<Deviation> frame_queue;
    
    public TaskProducer(BlockingQueue<Deviation> frame_queue) {
      super();
      this.frame_queue = frame_queue;
    }

    @Override
    public void run() {
      if(Debug.enabled) System.out.println("-- start  TaskProducer");
      for(int frm=0; frm < frames; ++frm) {
        try {
          switch(calculationName) {
          case "kandid.calculation.vm.scalar.ScalarExpressionCalculation":
            frame_queue.put(new kandid.calculation.vm.scalar.Deviation(frm, frames, false));
            if(Debug.enabled) System.out.println("put new Deviation " + frm);
            break;
          case "kandid.calculation.domainwarping.DomainWarpingCalculation":
            frame_queue.put(new kandid.calculation.domainwarping.Deviation(frm, frames, false));
            if(Debug.enabled) System.out.println("put new Deviation " + frm);
            break;
          }
        } catch (InterruptedException e) {
          e.printStackTrace();
        }
      }
      for(int frm=0; frm < FRAME_QUEUE_CONSUMERS; ++frm) {
        try {
          Deviation dev = new Deviation(0, true);
          frame_queue.put(dev);
          if(Debug.enabled) System.out.println("put stop ");
        } catch (InterruptedException e) {
          e.printStackTrace();
        }
      }
      if(Debug.enabled) System.out.println("-- end of TaskProducer");
    }
    
  }
  
  private class TaskCalculator implements Runnable {
    private final String timestamp;
    private final BlockingQueue<Deviation> frame_queue;
    
    public TaskCalculator(String timestamp, BlockingQueue<Deviation> frame_queue) {
      super();
      this.timestamp = timestamp;
      this.frame_queue = frame_queue;
    }

    @Override
    public void run() {
      if(Debug.enabled) System.out.println("-- start  TaskCalculator");
      try {
        while(true) {
          Deviation dev = frame_queue.take();
          if(dev.stop)
            break;
          if(Debug.enabled) System.out.println("take Deviation " + dev.frame);
          Gateway gw = new Gateway();
          calculate_image(gw);
          while (!gw.getReady()) {
            gw.calculate(dev, false, exportState.getPath(Kandid.localImageFolder) + exportState.getFilename());
          }
          generateOutput(gw, dev, timestamp);
        }
      } catch (Throwable exc) {
        exc.printStackTrace();
        kandid.Console.append("error while exporting deviation " + exportState.getFilename());
        kandid.Console.append(exc.toString());
        Debug.stackTrace(exc);
      }
      if(Debug.enabled) System.out.println("-- end of TaskCalculator");
    }
    
  }

  private void run_parallel() {
    final BlockingQueue<Deviation> frame_queue = new LinkedBlockingQueue<Deviation>(FRAME_QUEUE_LEN);

    SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd-HH-mm-ss-S");
    StringBuffer stringBuffer = new StringBuffer();
    simpleDateFormat.format(new Date(), stringBuffer, new FieldPosition(0));
    final String timestamp = stringBuffer.toString();

    TaskProducer tp = new TaskProducer(frame_queue);
    new Thread(tp, "TaskProducer").start();

    for(int tcon=0; tcon<FRAME_QUEUE_CONSUMERS; ++tcon) {
      TaskCalculator tc = new TaskCalculator(timestamp, frame_queue);
      new Thread(tc, "TaskCalculator-"+tcon).start();
    }

//    Thread me = Thread.currentThread();
//    me.setPriority(Thread.MIN_PRIORITY);
//    
//    try {
//      long startTime = System.currentTimeMillis();
//      (new File(exportState.getPath(Kandid.localImageFolder))).mkdirs();
//      (new File(exportState.getPath(Kandid.localExportFolder))).mkdirs();
//      while(dev.next()) { 
//        calculate_image();
//        while (!gw.getReady() && (me == engine)) {
//          gw.calculate(dev, false, exportState.getPath(Kandid.localImageFolder) + exportState.getFilename());
//        }
//        generateOutput(dev, timestamp);
//      }
//      if(Debug.enabled) System.out.println("engine ready " + (System.currentTimeMillis()-startTime) + "ms");
//    } catch (Throwable exc) {
//      kandid.Console.append("error while deviation exporting ");
//      kandid.Console.append(exc.toString());
//      Debug.stackTrace(exc);
//    }
  }

  private void calculate_image(Gateway gw) {
    // calculate image
    kandid.Console.append("exporting " + exportState.getAbsolutPath(Kandid.localImageFolder) + exportState.getFullname());
    Calculation calculation = gw.createCalculation(calculationName);
    if(Debug.enabled) System.out.println("calculation " + calculation);
    gw.setChromosome(chromosome, calculation, colorator);
    gw.activateBitmapCanvas(new Dimension(exportState.xDim, exportState.yDim));
    gw.getCalculation().setImageFormat(exportState.fileExtension);
  }
  
}
