/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.colorator;

import kandid.soup.ChromosomeType;
import kandid.soup.LookUpColorGene;
import kandid.soup.LookUpTableChromosome;
import kandid.soup.TransparentLookUpTableChromosome;

public class LookUpTableColorator implements Colorator {
  private LookUpColorGene hsbSegment[];
  private ColorF32 rgbSegment[];

  /**
   * Uses value as an index to the color table
   *
   * @param value a value between 0.0 (inclusive) and 1.0 (exclusive).
   */
  public void getColor(double value, ColorF32 cout) {
    assert cout != null;
    int ix = (int)(rgbSegment.length * value);
    if (ix >= rgbSegment.length) {
      ix = rgbSegment.length-1;
    }
    else if (ix < 0) {
      ix = 0;
    }
    ColorF32.copy(rgbSegment[ix], cout);
  }

  /**
   * Convert red / green / blue to gray scale.
   *
   * @param redValue a value between 0.0 (inclusive) and 1.0 (exclusive).
   * @param greenValue a value between 0.0 (inclusive) and 1.0 (exclusive).
   * @param blueValue a value between 0.0 (inclusive) and 1.0 (exclusive).
   */
  public void getColor(double redValue, double greenValue, double blueValue, ColorF32 cout) {
    getNearestColor((float)redValue, (float)greenValue, (float)blueValue, cout);
  }
  
  /**
   * Method getColor.
   * @param colorGene
   */
  public void getColor(kandid.soup.ColorGene colorGene, ColorF32 cout) {
    getNearestColor(colorGene.getRed(), colorGene.getGreen(), colorGene.getBlue(), cout);
  }

  /**
   * Method getColor.
   * @param color
   */
  public void getColor(java.awt.Color color, ColorF32 cout) {
    getNearestColor(color.getRed() / 256, color.getGreen() / 256, color.getBlue() / 256, cout);
  }
  
  /**
   * Method getNearestColor.
   * @param red
   * @param green
   * @param blue
   * @return int
   */
  private void getNearestColor(float red, float green, float blue, ColorF32 cout) {
    assert cout != null;
    ColorF32 rgb = new ColorF32(red, green, blue);
    ColorF32 hsb = new ColorF32();
    ColorF32.rgb2hsb(rgb, hsb);
    int lx = 0;
    int nearestIndex = 0;
    double minDistance = Double.MAX_VALUE;
    while(lx < hsbSegment.length) {
      double dist = hsbDistance(hsbSegment[lx], hsb);
      if(dist < minDistance) {
        minDistance = dist;
        nearestIndex = lx;
      }
      ++lx;
    }
    LookUpColorGene nearestColorSegment = hsbSegment[nearestIndex];
    hsb.r = (float)(double)nearestColorSegment.getHue().getValue();
    hsb.g = (float)(double)nearestColorSegment.getSaturation().getValue();
    hsb.b = (float)(double)nearestColorSegment.getBrightness().getValue();
    ColorF32.hsb2rgb(hsb, cout);
  }

  /**
   * Method hsbDistance.
   * @param colorSegment
   * @param hsb
   * @return double
   */
  private static double hsbDistance(LookUpColorGene colorSegment, ColorF32 hsb) {
    double diffHue = diffHue(colorSegment.getHue().getValue(), hsb.r);
    double diffSaturation = colorSegment.getSaturation().getValue() - hsb.g;
    double diffBrightness = colorSegment.getBrightness().getValue() - hsb.b;
    return diffHue*diffHue+diffSaturation*diffSaturation+diffBrightness*diffBrightness;
  }


  /**
   * Method diffHue.
   * @param hue1
   * @param hue2
   * @return double
   */
  public static double diffHue(double hue1, double hue2) {
    hue1 = hue1 - Math.floor(hue1);
    hue2 = hue2 - Math.floor(hue2);
    double diff1 = Math.abs(hue1 - hue2);
    double diff2 = Double.MAX_VALUE;
    if(hue1 < hue2) {
      diff2 = Math.abs((1.0+hue1) - hue2);
    }
    if(hue2 < hue1) {
      diff2 = Math.abs(hue1 - (1.0+hue2));
    }
    return diff1 < diff2 ? diff1 : diff2;
  }


  public static boolean isMinDiffCW(double hue1, double hue2) {
    hue1 = hue1 - Math.floor(hue1);
    hue2 = hue2 - Math.floor(hue2);
    boolean isCW1 = (hue2 - hue1) > 0.0;
    double diff1 = Math.abs(hue1 - hue2);
    
    boolean isCW2 = false;
    double diff2 = Double.MAX_VALUE;
    if(hue1 < hue2) {
      diff2 = Math.abs((1.0+hue1) - hue2);
    }
    if(hue2 < hue1) {
      diff2 = Math.abs(hue1 - (1.0+hue2));
      isCW2 = true;
    }
    return diff1 < diff2 ? isCW1 : isCW2;
  }


  /**
   * Method getShortName.
   * @return String
   */
  public String getShortName() {
    return "LUT";
  }

  /**
   * @see kandid.colorator.Colorator#setColorTable(ChromosomeType)
   */
  public void setColorTable(kandid.soup.ChromosomeType colorTable) {
    int size = 0;
    if(colorTable instanceof LookUpTableChromosome)
      size = ((LookUpTableChromosome)colorTable).getLutColor().size();
    else if(colorTable instanceof TransparentLookUpTableChromosome)
      size = ((TransparentLookUpTableChromosome)colorTable).getLutColor().size();
    assert size >= 2 : "too less entries in color table";
    hsbSegment = new LookUpColorGene[size];
    rgbSegment = new ColorF32[size];
    if(colorTable instanceof LookUpTableChromosome)
      ((LookUpTableChromosome)colorTable).getLutColor().toArray(hsbSegment);
    else if(colorTable instanceof TransparentLookUpTableChromosome)
      ((TransparentLookUpTableChromosome)colorTable).getLutColor().toArray(hsbSegment);
    for (int ix = 0; ix < rgbSegment.length; ++ix) {
      rgbSegment[ix] = new ColorF32();
      ColorF32 hsb = new ColorF32((float)(double)hsbSegment[ix].getHue().getValue(),
                                  (float)(double)hsbSegment[ix].getSaturation().getValue(),
                                  (float)(double)hsbSegment[ix].getBrightness().getValue());
      ColorF32.hsb2rgb(hsb, rgbSegment[ix]);
    }
  }

  /**
   * Get color value from a look up table.
   *
   * @param index an index between 0 and colorTable.length-1.
   */
  public void getDiscretColor(int index, ColorF32 cout) {
    assert cout != null;
    if (index >= rgbSegment.length) {
      index = rgbSegment.length-1;
    }
    else if (index < 0) {
      index = 0;
    }
    ColorF32.copy(rgbSegment[index], cout);
  }

  /**
   * Get number of entries in the color table.
   *
   * @return a number >= 2.
   */
  public int getNumberOfDiscretColors() {
    return rgbSegment.length;
  }

}
