/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.voronoi;

import java.awt.Component;
import java.awt.Dimension;
import java.util.List;

import kandid.calculation.Deviation;
import kandid.colorator.ColorF32;
import kandid.soup.KingGene;
import kandid.soup.VoronoiTransparentChromosome;

/**
 * @author thomas jourdan
 */
public class VoronoiTransparentCalculation extends VoronoiCalculationBase implements Cloneable {
  protected double kingRed[];
  protected double kingGreen[];
  protected double kingBlue[];
  protected int    prioritySize;
  protected double priorityWeight;

  /* For documentation please see super class RefinementCalculation. */
  public void activateCanvas(Component viewComponent, Dimension canvasSize, boolean zoomMode) {
    depth = 6;
    xwMin = 0.0;
    xwMax = 1.0;
    ywMin = 0.0;
    ywMax = 1.0;
    super.activateCanvas(viewComponent, canvasSize, zoomMode);
    VoronoiTransparentChromosome voronoiTransparentChromosome = (VoronoiTransparentChromosome) chromosome;

    prioritySize = voronoiTransparentChromosome.getPrioritySize().getValue();
    priorityWeight = voronoiTransparentChromosome.getPriorityWeight().getValue();

    List<KingGene> kingList = voronoiTransparentChromosome.getKing();
    numberOfKings = kingList.size();
    kingRed = new double[numberOfKings];
    kingGreen = new double[numberOfKings];
    kingBlue = new double[numberOfKings];
    for (int kx = 0; kx < numberOfKings; ++kx) {
      KingGene kingGene = kingList.get(kx);
      kingRed[kx]   = kingGene.getColor().getRed();
      kingGreen[kx] = kingGene.getColor().getGreen();
      kingBlue[kx]  = kingGene.getColor().getBlue();
    }
  }

  /* For documentation please see super class RefinementCalculation. */
  public void calculate(Deviation dev, boolean paintOnScreen, String doExport) {
    ColorF32 colorF32 = new ColorF32();
    if (enterDepth()) {
      while (more) {
        PairList nearest = new PairList(prioritySize);
        int kx = numberOfKings;
        while (kx > 0) {
          --kx;
          double dist = kw[kx] * distance(kx);
          nearest.insert(new Pair(dist, kx));
        }
        double sumRed = 0;
        double sumGreen = 0;
        double sumBlue = 0;
        double factor = 1;
        double divisor = 0;
        for(int px = prioritySize-1; px >= 0; --px) {
          int nk = nearest.getKing(px);
          if(nk >= 0) {
            sumRed   += factor*kingRed[nk];
            sumGreen += factor*kingGreen[nk];
            sumBlue  += factor*kingBlue[nk];
            divisor  += factor;
            factor   *= priorityWeight;
          }
          else {
            break;
          }
        }
        colorF32.r = (float) (sumRed / divisor);
        colorF32.g = (float) (sumGreen / divisor);
        colorF32.b = (float) (sumBlue / divisor);
        setPixel(colorF32);
        nextLocation();
      }
    }
  }
  
  public boolean hasWhiteBackground() {
    return false;
  }
}
