/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.voronoi;

import java.awt.Component;
import java.awt.Dimension;
import java.util.List;

import kandid.calculation.RefinementCalculation;
import kandid.soup.*;

/**
 * 
 * @author thomas jourdan
 */
public abstract class VoronoiCalculationBase extends RefinementCalculation implements Cloneable {
  protected double dw[];
  protected DistanceFunctionBase df[];
  protected int numberOfDistanceFunctions;
  protected double kw[];
  protected double ykp[];
  protected double xkp[];
  protected int numberOfKings;

  /* For documentation please see super class RefinementCalculation. */
  public boolean reject(ChromosomeType chromosome) {
    return false;
  }

  /* For documentation please see super class RefinementCalculation. */
  public void activateCanvas(Component viewComponent, Dimension canvasSize, boolean zoomMode) {
    depth = 6;
    xwMin = 0.0;
    xwMax = 1.0;
    ywMin = 0.0;
    ywMax = 1.0;
    super.activateCanvas(viewComponent, canvasSize, zoomMode);
    VoronoiChromosomeBase voronoiChromosomeBase = (VoronoiChromosomeBase) chromosome;

    List<KingGene> kingList = voronoiChromosomeBase.getKing();
    numberOfKings = kingList.size();
    xkp = new double[numberOfKings];
    ykp = new double[numberOfKings];
    kw = new double[numberOfKings];
    for (int kx = 0; kx < numberOfKings; ++kx) {
      KingGene kingGene = kingList.get(kx);
      xkp[kx] = kingGene.getPoint().getX();
      ykp[kx] = kingGene.getPoint().getY();
      kw[kx] = kingGene.getWeight().getValue();
    }

    List<DistanceFunctionGene> dfList = voronoiChromosomeBase.getDistance();
    numberOfDistanceFunctions = dfList.size();
    df = new DistanceFunctionBase[numberOfDistanceFunctions];
    dw = new double[numberOfDistanceFunctions];
    for (int kx = 0; kx < numberOfDistanceFunctions; ++kx) {
      DistanceFunctionGene distanceFunctionGene = dfList.get(kx);
      df[kx] = distanceFunctionGene.getDistanceFunction();
      dw[kx] = distanceFunctionGene.getWeight().getValue();
    }
  }

  /**
   * Method distance.
   * @param kx
   */
  protected double distance(int kx) {
    // helper variables
    double xd = xw - xkp[kx];
    double yd = yw - ykp[kx];
    // calulate sum of all distance functions   
    double dist = 0.0;
    for (int dx = 0; dx < numberOfDistanceFunctions; ++dx) {
      if (dw[dx] > 0.0) {
        String dfx = df[dx].value();
        if (dfx.equals("abs(dx,dy)")) {
          dist += dw[dx] * Math.abs(xd) + Math.abs(yd);
        } else if (dfx.equals("abs(dx)")) {
          dist += dw[dx] * Math.abs(xd);
        } else if (dfx.equals("abs(dy)")) {
          dist += dw[dx] * Math.abs(yd);
        } else if (dfx.equals("dx")) {
          dist += dw[dx] * xd;
        } else if (dfx.equals("dy")) {
          dist += dw[dx] * yd;
        } else if (dfx.equals("euclide(dx,dy)")) {
          dist += dw[dx] * Math.sqrt(xd * xd + yd * yd);
        } else if (dfx.equals("product(dx,dy)")) {
          dist += dw[dx] * Math.abs(xd) * Math.abs(yd);
        } else if (dfx.equals("quotient(dx,dy)")) {
          dist += dw[dx] * Math.abs(xd) / (Math.abs(yd) + 1.0);
        }
      }
    }
    return dist;
  }

}
