/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.vm.vector;

import javax.vecmath.Vector3d;

import junit.framework.TestCase;
import kandid.calculation.vm.scalar.ScalarVM;
import kandid.soup.VectorExprGene;
import kandid.soup.util.SoupFactory;

/**
 * 
 * @author thomas jourdan
 */
public class VectorVMTest extends TestCase {

	private VectorVM vectorVM;
	private SoupFactory soupFactory;

	/**
	 * Constructor for VectorVMTest.
	 * @param arg0
	 */
	public VectorVMTest(String arg0) {
		super(arg0);
	}

	/**
	  * @see junit.framework.TestCase#setUp()
	  */
	protected void setUp() throws Exception {
		super.setUp();
		vectorVM = new VectorVM(151258);
		soupFactory = SoupFactory.getSoupFactory();
	}

	private void compileExpr(String expr) {
		kandid.soup.EntityType entity = (kandid.soup.EntityType) soupFactory.unmarshalFromString(expr);
		VectorExprGene vectorExprGene = entity.getVectorExpression().getChromosome().getVectorExpression();
		System.out.println("----\n");
		vectorVM.compile(vectorExprGene);
		System.out.println("" + vectorVM);
	}

	private Vector3d executeLispExpr(String expr, double x, double y, Vector3d result) {
		compileExpr((new VectorLisp()).getAsXML(expr));
		//!!		Vector3d result = new Vector3d();
		vectorVM.execute(x, y, ScalarVM.epsilon, 1.0, 1.0, result);
		return result;
	}

	private String trimLines(String expr) {
		char[] inp = expr.toCharArray();
		StringBuffer trimmed = new StringBuffer();
		for (int ix = 0; ix < inp.length; ix++) {
			if (inp[ix] != ' ') {
				trimmed.append(inp[ix]);
			}
		}
		return trimmed.toString();
	}

	private String expr1 =
		VectorHeaderFooter.header
			+ "<vectorExpression>\n"
			+ "  <list>\n"
			+ "    <opr>\n"
			+ "      <vectorOpr>sub</vectorOpr>\n"
			+ "    </opr>\n"
			+ "    <args>\n"
			+ "      <const>\n"
			+ "        <cx>\n"
			+ "          <value>0.7</value>\n"
			+ "        </cx>\n"
			+ "        <cy>\n"
			+ "          <value>0.8</value>\n"
			+ "        </cy>\n"
			+ "        <cz>\n"
			+ "          <value>0.9</value>\n"
			+ "        </cz>\n"
			+ "      </const>\n"
			+ "      <var>\n"
			+ "        <vectorVar>xyr</vectorVar>\n"
			+ "      </var>\n"
			+ "    </args>\n"
			+ "  </list>\n"
			+ "</vectorExpression>\n"
			+ VectorHeaderFooter.footer;

	public void testGetAsXML() {
		assertEquals(trimLines(expr1), trimLines((new VectorLisp()).getAsXML("(sub [0.7 0.8 0.9] xyr)")));
	}

	public void ignore_testExecuteOpr() {
		Vector3d result = new Vector3d();
		executeLispExpr("(add [0.7 0.8 0.9] xyr)", 1, 2, result);
		assertEquals(new Vector3d(1.7, 2.8, Math.sqrt(5) + 0.9), result);

		executeLispExpr("(sub [0.7 0.8 0.9] xyr)", 1, 2, result);
		assertEquals(new Vector3d(-0.3, -1.2, -Math.sqrt(5) + 0.9), result);

		executeLispExpr("(mult [0.7 0.8 0.9] xyr)", 1, 2, result);
		assertEquals(new Vector3d(0.7, 1.6, Math.sqrt(5) * 0.9), result);

		executeLispExpr("(mod [1.7 2.8 -0.9] xyr)", 1, -1, result);
		assertEquals(new Vector3d(0.7, 0.8, -0.9), result);

		executeLispExpr("(cross [0.7 0.8 0.9] xyr)", 3, 4, result);
		assertEquals(new Vector3d(0.4, -0.8, 0.4), result);

		executeLispExpr("(sincos [0.7 0.8 0.9] xyr)", 1, 2, result);
		assertEquals(new Vector3d(-0.19098300562505255, 1.5877852522924734, 1.046385872453267), result);

		executeLispExpr("(atan [0.7 0.8 0.9] xyr)", 1, 2, result);
		assertEquals(new Vector3d(0.19440011221421477, 0.12111894159084341, 0.12180223551427653), result);

		executeLispExpr("(atan [0 0 0] xyr)", 0, 0, result);
		assertEquals(new Vector3d(0, 0, 0), result);

		executeLispExpr("(maxvalue [0.7 0.8 99] xyr)", 3, 4, result);
		assertEquals(new Vector3d(3, 4, 99), result);
		executeLispExpr("(maxvalue [10.7 0.8 0.9] xyr)", 3, -4, result);
		assertEquals(new Vector3d(10.7, 0.8, 5), result);

		executeLispExpr("(minvalue [0.7 0.8 9] xyr)", 3, 4, result);
		assertEquals(new Vector3d(0.7, 0.8, 5), result);
		executeLispExpr("(minvalue [10.7 0.8 0.9] xyr)", 3, -4, result);
		assertEquals(new Vector3d(3, -4, 0.9), result);
		executeLispExpr("(minvalue [2 2 3] xym)", 1, 1, result);
		assertEquals(new Vector3d(1, 1, 2), result);

		executeLispExpr("(maxvector xyr [0.7 0.8 0.9])", 3, 4, result);
		assertEquals(new Vector3d(3, 4, 5), result);
		executeLispExpr("(maxvector [0.7 0.8 0.9] xyr)", 3, 4, result);
		assertEquals(new Vector3d(3, 4, 5), result);
		executeLispExpr("(maxvector xyr [0.7 0.8 -99])", 3, 4, result);
		assertEquals(new Vector3d(0.7, 0.8, -99), result);

		executeLispExpr("(minvector xyr [0.7 0.8 0.9])", 3, 4, result);
		assertEquals(new Vector3d(0.7, 0.8, 0.9), result);
		executeLispExpr("(minvector [0.7 0.8 0.9] xyr)", 3, 4, result);
		assertEquals(new Vector3d(0.7, 0.8, 0.9), result);
		executeLispExpr("(minvector xyr [0.7 0.8 -99])", 3, 4, result);
		assertEquals(new Vector3d(3, 4, 5), result);

		executeLispExpr("(maxmix [0 0 0] xym)", 1, 1, result);
		assertEquals(new Vector3d(1, 1, 2), result);
		executeLispExpr("(maxmix [2 2 3] xym)", 1, 1, result);
		assertEquals(new Vector3d(1.7391304347826086, 1.7391304347826086, 2.739130434782609), result);
		executeLispExpr("(maxmix [2 2 3] xym)", 100, 100, result);
		assertEquals(new Vector3d(99.97224119832714, 99.97224119832714, 199.944199143576), result);

		executeLispExpr("(invmix [0 0 0] xym)", 1, 1, result);
		assertEquals(new Vector3d(0, 0, 0), result);
		executeLispExpr("(invmix [2 2 3] xym)", 1, 1, result);
		assertEquals(new Vector3d(1.2608695652173914, 1.2608695652173914, 2.2608695652173916), result);
		executeLispExpr("(invmix [2 2 3] xym)", 100, 100, result);
		assertEquals(new Vector3d(2.0277588016728596, 2.0277588016728596, 3.055800856424013), result);

		executeLispExpr("(step xyr [0.7 0.8 99])", 3, 4, result);
		assertEquals(new Vector3d(1, 1, -1), result);
		executeLispExpr("(step [0.7 0.8 99] xyr)", 3, 4, result);
		assertEquals(new Vector3d(-1, -1, 1), result);

		executeLispExpr("(noise [0.7 0.8 99] xyr)", 3, 4, result);
		assertEquals(new Vector3d(-0.04891896714707681, -0.08266644176030277, 0.1465003573902327), result);

		executeLispExpr("(turbulence [0.7 0.8 99] xyr)", 3, 4, result);
		assertEquals(new Vector3d(-0.1351251425583108, 0.09117528639093217, 0.1566956664247885), result);

		executeLispExpr("(waves [0.7 0.8 99] xyr)", 3, 4, result);
		assertEquals(new Vector3d(-0.008955816236990988, -0.005374755236860104, -0.9999454511930503), result);

		//!!		executeLispExpr("(shuffle [0.7 0.8 99] xyr)", 3, 4, result);
		//!!		assertEquals(new Vector3d(3, 0.7, 4), result);

	}

	/**
	 * @param result
	 * @param result1
	 * @param d
	 */
	static public void assertEquals(Vector3d expected, Vector3d actual) {
		assertEquals(null, expected.x, actual.x, ScalarVM.epsilon);
		assertEquals(null, expected.y, actual.y, ScalarVM.epsilon);
		assertEquals(null, expected.z, actual.z, ScalarVM.epsilon);
	}

	public void ignore_testExecute1M() {
		kandid.soup.EntityType entity = (kandid.soup.EntityType) soupFactory.unmarshalFromString(expr1);
		VectorExprGene vectorExprGene = entity.getVectorExpression().getChromosome().getVectorExpression();
		vectorVM.compile(vectorExprGene);
		Vector3d result = new Vector3d();
		for (int i = 0; i < 100; ++i) {
			//!!			assertEquals(-0.3, vectorVM.execute(1.0, 2.0, ScalarVM.epsilon, result), ScalarVM.epsilon);
		}
		long start = System.currentTimeMillis();
		final int rounds = 1000000;
		for (int ix = 0; ix < rounds; ++ix) {
			vectorVM.execute(1.0, 2.0, 1.0, 1.0, ScalarVM.epsilon, result);
		}
		long done = System.currentTimeMillis() - start;
		System.out.println((1000.0 / (double) done) + " mips");
	}

}
