/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.lsys;

import java.awt.AlphaComposite;
import java.util.Iterator;
import java.util.List;

import kandid.calculation.VectorCalculation;
import kandid.colorator.ColorF32;
import kandid.colorator.Colorator;
import kandid.colorator.GrayColorator;
import kandid.colorator.TransparentLookUpTableColorator;
import kandid.soup.ChromosomeType;
import kandid.soup.Lsys0LChromosome;
import kandid.soup.LsysILChromosome;
import kandid.soup.LsysChromosome;
import kandid.soup.LsysD0LChromosome;
import kandid.soup.LsysProductionGene;
import kandid.util.Debug;
import kandid.util.TextureRandomizer;

/**
 * @author thomas jourdan
 *
 */
public class LsysMashine {

  private static final int maxDepthBranchStack = 1000;
  private static final int maxRules = 64;
  private static final int maxSuccessorCharacters = 32;
  private static final int maxCharacters = 127;
  private static final int maxDirections = 72;

  static int maxString;
  static int noTurtleDirections;
  static byte rule[][][];
  static byte lContext[][];
  static byte rContext[][];
  static int sizeofRules[][];
  static int noRules[];
  static boolean ignorCharacter[];

  static final int sys0 = 0;
  public static final int sysDOL = 1;
  public static final int sysOL = 2;
  public static final int sysIL = 3;
  public static final int sys2L = 4;
  static int typeOfLSystem;

  static int depth;
  static double turtleX;
  static double turtleY;
  static int turtleDirection;
  static int startDirection;
  static double COS[];
  static double SIN[];
  static TextureRandomizer ZVarJitter;
  static int jitter;
  static TextureRandomizer ZVarSelect;
  static long seed;

  static byte string[];
  static int length;
  static byte newString[];
  static int newLength;
  static int turtleIndex;

  private static final int maxDepthLeafStack = 100;
  static int leafTOS;
  static final int maxPolyPoints = 100;
  static int[][] xPolyPoints;
  static int[][] yPolyPoints;

  static int branchTOS;
  static double turtleStackX[];
  static double turtleStackY[];
  static int turtleStackDir[];
  static int turtleStackColor[];
  static int turtleStackAlpha[];
  static boolean stackMessageSend;

  static final int stateAbort = -1;
  static final int stateReady = 0;
  static int calculationState = -2;

  static Object sync = new Object();
  static LsysChromosome lsysChromosome;
  static VectorCalculation vectorCalculation;
  static Colorator colorator;
  static boolean transparenceEnabled;

  private static int baseIndex;
  private static int numberOfDiscretColors;
  private static ColorF32 realizedColor;
  private static int colorIndex;
  private static int realizedAlphaIndex;
  private static int alphaIndex;

  /**
   * expands and draws the Lindenmayer system.
   * Due to memory requierements only one Lindenmayer system can be executet at one time.
   * Java synchronized is used to serialize the execute calls.
   * @param chromosome
   * @param setColorator
   * @param setVectorCalculation
   */
  public static void execute(ChromosomeType chromosome, Colorator setColorator, VectorCalculation setVectorCalculation) {
    synchronized (sync) {
      try {
        lsysChromosome = (LsysChromosome) chromosome;
        vectorCalculation = setVectorCalculation;
        colorator = setColorator;
        if (Debug.enabled)
          System.out.println(vectorCalculation + ", priority: " + Thread.currentThread().getPriority());
        transparenceEnabled = colorator instanceof TransparentLookUpTableColorator || colorator instanceof GrayColorator;
        numberOfDiscretColors = colorator.getNumberOfDiscretColors();
        if (Debug.enabled)
          assert numberOfDiscretColors > 1 : "numberOfDiscretColors " + numberOfDiscretColors;

        if(!vectorCalculation.isAborted()) {
          allocate1();
          initCalculation();

          allocate2();
          do {
            calculate();
            if (vectorCalculation.isAborted()) {
              calculationState = stateAbort;
            }
          }
          while (calculationState > stateReady && !vectorCalculation.isAborted());
        }
      }
      catch (RuntimeException exc) {
        Debug.stackTrace(exc);
      }
      finally {
        free();
      }
    }
  }

  /**
   * allocate static buffers
   */
  private static void allocate1() {
    rule = new byte[maxCharacters][maxRules][maxSuccessorCharacters];
    lContext = new byte[maxCharacters][maxRules];
    rContext = new byte[maxCharacters][maxRules];
    sizeofRules = new int[maxCharacters][maxRules];
    noRules = new int[maxCharacters];
    ignorCharacter = new boolean[maxCharacters];
    COS = new double[maxDirections];
    SIN = new double[maxDirections];
  }

  /**
   * allocate static buffers
   */
  private static void allocate2() {
    maxString = 32 * 1024 * 1024;
    string = new byte[maxString];
    assert string[0] == 0;
    assert string[maxString - 1] == 0;
    newString = new byte[maxString];
    assert newString[0] == 0;
    assert newString[maxString - 1] == 0;

    xPolyPoints = new int[maxDepthLeafStack][maxPolyPoints];
    yPolyPoints = new int[maxDepthLeafStack][maxPolyPoints];

    turtleStackX = new double[maxDepthBranchStack];
    turtleStackY = new double[maxDepthBranchStack];
    turtleStackDir = new int[maxDepthBranchStack];
    turtleStackColor = new int[maxDepthBranchStack];
    turtleStackAlpha = new int[maxDepthBranchStack];
  }

  /**
   * free static buffers
   */
  private static void free() {
    rule = null;
    lContext = null;
    rContext = null;
    sizeofRules = null;
    noRules = null;
    ignorCharacter = null;
    COS = null;
    SIN = null;

    string = null;
    newString = null;

    xPolyPoints = null;
    yPolyPoints = null;

    turtleStackX = null;
    turtleStackY = null;
    turtleStackDir = null;
    turtleStackColor = null;
    turtleStackAlpha = null;
  }

  /**
   * entry point for JUnit testing
   * @param chromosome
   */
  static void testInitCalculation(ChromosomeType chromosome) {
    lsysChromosome = (LsysChromosome) chromosome;
    allocate1();
    initCalculation();
    allocate2();
  }

  /**
   * Convert from char to byte.
   * @param ch
   * @return
   */
  private static byte mapChar(char ch) {
    assert ch > 32;
    assert ch < 127;
    return (byte) ch;
  }

  /**
   * Initialize the Lindenmayer system expander / processor.
   */
  private static void initCalculation() {
    try {
      String ignor = "";
      if (lsysChromosome instanceof LsysD0LChromosome) {
        jitter = 0;
        seed = 1512;
        depth = ((LsysD0LChromosome) lsysChromosome).getDepth().getValue();
        ignor = "";
      }
      else if (lsysChromosome instanceof Lsys0LChromosome) {
        jitter = ((Lsys0LChromosome) lsysChromosome).getJitter().getValue();
        seed = ((Lsys0LChromosome) lsysChromosome).getNoiseSeed().getValue();
        depth = ((Lsys0LChromosome) lsysChromosome).getDepth().getValue();
        ignor = "";
      }
      else if (lsysChromosome instanceof LsysILChromosome) {
        jitter = ((LsysILChromosome) lsysChromosome).getJitter().getValue();
        seed = ((LsysILChromosome) lsysChromosome).getNoiseSeed().getValue();
        depth = ((LsysILChromosome) lsysChromosome).getDepth().getValue();
        ignor = ((LsysILChromosome) lsysChromosome).getIgnore().getValue();
      }
      baseIndex = lsysChromosome.getBaseIndex().getValue();

      ZVarJitter = new TextureRandomizer(seed);
      ZVarSelect = new TextureRandomizer(seed);

      noTurtleDirections = 2 * lsysChromosome.getDirections().getValue();
      startDirection = noTurtleDirections / 2;
      if (noTurtleDirections > maxDirections || noTurtleDirections < 1) {
        if (Debug.enabled)
        System.out.println("invalid number of directions");
        calculationState = stateAbort;
        return;
      }
      for (int direction = 0; direction < noTurtleDirections; direction++) {
        COS[direction] = Math.cos((double) direction * 2 * Math.PI / (double) noTurtleDirections);
        SIN[direction] = Math.sin((double) direction * 2 * Math.PI / (double) noTurtleDirections);
      }

      initIgnore(ignor);

      // get production rules
      typeOfLSystem = sys0;
      int index = maxCharacters;
      while (index-- > 0)
        noRules[index] = 0;
      List<LsysProductionGene> productions = lsysChromosome.getProduction();
      for (Iterator<LsysProductionGene> ProductionIiter = productions.iterator(); ProductionIiter.hasNext();) {
        LsysProductionGene production = ProductionIiter.next();
        String predecessor = production.getPredecessor().getValue();
        String successor = production.getSuccessor().getValue();
        byte pred = 0;
        if (predecessor.length() == 1) {
          pred = mapChar(predecessor.charAt(0));
          // get successor
          sizeofRules[pred][noRules[pred]] = index = successor.length();
          while (index-- > 0) {
            rule[pred][noRules[pred]][index] = mapChar(successor.charAt(index));
          }
        }
        else {
          System.out.println("invalid rule");
        }

        // D0L and 0L are context free
        lContext[pred][noRules[pred]] = 0;
        rContext[pred][noRules[pred]] = 0;
        // get context for IL systems
        if (lsysChromosome instanceof LsysILChromosome) {
          String leftContext = production.getLeftContext().getValue();
          String rightContext = production.getRightContext().getValue();
          if (leftContext.length() > 0 && rightContext.length() > 0) {
            typeOfLSystem = sys2L;
            lContext[pred][noRules[pred]] = mapChar(leftContext.charAt(0));
            rContext[pred][noRules[pred]] = mapChar(rightContext.charAt(0));
          }
          else if (leftContext.length() > 0 && rightContext.length() == 0) {
            typeOfLSystem = sysIL;
            lContext[pred][noRules[pred]] = mapChar(leftContext.charAt(0));
          }
          else if (leftContext.length() == 0 && rightContext.length() > 0) {
            typeOfLSystem = sysIL;
            rContext[pred][noRules[pred]] = mapChar(rightContext.charAt(0));
          }
        }

        ++noRules[pred];
      }

      // check if Lindenmayer system is not deterministic
      if (typeOfLSystem < sysOL) {
        int rx = maxCharacters;
        while (rx-- != 0) {
          if (noRules[rx] > 1) {
            typeOfLSystem = sysOL;
            break;
          }
        }
      }
      if (typeOfLSystem < sysDOL)
        typeOfLSystem = sysDOL;

      calculationState = 1;
    }
    catch (Exception exc) {
      Debug.stackTrace(exc);
      calculationState = stateAbort;
    }
  }

  /**
   * Initialize the ignor character set for kontext sensitive Lindenmayer systems.
   * @param ignor
   */
  static void initIgnore(String ignor) {
    int ix = maxCharacters;
    while (ix-- > 0)
      ignorCharacter[ix] = false;
    ix = ignor.length();
    while (ix-- > 0)
      ignorCharacter[ignor.charAt(ix)] = true;
    //    ignorZeichen['['] = true;
    //    ignorZeichen[']'] = true;
  }

  /**
   * expand and draw.
   */
  private static void calculate() {

    switch (calculationState) {
      case 1 :
        stackMessageSend = false;
        turtleX = 0.0;
        turtleY = 0.0;
        turtleDirection = startDirection;

        expand(depth);

        ++calculationState;
        return;

      case 2 :
        ZVarJitter.setSaat(seed);
        postProcess();
        newString = null; // free buffer

        ++calculationState;
        return;

      case 3 :
        filter();

        ++calculationState;
        return;

      case 4 :
      {
        ColorF32 cout = new ColorF32();
        colorator.getDiscretColor((baseIndex + 1) % numberOfDiscretColors, cout);
        vectorCalculation.g2d.setColor(new java.awt.Color(cout.r, cout.r, cout.r));
        vectorCalculation.g2d.fillRect(0, 0, vectorCalculation.canvasSize.width, vectorCalculation.canvasSize.height);
      }

        ++calculationState;
        return;

      case 5 :
        ZVarJitter.setSaat(seed);
        turtleInterpretation();

        ++calculationState;
        return;

      default :
        string = null; // free buffer
        calculationState = stateReady;
    }
  }

  /**
   * Expand Lindenmayer System
   * @param maxDepth
   */
  static void expand(int maxDepth) {
    int actDepth;
    int sizeofRule;
    byte character;
    int selection;
    int actPos;
    int hits[] = new int[maxRules];

    // get axiom
    String ax = lsysChromosome.getAxiom().getValue();
    length = ax.length();
    int Index = length;
    while (Index-- > 0) {
      string[Index] = mapChar(ax.charAt(Index));
    }

    for (actDepth = 1; actDepth <= maxDepth; actDepth++) {
      if(vectorCalculation != null && vectorCalculation.isAborted())
        break;
      newLength = 0;

      switch (typeOfLSystem) {
        case sysDOL :
          for (actPos = 0; actPos < length; actPos++) {
            character = string[actPos];
            if (noRules[character] != 0) {
              sizeofRule = sizeofRules[character][0];
              if (newLength + sizeofRule >= maxString) {
                System.out.println("production overflow: depth " + actDepth + ", size " + newLength);
                return;
              }
              System.arraycopy(rule[character][0], 0, newString, newLength, sizeofRule);
              newLength += sizeofRule;
            }
            else {
              if (length + newLength + 1 >= maxString) {
                System.out.println("production overflow: depth " + actDepth + ", size " + newLength);
                return;
              }
              newString[newLength] = character;
              ++newLength;
            }
          }
          break;

        case sysOL :
          for (actPos = 0; actPos < length; actPos++) {
            character = string[actPos];
            if (noRules[character] != 0) {
              selection = ZVarSelect.wuerfeln() % noRules[character];
              sizeofRule = sizeofRules[character][selection];
              if (newLength + sizeofRule >= maxString) {
                System.out.println("production overflow: depth " + actDepth + ", size " + newLength);
                return;
              }
              System.arraycopy(rule[character][selection], 0, newString, newLength, sizeofRule);
              newLength += sizeofRule;
            }
            else {
              if (length + newLength + 1 >= maxString) {
                System.out.println("production overflow: depth " + actDepth + ", size " + newLength);
                return;
              }
              newString[newLength] = character;
              ++newLength;
            }
          }
          break;

        case sysIL :
        case sys2L :
          for (actPos = 0; actPos < length; actPos++) {
            character = string[actPos];

            int AnzTreffer = 0;
            for (int rx = 0; rx < noRules[character]; ++rx) {
              boolean contextFound = true;
              if (lContext[character][rx] != 0)
                contextFound = testLeftContext(actPos, lContext[character][rx]);
              if (contextFound && rContext[character][rx] != 0)
                contextFound = testRightContext(actPos, rContext[character][rx]);
              if (contextFound) {
                hits[AnzTreffer] = rx;
                ++AnzTreffer;
              }
            }
            if (AnzTreffer == 0) {
              for (int rx = 0; rx < noRules[character]; ++rx) {
                if (rContext[character][rx] == '\0' && lContext[character][rx] == '\0') {
                  hits[AnzTreffer] = rx;
                  ++AnzTreffer;
                }
              }
            }
            if (AnzTreffer != 0) {
              selection = hits[ZVarSelect.wuerfeln() % AnzTreffer];
              sizeofRule = sizeofRules[character][selection];
              if (length + newLength + sizeofRule >= maxString) {
                System.out.println("production overflow: depth " + actDepth + ", size " + newLength);
                return;
              }
              System.arraycopy(rule[character][selection], 0, newString, newLength, sizeofRule);
              newLength += sizeofRule;
            }
            else {
              if (length + newLength + 1 >= maxString) {
                System.out.println("production overflow: depth " + actDepth + ", size " + newLength);
                return;
              }
              newString[newLength] = character;
              ++newLength;
            }
          }
          break;

      }

      System.arraycopy(newString, 0, string, 0, newLength);
      length = newLength;
      if (Debug.enabled)
        System.out.println("depth " + actDepth + ", size " + length);
    }
  }

  /**
   * Determines if testCharacter can be found "left" of an given position.
   * @param actPos position
   * @param testCharacter Occurence of this charater will be tested
   * @return true if the left context condition is full filled 
   */
  static boolean testLeftContext(int actPos, byte testCharacter) {
    int branchCount = 0;
    byte character = 0;
    --actPos;
    while (actPos >= 0 && ignorCharacter[string[actPos]])
      --actPos;
    if (actPos >= 0) {
      character = string[actPos];
      while (branchCount > 0 || character == '[' || character == ']') {
        if (character == '[') {
          if (branchCount > 0)
            --branchCount;
        }
        else if (character == ']')
          ++branchCount;
        --actPos;
        while (actPos >= 0 && ignorCharacter[string[actPos]])
          --actPos;
        if (actPos >= 0)
          character = string[actPos];
        else
          return false;
      }
      return character == testCharacter;
    }
    else
      return false;
  }

  /**
   * Determines if testCharacter can be found "right" of an given position.
   * @param actPos position
   * @param testCharacter Occurence of this charater will be tested
   * @return true if the left context condition is full filled 
   */
  static boolean testRightContext(int actPos, byte testCharacter) {
    int branchCount = 0;
    byte character;
    ++actPos;
    while (actPos < length && ignorCharacter[string[actPos]])
      ++actPos;
    if (actPos < length) {
      character = string[actPos];
      while (branchCount != 0 || character == '[' || character == ']') {
        if (character == '[')
          ++branchCount;
        else if (character == ']')
          --branchCount;
        if (branchCount < 0)
          return false;
        ++actPos;
        while (actPos < length && ignorCharacter[string[actPos]])
          ++actPos;
        if (actPos < length)
          character = string[actPos];
        else
          return false;
      }
      return character == testCharacter;
    }
    else
      return false;
  }

  /**
   * Draws the Lindenmayer system.
   */
  private static void turtleInterpretation() {

    int character = string[0];
    turtleIndex = 0;
    int oldX, oldY;
    double turtlePosOldX;
    double turtlePosOldY;
    int nPoints[] = new int[maxDepthLeafStack];
    ColorF32 lut[] = new ColorF32[numberOfDiscretColors];
    for (int lx = 0; lx < lut.length; lx++) {
      lut[lx] = new ColorF32();
      colorator.getDiscretColor(lx, lut[lx]);
    }

    vectorCalculation.initTransformation(0.05);
    branchTOS = 0;
    leafTOS = 0;
    colorIndex = 0;
    realizedColor = null;
    alphaIndex = 0;
    realizedAlphaIndex = Integer.MAX_VALUE;
    turtleDirection = startDirection;
    turtleX = turtleY = 0.0;
    turtlePosOldX = turtlePosOldY = 0.0;
    nPoints[leafTOS] = 0;

    turtleIndex = 0;
    while (turtleIndex < length) {
      if ((turtleIndex % 10000) == 0) {
        if (vectorCalculation.isAborted())
          break;
        vectorCalculation.getViewComponent().repaint();
      }
      nextState(character);
      if (character == 'F') {
        vectorCalculation.transform(turtleX, turtleY);
        int newX = vectorCalculation.xp;
        int newY = vectorCalculation.yp;
        if (leafTOS > 0 && leafTOS < maxDepthLeafStack && nPoints[leafTOS] > 0) {
          if (nPoints[leafTOS] < maxPolyPoints) {
            xPolyPoints[leafTOS][nPoints[leafTOS]] = newX;
            yPolyPoints[leafTOS][nPoints[leafTOS]] = newY;
            ++nPoints[leafTOS];
          }
        }
        else {
          vectorCalculation.transform(turtlePosOldX, turtlePosOldY);
          oldX = vectorCalculation.xp;
          oldY = vectorCalculation.yp;
          setColor(lut);
          if (transparenceEnabled)
            setAlpha();
          vectorCalculation.g2d.drawLine(oldX, oldY, newX, newY);
        }
        turtlePosOldX = turtleX;
        turtlePosOldY = turtleY;
      }
      else if (character == 'f' || character == ']') {
        turtlePosOldX = turtleX;
        turtlePosOldY = turtleY;
      }
      else if (character == '{') {
        ++leafTOS;
        ++colorIndex;
        vectorCalculation.transform(turtlePosOldX, turtlePosOldY);
        oldX = vectorCalculation.xp;
        oldY = vectorCalculation.yp;
        if (leafTOS < maxDepthLeafStack) {
          xPolyPoints[leafTOS][0] = vectorCalculation.xp;
          yPolyPoints[leafTOS][0] = vectorCalculation.yp;
          nPoints[leafTOS] = 1;
        }
      }
      else if (character == '}') {
        if (leafTOS > 0 && leafTOS < maxDepthLeafStack) {
          vectorCalculation.transform(turtleX, turtleY);
          if (nPoints[leafTOS] < maxPolyPoints) {
            xPolyPoints[leafTOS][nPoints[leafTOS]] = vectorCalculation.xp;
            yPolyPoints[leafTOS][nPoints[leafTOS]] = vectorCalculation.xp;
            ++nPoints[leafTOS];
          }
          if (nPoints[leafTOS] > 2) {
            setColor(lut);
            if (transparenceEnabled)
              setAlpha();
            vectorCalculation.g2d.fillPolygon(xPolyPoints[leafTOS], yPolyPoints[leafTOS], nPoints[leafTOS]);
            nPoints[leafTOS] = 0;
          }
        }
        --leafTOS;
        --colorIndex;
      }
      else if (character == '.') {
        if (leafTOS > 0 && leafTOS < maxDepthLeafStack && nPoints[leafTOS] > 0) {
          if (nPoints[leafTOS] < maxPolyPoints) {
            vectorCalculation.transform(turtleX, turtleY);
            if (leafTOS < maxDepthLeafStack) {
              xPolyPoints[leafTOS][nPoints[leafTOS]] = vectorCalculation.xp;
              yPolyPoints[leafTOS][nPoints[leafTOS]] = vectorCalculation.xp;
              ++nPoints[leafTOS];
            }
          }
        }
      }
      ++turtleIndex;
      character = string[turtleIndex];
    }
  }

  /**
   * Sets color from color table;
   * @param lut
   */
  private static void setColor(ColorF32[] lut) {
    int n1 = baseIndex + colorIndex;
    int n2 = numberOfDiscretColors;
    int n = (int) (n1 / n2);
    n1 -= n * n2;
    if (n1 < 0)
      n1 += n2;
    ColorF32 newColor = lut[n1];
    if (realizedColor == null || !realizedColor.equals(newColor)) {
      vectorCalculation.g2d.setColor(new java.awt.Color(newColor.r, newColor.g, newColor.b));
      realizedColor = newColor;
    }
  }

  /**
   * Sets alpha composite.
   */
  private static void setAlpha() {
    if (alphaIndex != realizedAlphaIndex) {
      float newAlpha = toAlpha(alphaIndex);
      AlphaComposite ac = AlphaComposite.getInstance(AlphaComposite.SRC_OVER, newAlpha);
      vectorCalculation.g2d.setComposite(ac);
      realizedAlphaIndex = alphaIndex;
    }
  }

  /**
   * Non linear scale of great numbers to a colors alpha value.
   * @param rawValue
   * @return a number between 0.0 and 1.0
   */
  public static float toAlpha(int rawValue) {
    float alpha = (float) ((0.5 * Math.PI + Math.atan((double) rawValue / 10.0)) / Math.PI);
    if (alpha < 0.01)
      return 0.0f;
    else if (alpha > 0.99)
      return 1.0f;
    return alpha;
  }


  /**
   * Removes any non turtle characters from the string and calculates the bounds.
   */
  private static void postProcess() {
    byte character = string[0];
    int destination = 0;
    int source = 0;
    int newLength = 0;

    while (source < length) {
      if (character == 'F'
        || character == 'f'
        || character == '['
        || character == ']'
        || character == '+'
        || character == '-'
        || character == '|'
        || character == '{'
        || character == '}'
        || character == '.'
        || character == 39
        || character == ','
        || character == ':'
        || character == ';') {
        string[destination++] = character;
        ++newLength;
        nextState(character);
        if (character == 'F' || character == 'f') {
          vectorCalculation.xmax = Math.max(turtleX, vectorCalculation.xmax);
          vectorCalculation.ymax = Math.max(turtleY, vectorCalculation.ymax);
          vectorCalculation.xmin = Math.min(turtleX, vectorCalculation.xmin);
          vectorCalculation.ymin = Math.min(turtleY, vectorCalculation.ymin);
        }
      }
      character = string[++source];
    }
    if (Debug.enabled)
      System.out.println("postProcess length=" + length + ", new length=" + newLength);
    length = newLength;
  }

  /**
   * Remove character sequences making no sense for the turtle interpretation.
   * 
   */
  private static void filter() {
    byte character;
    byte character2;
    int destination;
    int source;
    int newLength = 0;
    int oldLength = length;

    do {
      character = string[0];
      character2 = string[1];
      destination = 0;
      source = 0;
      oldLength = newLength;
      newLength = 0;
      while (source < length) {
        if (character == '-' && character2 == ']') {
          ++source;
          string[destination++] = string[source++];
          ++newLength;
        }
        else if (character == '+' && character2 == ']') {
          ++source;
          string[destination++] = string[source++];
          ++newLength;
        }
        else if (character == '[' && character2 == ']') {
          source += 2;
        }
        else if (character == '{' && character2 == '}') {
          source += 2;
        }
        else {
          string[destination++] = string[source++];
          ++newLength;
        }
        character = string[source];
        character2 = string[source + 1];
      }
      if (Debug.enabled)
        System.out.println("Filter2  Laenge " + newLength);
    }
    while (newLength != 0 && newLength < oldLength && !vectorCalculation.isAborted());

    length = newLength;
    if (Debug.enabled)
      System.out.println(" Laenge " + length);
  }

  /**
   * Next turtle state.
   * @param character
   */
  private static void nextState(int character) {

    switch (character) {
      case 'F' :
      case 'f' :
        if (jitter != 0) {
          turtleX += jitter / 100.0 * ZVarJitter.GaussVerteilt()
            + (100 - jitter) / 100.0 * SIN[turtleDirection];
          turtleY += jitter / 100.0 * ZVarJitter.GaussVerteilt()
            + (100 - jitter) / 100.0 * COS[turtleDirection];
        }
        else {
          turtleX += SIN[turtleDirection];
          turtleY += COS[turtleDirection];
        }
        break;
      case '+' :
        ++turtleDirection;
        if (turtleDirection >= noTurtleDirections)
          turtleDirection = 0;
        break;
      case '-' :
        --turtleDirection;
        if (turtleDirection < 0)
          turtleDirection = noTurtleDirections - 1;
        break;
      case '|' :
        turtleDirection += noTurtleDirections / 2;
        if (turtleDirection >= noTurtleDirections)
          turtleDirection -= noTurtleDirections;
        break;
      case '[' :
        if (branchTOS >= maxDepthBranchStack) {
          if (!stackMessageSend) {
            System.out.println("stack overflow");
            stackMessageSend = true;
          }
        }
        else {
          turtleStackX[branchTOS] = (float) turtleX;
          turtleStackY[branchTOS] = (float) turtleY;
          turtleStackDir[branchTOS] = turtleDirection;
          turtleStackColor[branchTOS] = colorIndex;
          turtleStackAlpha[branchTOS] = alphaIndex;
          ++branchTOS;
        }
        break;
      case ']' :
        if (branchTOS == 0) {
          if (!stackMessageSend) {
            System.out.println("stack underflow");
            stackMessageSend = true;
          }
        }
        else
          --branchTOS;
        turtleX = turtleStackX[branchTOS];
        turtleY = turtleStackY[branchTOS];
        turtleDirection = turtleStackDir[branchTOS];
        colorIndex = turtleStackColor[branchTOS];
        alphaIndex = turtleStackAlpha[branchTOS];
        break;
      case 39 :
        if (leafTOS == 0)
          ++colorIndex;
        break;
      case ',' :
        if (leafTOS == 0)
          --colorIndex;
        break;
      case ':' :
        if (leafTOS == 0)
          ++alphaIndex;
        break;
      case ';' :
        if (leafTOS == 0)
          --alphaIndex;
        break;
    }
  }

  /**
   * @return classification of the Lsystem: D0L, 0L, IL, ....
   */
  public static int getTypeOfLSystem() {
    return typeOfLSystem;
  }

}
