/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.lca;

import kandid.colorator.ColorF32;
import kandid.colorator.Colorator;
import kandid.soup.ChromosomeType;
import kandid.soup.ColorGene;
import kandid.soup.LcaStateGene;
import kandid.soup.TotalLcaChromosome;
import kandid.util.Debug;

/**
 *
 * @author thomas jourdan
 */
public class TotalLcaCalculation extends LcaCalculation implements Cloneable {

  public void setChromosome(ChromosomeType chromosome, Colorator colorator) {
    super.setChromosome(chromosome, colorator);
    TotalLcaChromosome lcaChromosome = (TotalLcaChromosome) chromosome;
    maxStates = lcaChromosome.getMaxStates().getValue();
    size = width * width;
    totalDisplayRows = width;
    actDisplayRow = 0;
    neighbors = lcaChromosome.getLeftNeighbors().getValue() + 1 + lcaChromosome.getRightNeighbors().getValue();
    // initialize cell array
    state = new int[width];
    setRandomFirstState();
    nextState = new int[width];
    // initialize rules
    java.util.List<LcaStateGene> ruleList = lcaChromosome.getRule();
    rule = new int[ruleList.size()];
    for (int px = 0; px < rule.length; ++px) {
      rule[px] = ruleList.get(px).getValue();
    }
    // initialize palette
    intPalette = new ColorF32[lcaChromosome.getColor().size()];
    java.util.List<ColorGene> colorList = lcaChromosome.getColor();
    for (int px = 0; px < intPalette.length; ++px) {
      intPalette[px] = new ColorF32();
      colorator.getColor(colorList.get(px), intPalette[px]);
    }
    if (Debug.enabled) assert invariant();
  }

  public static int numberofRules(int maxStates, int leftNeighbors, int rightNeighbors) {
    return maxStates * (leftNeighbors + 1 + rightNeighbors);
  }

  /* For documentation please see super class LCACalculation. */
  protected int getRuleNumber(int index, int maxStates) {
    // calculate sum of all neighbors
    int ruleNumber = 0;
    for (int nx = 0; nx < neighbors; nx++) {
      if (Debug.enabled) assert state[(index + nx) % state.length] < maxStates : "state out of range";
      ruleNumber += state[(index + nx) % state.length];
    }
    return ruleNumber;
  }

  protected boolean invariant() {
    if (Debug.enabled) {
      if (neighbors < 1) {
        if (Debug.enabled) System.out.println("leftNeighbors " + neighbors);
        return false;
      }
      if (maxStates < 0) {
        if (Debug.enabled) System.out.println("maxStates " + maxStates);
        return false;
      }
      if (intPalette.length != maxStates) {
        if (Debug.enabled) System.out.println("palette.length " + intPalette.length);
        return false;
      }
      for (int rx = 0; rx < rule.length; ++rx) {
        if (rule[rx] < 0 || rule[rx] >= maxStates) {
          if (Debug.enabled) System.out.println("rule[" + rx + "] " + rule[rx]);
          return false;
        }
      }
      for (int fx = 0; fx < state.length; ++fx) {
        if (state[fx] < 0 || state[fx] >= maxStates) {
          if (Debug.enabled) System.out.println("state[" + fx + "] " + state[fx] + ", " + maxStates);
          return false;
        }
      }
    }
    return true;
  }

  public boolean hasWhiteBackground() {
    return false;
  }
}
