/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.lca;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.util.Random;

import kandid.calculation.*;
import kandid.colorator.*;
import kandid.soup.ChromosomeType;
import kandid.soup.LcaChromosome;
import kandid.util.Debug;

public abstract class LcaCalculation extends PixelCalculation implements Cloneable {
  protected int    totalDisplayRows;
  protected int    actDisplayRow;
  protected int    rule[];
  protected int    state[];
  protected int    nextState[];
  protected int    neighbors;
  protected ColorF32    intPalette[];
  protected int    width;
  protected int    maxStates;
  protected String initialFillMode;
  protected long   initialStateSeed;
  
  public void setChromosome(ChromosomeType chromosome, Colorator colorator) {
    super.setChromosome(chromosome, colorator);
    LcaChromosome lcaChromosome = (LcaChromosome)chromosome;
    width = lcaChromosome.getWidth().getValue();
    initialFillMode = lcaChromosome.getLcaInitialState().getFillMode().value();
    initialStateSeed = lcaChromosome.getLcaInitialState().getStateSeed().getValue();
  }

  /**
   * return the preferred size for this calculation.
   *
   * @return Dimension(0,0), if there is no preferred size..
   */
  public Dimension getPreferredSize() {
    return new Dimension(width,width);
  }

  public void activateCanvas(Component viewComponent, Dimension canvasSize, boolean zoomMode) {
    super.activateCanvas(viewComponent, canvasSize, zoomMode);
    bufAwtImg = new BufferedImage(width, width, BufferedImage.TYPE_INT_RGB);
    if(!zoomMode) {
      double xScale = (double)canvasSize.width / (double)width;
      double yScale = (double)canvasSize.height / (double)width;
      affineTransform.setToScale(xScale, yScale);
    }
    if (Debug.enabled) System.out.println("affineTransform " + affineTransform);
    
    // paint background
    Graphics gr = bufAwtImg.getGraphics();
    gr.setColor(Color.black);
    gr.fillRect(0, 0, canvasSize.width, canvasSize.height);
    
    // paint state
    paint(intPalette);
  }

  /* (non-Javadoc)
   * @see kandid.calculation.Calculation#calculate(boolean, java.lang.String)
   */
  public void calculate(Deviation dev, boolean paintOnScreen, String doExport) {
    // calculate next part of image
    int maxPart = paintOnScreen ? 200 : totalDisplayRows;
    for (int px = 0; px < maxPart && !getReady(); ++px) {
      // calculate next state
      for (int sx = 0; sx < state.length; sx++) {
        int ruleNumber = getRuleNumber(sx, maxStates);
        try {
          nextState[sx] = rule[ruleNumber];
        } catch (Exception exc) {
          nextState[sx] = 0;
        }
      }
  
      // copy temporary cell array to actual cell array
      System.arraycopy(nextState, 0, state, 0, state.length);
      // paint state
      paint(intPalette);
      // ready for this turn
      ++actDisplayRow;
    }
  }

   /**
   * Method getRuleNumber
   * @param index
   * @return.
   */
  protected abstract int getRuleNumber(int index, int maxStates);
  
  /**
   * Method paint.
   */
  private void paint(ColorF32 palette[]) {
    for (int sx = 0; sx < state.length; sx++) {
      setPixel(sx, actDisplayRow, palette[state[sx] % palette.length]);
    }
  }

  /* For documentation please see super class Calculation. */
  public int getPercent() {
    return 100 - (100 * (totalDisplayRows - actDisplayRow)) / totalDisplayRows;
  }

  /* For documentation please see super class Calculation. */
  public boolean getReady() {
    return aborted || actDisplayRow >= totalDisplayRows;
  }

  /**
   * rgb values used in this calculation.
   *
   * @return palette array
   */
  public ColorF32[] getPalette() {
    return intPalette;
  }

  /**
   * Sets the initial state of all cells to a random value.
   */
  protected void setRandomFirstState() {
    Random rand = new Random();
    rand.setSeed(initialStateSeed);
    if(initialFillMode.equals("uniform random")) {
      for (int fx = 0; fx < state.length; fx++) {
        state[fx] = rand.nextInt(maxStates);
      }
    }
    else if(initialFillMode.equals("random ramp")) {
      for (int fx = 0; fx < state.length; fx++) {
        state[fx] = rand.nextInt(maxStates) * (int)Math.round((double)(state.length - fx) / (double)(state.length));
        state[fx] %= maxStates;
      }
    }
    else if(initialFillMode.equals("single point")) {
      clearFirstState();
      state[width / 2] = rand.nextInt(maxStates);
    }
    else if(initialFillMode.equals("two points")) {
      clearFirstState();
      state[width / 2]     = rand.nextInt(maxStates);
      state[width / 2 + 1] = rand.nextInt(maxStates);
    }
    else if(initialFillMode.equals("three points")) {
      clearFirstState();
      state[width / 2 - 1] = rand.nextInt(maxStates);
      state[width / 2]     = rand.nextInt(maxStates);
      state[width / 2 + 1] = rand.nextInt(maxStates);
    }
    else if(initialFillMode.equals("two points with single gap")) {
      clearFirstState();
      state[width / 2 - 1] = rand.nextInt(maxStates);
      state[width / 2 + 1] = rand.nextInt(maxStates);
    }
    else {
      if (Debug.enabled) assert false: "wrong initial mode: " + initialFillMode;
    }
  }

  /**
   * Sets the initial state of all cells to a random value.
   */
  protected void clearFirstState() {
    for (int fx = 0; fx < state.length; fx++) {
      state[fx] = 0;
    }
  }
  
}
