/*
 * Copyright (C) 2002 - 2026 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.domainwarping;

import java.awt.Component;
import java.awt.Dimension;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import kandid.calculation.Deviation;
import kandid.calculation.RefinementCalculation;
import kandid.colorator.ColorF32;
import kandid.noise.AbstractNoise;
import kandid.noise.CellNoise;
import kandid.noise.ChladniNoise;
import kandid.noise.ImageNoise;
import kandid.noise.PerlinNoise;
import kandid.soup.ChromosomeType;
import kandid.soup.DomainWarpingChromosome;
import kandid.soup.WarpingLayerGene;
import kandid.soup.WarpingTurbulenceGene;
import kandid.util.Debug;

public abstract class DomainWarpingCalculation extends RefinementCalculation implements Cloneable {
  
  abstract protected String getConstraint();
  private String constraint;  
  protected static final String FREQUENCY_KEY = "frequency=";

  int frequency[] = {
      1, // PerlinNoise
      1, // CellNoise
      1, // ChladniNoise
      0, // TODO ImageNoise
      };

  class WarpingTurbulence {
    double timeCoord;
    double octaves;
    double ampscale;
    double freqscale;
    double scaleOut;
    double scaleXin;
    double scaleYin;
    double source1;
    double source2;
    double source3;
    double source4;
    long seed1;
    long seed2;
    long seed3;
    long seed4;
    NoiseComposer composer;
   
    public WarpingTurbulence(NoiseComposer composer) {
      this.composer = composer;
    }

    void fill(WarpingTurbulenceGene wpg) {
      timeCoord = wpg.getTimeCoord().getValue();
      octaves = wpg.getOctaves().getValue();
      ampscale = wpg.getAmpscale().getValue();
      freqscale = wpg.getFreqscale().getValue();
      scaleOut = wpg.getScaleOut().getValue();
      scaleXin = wpg.getScaleXin().getValue();
      scaleYin = wpg.getScaleYin().getValue();
      source1 = wpg.getSource1().getValue();
      source2 = wpg.getSource2().getValue();
      source3 = wpg.getSource3().getValue();
      source4 = wpg.getSource4().getValue();
      seed1 = wpg.getSeed1().getValue();
      seed2 = wpg.getSeed2().getValue();
      seed3 = wpg.getSeed3().getValue();
      seed4 = wpg.getSeed4().getValue();
      composer.setWeight(source1, source2, source3, source4);
      composer.setSeed(seed1, seed2, seed3, seed4);
    }
  }

  class WarpingLayer {
    NoiseComposer compose1 = new NoiseComposer();
    WarpingTurbulence term1 = new WarpingTurbulence(compose1);
    NoiseComposer compose2 = new NoiseComposer();
    WarpingTurbulence term2 = new WarpingTurbulence(compose2);
    int n = 0;
   
    void fill(WarpingLayerGene wlg) {
      List<WarpingTurbulenceGene> tl = wlg.getTerm();
      switch(tl.size()) {
      case 1:
        term1.fill(tl.get(0));
        n = 1;
        break;
      case 2:
        term1.fill(tl.get(0));
        term2.fill(tl.get(1));
        n = 2;
        break;
      }
    }
  }

  private double mainscale;
  private double gradientgain;

  private WarpingLayer lay0 = new WarpingLayer();
  private WarpingLayer lay1ll = new WarpingLayer();
  private WarpingLayer lay1rl = new WarpingLayer();
  private WarpingLayer lay1lr = new WarpingLayer();
  private WarpingLayer lay1rr = new WarpingLayer();

  class NoiseComposer {
    private double source1 = 1.0 / 4.0;
    private double source2 = 1.0 / 4.0;
    private double source3 = 1.0 / 4.0;
    private double source4 = 1.0 / 4.0;
    private AbstractNoise noise1;
    private AbstractNoise noise2;
    private AbstractNoise noise3;
    private AbstractNoise noise4;

    void setWeight(double w1, double w2, double w3, double w4) {
      double sum = w1 + w2 + w3 + w4;
      w1 = w1 / sum;
      w2 = w2 / sum;
      w3 = w3 / sum;
      w4 = w4 / sum;
      
      w1 = w1 * w1;
      w2 = w2 * w2;
      w3 = w3 * w3;
      w4 = w4 * w4;
      
      sum = w1 + w2 + w3 + w4;
      source1 = w1 / sum;
      source2 = w2 / sum;
      source3 = w3 / sum;
      source4 = w4 / sum;
    }

    private AbstractNoise factory(long seed, int n) {
      int sum = 0;
      for(int i=0; i<frequency.length; ++i) {
        sum += frequency[i];
      }
      int selector[] = new int[sum];
      int k0 = 0;
      int k1 = 0;
      for(int i=0; i<frequency.length; ++i) {
        for(int j=0; j<frequency[i]; ++j) {
          selector[k0] = k1;
          ++k0;
        }
        ++k1;
      }

      int p = (int)seed + 131 + n;
      if(p < 0)
        p = -p;
//      switch(p % 4) {
      switch(selector[p % selector.length]) {
      case 0:
        if (Debug.enabled)
          System.out.println("-> PerlinNoise  " + Arrays.toString(frequency) + " " + Arrays.toString(selector));
        return new PerlinNoise(seed + 3 * n);
      case 1:
        if (Debug.enabled)
         System.out.println("-> CellNoise    " + Arrays.toString(frequency) + " "  + Arrays.toString(selector));
        return new CellNoise(seed + 5 * n);
      case 2:
        if (Debug.enabled)
          System.out.println("-> ChladniNoise " + Arrays.toString(frequency) + " "  + Arrays.toString(selector));
        return new ChladniNoise(seed + 7 * n);
      case 3:
        if (Debug.enabled)
          System.out.println("-> ImageNoise   " + Arrays.toString(frequency) + " "  + Arrays.toString(selector));
        return new ImageNoise(seed + 11 * n);
      }
      assert false : "can not create noise pattern " + p;
      return new PerlinNoise(seed);
    
//    int p = (int)seed + 131;
//    if(p < 0)
//      p = -p;
//    switch(p % 2) {
//    case 0:
//      return new ImageNoise(seed);
//    case 1:
//      return new CellNoise(seed);
//    }
//    assert false : "can not create noise pattern " + p;
//    return new ImageNoise(seed);
    
//      return new ImageNoise(seed);
    }

    void setSeed(long seed1, long seed2, long seed3, long seed4) {
      int i = 7;
      noise1 = factory(seed1, i);
      i = 13;
      noise2 = factory(seed2, i);
      i = 23;
      noise3 = factory(seed3, i);
      i = 41;
      noise4 = factory(seed4, i);
    }

    public double sturbulence(double x, double y, double z, double octaves, double ampscale, double freqscale) {
      double n1 = source1 * noise1.sturbulence(x, y, z, octaves, ampscale, freqscale);
      double n2 = source2 * noise2.sturbulence(y, z, x, octaves, ampscale, freqscale);
      double n3 = source3 * noise3.sturbulence(z, x, y, octaves, ampscale, freqscale);
      double n4 = source4 * noise4.sturbulence((x+z)/2.0, (y+z)/2.0, z*z, octaves, ampscale, freqscale);
      return (n1 + n2 + n3 + n4) / 4.0;
    }

    public double uturbulence(double x, double y, double z, double octaves, double ampscale, double freqscale) {
      double n1 = source1 * noise1.uturbulence(x, y, z, octaves, ampscale, freqscale);
      double n2 = source2 * noise2.uturbulence(y, z, x, octaves, ampscale, freqscale);
      double n3 = source3 * noise3.uturbulence(z, x, y, octaves, ampscale, freqscale);
      double n4 = source4 * noise4.uturbulence((x+z)/2.0, (y+z)/2.0, z*z, octaves, ampscale, freqscale);
      return (n1 + n2 + n3 + n4) / 4.0;
    }
  }

  public DomainWarpingCalculation() {
    super();
    constraint = getConstraint();
    try {
      if(constraint.startsWith(FREQUENCY_KEY)) {
        String detail = constraint.substring(FREQUENCY_KEY.length());
        String[] parts = detail.split(",");
        frequency[0] = Integer.parseInt(parts[0]);
        frequency[1] = Integer.parseInt(parts[1]);
        frequency[2] = Integer.parseInt(parts[2]);
        frequency[3] = Integer.parseInt(parts[3]);
      }
    } catch (Exception ex) {
      System.err.println("Can not parse constraints: '" + constraint + "'");
      ex.printStackTrace();
    }
  }

  @Override
  public boolean reject(ChromosomeType chromosome) {
    DomainWarpingChromosome domainWarpingChromosome = (DomainWarpingChromosome) chromosome;
    init_params(domainWarpingChromosome);
    scanResults(domainWarpingChromosome);
    return scan_range < 0.001;
  }

  private void init_params(DomainWarpingChromosome domainWarpingChromosome) {
    mainscale = domainWarpingChromosome.getMainscale().getValue();
    gradientgain = domainWarpingChromosome.getGradientgain().getValue();
    
    List<WarpingLayerGene> wlg = domainWarpingChromosome.getWarpingLayer();
    int lindex = 0;
    for (Iterator<WarpingLayerGene> iterator = wlg.iterator(); iterator.hasNext();) {
      WarpingLayerGene warpingLayerGene = iterator.next();
      switch(lindex) {
      case 0:
        lay0.fill(warpingLayerGene);
        break;
      case 1:
        lay1ll.fill(warpingLayerGene);
        break;
      case 2:
        lay1rl.fill(warpingLayerGene);
        break;
      case 3:
        lay1lr.fill(warpingLayerGene);
        break;
      case 4:
        lay1rr.fill(warpingLayerGene);
        break;
      }
      ++lindex;
    }
  }

  @Override
  public void activateCanvas(Component viewComponent, Dimension canvasSize, boolean zoomMode) {
    depth = 6;
    xwMin = -1.0;
    xwMax = 1.0;
    ywMin = -1.0;
    ywMax = 1.0;
    super.activateCanvas(viewComponent, canvasSize, zoomMode);
    DomainWarpingChromosome domainWarpingChromosome = (DomainWarpingChromosome) chromosome;
    init_params(domainWarpingChromosome);
    scanResults(domainWarpingChromosome);
  }

  private double calc(final double xwk, final double ywk) {
    double value_lay1_ll = 0.0;
    if(lay1ll.n >= 1 ) {
      value_lay1_ll += lay1ll.term1.scaleOut * lay1ll.term1.composer.sturbulence(
                                                                   mainscale * lay1ll.term1.scaleXin * xwk,
                                                                   mainscale * lay1ll.term1.scaleYin * ywk,
                                                                   lay1ll.term1.timeCoord,
                                                                   lay1ll.term1.octaves, lay1ll.term1.ampscale, lay1ll.term1.freqscale);
    }
    if(lay1ll.n >= 2 ) {
      double val_lay1_llt2 = lay1ll.term2.scaleOut * lay1ll.term2.composer.sturbulence(
                                                                   mainscale * lay1ll.term2.scaleXin * xwk,
                                                                   mainscale * lay1ll.term2.scaleYin * ywk,
                                                                   lay1ll.term2.timeCoord,
                                                                   lay1ll.term2.octaves, lay1ll.term2.ampscale, lay1ll.term2.freqscale);
      if(val_lay1_llt2 < 0.0)
        value_lay1_ll = -value_lay1_ll;
    }
    
    double value_lay1_lr = 0.0;
    if(lay1lr.n >= 1 ) {
      value_lay1_lr += lay1lr.term1.scaleOut * lay1lr.term1.composer.sturbulence(
                                                                   mainscale * lay1lr.term1.scaleXin * xwk,
                                                                   mainscale * lay1lr.term1.scaleYin * ywk,
                                                                   lay1lr.term1.timeCoord,
                                                                   lay1lr.term1.octaves, lay1lr.term1.ampscale, lay1lr.term1.freqscale);
    }
    if(lay1lr.n >= 2 ) {
      double val_lay1_lrt2 = lay1lr.term2.scaleOut * lay1lr.term2.composer.sturbulence(
                                                                   mainscale * lay1lr.term2.scaleXin * xwk,
                                                                   mainscale * lay1lr.term2.scaleYin * ywk,
                                                                   lay1lr.term2.timeCoord,
                                                                   lay1lr.term2.octaves, lay1lr.term2.ampscale, lay1lr.term2.freqscale);
      if(val_lay1_lrt2 < 0.0)
        value_lay1_lr = -value_lay1_lr;
    }
    
    double value_lay1_rl = 0.0;
    if(lay1rl.n >= 1 ) {
      value_lay1_rl += lay1rl.term1.scaleOut * lay1rl.term1.composer.sturbulence(
                                                                   mainscale * lay1rl.term1.scaleXin * xwk,
                                                                   mainscale * lay1rl.term1.scaleYin * ywk,
                                                                   lay1rl.term1.timeCoord,
                                                                   lay1rl.term1.octaves, lay1rl.term1.ampscale, lay1rl.term1.freqscale);
    }
    if(lay1rl.n >= 2 ) {
      double val_lay1_rlt2 = lay1rl.term2.scaleOut * lay1rl.term2.composer.sturbulence(
                                                                   mainscale * lay1rl.term2.scaleXin * xwk,
                                                                   mainscale * lay1rl.term2.scaleYin * ywk,
                                                                   lay1rl.term2.timeCoord,
                                                                   lay1rl.term2.octaves, lay1rl.term2.ampscale, lay1rl.term2.freqscale);
      if(val_lay1_rlt2 < 0.0)
        value_lay1_rl = -value_lay1_rl;
    }
    
    double value_lay1_rr = 0.0;
    if(lay1rr.n >= 1 ) {
      value_lay1_rr += lay1rr.term1.scaleOut * lay1rr.term1.composer.sturbulence(
                                                                   mainscale * lay1rr.term1.scaleXin * xwk,
                                                                   mainscale * lay1rr.term1.scaleYin * ywk,
                                                                   lay1rr.term1.timeCoord,
                                                                   lay1rr.term1.octaves, lay1rr.term1.ampscale, lay1rr.term1.freqscale);
    }
    if(lay1rr.n >= 2 ) {
      double val_lay1_rrt2 = lay1rr.term2.scaleOut * lay1rr.term2.composer.sturbulence(
                                                                   mainscale * lay1rr.term2.scaleXin * xwk,
                                                                   mainscale * lay1rr.term2.scaleYin * ywk,
                                                                   lay1rr.term2.timeCoord,
                                                                   lay1rr.term2.octaves, lay1rr.term2.ampscale, lay1rr.term2.freqscale);
      if(val_lay1_rrt2 < 0.0)
        value_lay1_rr = -value_lay1_rr;
    }
    
    double value_lay0 = 0.0;
    if(lay0.n >= 1 ) {
      double factor = lay0.term1.scaleOut * lay0.term1.composer.uturbulence(
                                                                mainscale * lay0.term1.scaleXin * value_lay1_ll,
                                                                mainscale * lay0.term1.scaleYin * value_lay1_lr,
                                                                lay0.term1.timeCoord,
                                                                lay0.term1.octaves, lay0.term1.ampscale, lay0.term1.freqscale);
      double octaves = 7.0 * factor;
      double ampscale = factor;
      double freqscale = 1.0 / factor;
      value_lay0 += lay0.term1.scaleOut * lay0.term1.composer.uturbulence(
                                                               mainscale * lay0.term1.scaleXin * value_lay1_ll,
                                                               mainscale * lay0.term1.scaleYin * value_lay1_lr,
                                                               lay0.term1.timeCoord,
                                                               octaves, ampscale, freqscale);
    }
    if(lay0.n >= 2 ) {
      double factor = lay0.term2.scaleOut * lay0.term2.composer.uturbulence(
                                                                mainscale * lay0.term2.scaleXin * value_lay1_rl,
                                                                mainscale * lay0.term2.scaleYin * value_lay1_rr,
                                                                lay0.term2.timeCoord,
                                                                lay0.term2.octaves, lay0.term2.ampscale, lay0.term2.freqscale);
      double octaves = 7.0 * factor;
      double ampscale = factor;
      double freqscale = 1.0 / factor;
      double val_lay0t2 = lay0.term2.scaleOut * lay0.term2.composer.sturbulence(
                                                               mainscale * lay0.term2.scaleXin * value_lay1_rl,
                                                               mainscale * lay0.term2.scaleYin * value_lay1_rr,
                                                               lay0.term2.timeCoord,
                                                               octaves, ampscale, freqscale);
      if(val_lay0t2 < 0.0)
        value_lay0 = 1.0 - value_lay0;
    }

    return gain(gradientgain, value_lay0);
  }

  private static final double LOG_0_5 = Math.log(0.5);

  private static double bias(double b, double x) {
    b = Math.abs(b);
    x = Math.abs(x) + 0.00001;
    return Math.pow(x, Math.log(b) / LOG_0_5);
  }

  private static double gain(double g, double x) {
    g = Math.abs(g);
    x = Math.abs(x) + 0.00001;
    if(x < 0.5)
      return 0.5 * bias(1.0 - g, 2.0 * x);
    else
      return 1.0 - 0.5 * bias(1.0 - g, 2.0 - 2.0 *x);
  }

  @Override
  public void calculate(Deviation dev, boolean paintOnScreen, String exportFilename) {
    if (enterDepth()) {
      ColorF32 colorF32 = new ColorF32();
      if(dev != null)
        lay0.term1.composer.setWeight(lay0.term1.composer.source1 * ((kandid.calculation.domainwarping.Deviation)dev).s1,
                                      lay0.term1.composer.source2 * ((kandid.calculation.domainwarping.Deviation)dev).s2,
                                      lay0.term1.composer.source3 * ((kandid.calculation.domainwarping.Deviation)dev).s3,
                                      lay0.term1.composer.source4 * ((kandid.calculation.domainwarping.Deviation)dev).s4);
      while (more) {
        double gray = (calc(xw, yw) - scan_minValue) / scan_range;
        colorator.getColor(gray, colorF32);
        setPixel(colorF32);
        nextLocation();
      }
    }
  }

  @Override
  public boolean hasWhiteBackground() {
    return false;
  }

  double  scan_maxValue = Double.NEGATIVE_INFINITY;
  double  scan_minValue = Double.POSITIVE_INFINITY;
  double  scan_range = 0.00009;
  
  public void scanResults(DomainWarpingChromosome scalarExpressionChromosome) {
    scan_maxValue = Double.NEGATIVE_INFINITY;
    scan_minValue = Double.POSITIVE_INFINITY;
    scan_range = 0.00009;
    double xt = -1.1;
    while (xt < 1.1) {
      double yt = -1.1;
      while (yt < 1.1) {
        double scan_value = calc(xt, yt);
        if (scan_value < scan_minValue)
          scan_minValue = scan_value;
        if (scan_value > scan_maxValue)
          scan_maxValue = scan_value;
        yt += 0.1;
      }
      xt += 0.1;
    }
    scan_range = scan_maxValue - scan_minValue;
  }
  
}
