/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.bridge.povray;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;

import kandid.calculation.bridge.BatchJob;
import kandid.calculation.bridge.BridgeBase;
import kandid.util.Debug;

/**
 * Read data stream from child process in am seperate thread.
 * @author thomas jourdan
 *
 */
public class TargaStreamReader implements Runnable {

  private static final int rowsPerTile = 25;
  private Process childProcess;
  private BatchJob job;
  private int externalHeight;
  private int externalWidth;
  private int increment;
  private Thread engine;    // Thread reading image from external process

  public TargaStreamReader(Process childProcess, BatchJob job) {
    this.childProcess = childProcess;
    this.job = job;
    externalHeight = job.canvasSize.height;
    externalWidth = job.canvasSize.width;
    if(job.preview) {
      externalHeight /= BridgeBase.previewScale;
      externalWidth /= BridgeBase.previewScale;
    }
    if(job.importedImage == null)
      job.importedImage = new BufferedImage(job.canvasSize.width, job.canvasSize.height, BufferedImage.TYPE_INT_RGB);
  }

  /**
   * start data reader
   */
  public void start() {
    if (engine == null) {
      engine = new Thread(this);
      engine.start();
    }
  }

  /**
   * Reads data stream from child process. It is assumed that the data stream is in Targa format.
   * @see java.lang.Runnable#run()
   */
  public void run() {
    try {
      datareader: {
//!!        job.streamReady = false;
        InputStream dataStream = childProcess.getInputStream();
        int targaHeader[] = new int[18];
        for (int hx = 0; hx < targaHeader.length; hx++) {
          int ch = dataStream.read();
          if(ch < 0)
            break datareader;
          targaHeader[hx] = ch;
        }
        int width = targaHeader[12] + 256 * targaHeader[13];
        int height = targaHeader[14] + 256 * targaHeader[15];
        if(Debug.enabled) assert(externalWidth == width) : "width " + width;
        if(Debug.enabled) assert(externalHeight == height) : "height" + height;

        int[] pixel = new int[rowsPerTile*width];
        increment = 0;
        int row = 0;
        outerLoop :
        for (int rx = 0; rx < height; rx++) {
          for (int cx = 0; cx < width; cx++) {
            if(job.state == BatchJob.stateDestroied)
              break datareader;
            int b = dataStream.read();
            if (b < 0)
              break outerLoop;
            int g = dataStream.read();
            if (g < 0)
              break outerLoop;
            int r = dataStream.read();
            if (r < 0)
              break outerLoop;
            int rgbColor = (r << 16) + (g << 8) + b;  //TODO Does Pov-Ray support floating point image file formats?
            if(!job.preview) {
              pixel[increment*width+cx] = rgbColor;
            }
            else {
              for (int rx2 = 0; rx2 < BridgeBase.previewScale; rx2++) {
                for (int cx2 = 0; cx2 < BridgeBase.previewScale; cx2++) {
                  job.importedImage.setRGB(cx*BridgeBase.previewScale+cx2, rx*BridgeBase.previewScale+rx2, rgbColor);
                }
              }
            }
          }
          if(increment >= rowsPerTile-1) {
            if(!job.preview) {
              job.importedImage.setRGB(0, row, width, rowsPerTile, pixel, 0, width);
            }
            row += rowsPerTile;
            synchronized(job.incrementReady) {
              job.incrementReady.notify();
            }
            increment = 0;
          }
          else {
            ++increment;
          }
        }
        if(!job.preview) {
          job.importedImage.setRGB(0, row, width, increment, pixel, 0, width);
        }
      }
    }
    catch (IOException exc) {
      job.exitCode = -3;
      Debug.stackTrace(exc);
    }
    catch (Throwable exc) {
      job.exitCode = -4;
      Debug.stackTrace(exc);
    }
    if (Debug.enabled) System.out.println("TargaStreamReader ready : " + job.imageFileName +  " " + Debug.currentTime());
    if(job.state != BatchJob.stateDestroied) {
//!!      job.streamReady = true;
      job.state = BatchJob.stateReady;
    }
    synchronized(job.incrementReady) {
      job.incrementReady.notify();
    }
 }
}
