/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.bridge.povray;

import java.awt.image.BufferedImage;

import kandid.calculation.bridge.BatchJob;
import kandid.calculation.bridge.PixelBridgeCalculation;
import kandid.calculation.bridge.StreamBridge;
import kandid.colorator.Colorator;
import kandid.soup.ChromosomeType;
import kandid.util.Debug;

/**
 * @author thomas jourdan
 *
 */
public class PovTextureCalculation extends PixelBridgeCalculation {
  
  private BufferedImage importedPreviewImage;

  /* (non-Javadoc)
   * @see kandid.calculation.Calculation#setChromosome(kandid.soup.ChromosomeType, kandid.colorator.Colorator)
   */
  public void setChromosome(ChromosomeType chromosome, Colorator colorator) {
    super.setChromosome(chromosome, colorator);
    importedPreviewImage = null;
  }

  /* (non-Javadoc)
   * @see kandid.calculation.newbridge.PixelBridgeCalculation#calculateView()
   */
  protected void calculateView() {
      calculateStreamingView();
  }
  
  /**
   * Calculate image for display.
   */
  private synchronized void calculateStreamingView() {
    if(!aborted && runCount < 2) {
      PovJob job = null;
      try {
        ++runCount;
        job = new PovJob();
        job.createJob(chromosome, canvasSize, runCount == 1, null, "png");
        job.importedImage = importedPreviewImage;
        if (Debug.enabled) System.out.println("PovTextureCalculation.calculateStreamingView(): " + job.imageFileName + " " + Debug.currentTime());
        bbridge = new StreamBridge(job);
        bbridge.execute();
        boolean ready = false;
        do {
          ready = bbridge.recieve(bufAwtImg.getGraphics());
          viewComponent.repaint();
        }
        while (!ready);
        bbridge.waitFor();
        if(job.state != BatchJob.stateDestroied) {
          ready = bbridge.recieve(bufAwtImg.getGraphics());
          viewComponent.repaint();
        }
        bbridge.cleanUp();
        if(runCount == 1)
          importedPreviewImage = job.importedImage;
      }
      catch (Exception exc) {
        if(job != null)
          System.err.println(job.imageFileName);
        Debug.stackTrace(exc);
      }
      ready = runCount >= 2;
    }
    bbridge = null;
  }

  /**
   * Calculate image for file export.
   * @param exportFilename
   */
  public void calculateExport(String exportFilename) {
    PovJob job = new PovJob();
    job.createJob(chromosome, canvasSize, false, exportFilename, "png");
    if (Debug.enabled) System.out.println("PovCalculation.calculateExport(): " + job.imageFileName + " " + Debug.currentTime());
    bbridge = new StreamBridge(job);
    bbridge.execute();
    bbridge.waitFor();
    bbridge = null;
    ready = true;
  }

}
