/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.bridge.povray;

import java.util.Iterator;
import java.util.List;

import javax.swing.JOptionPane;

import kandid.Kandid;
import kandid.calculation.bridge.BridgeBase;
import kandid.colorator.ColorF32;
import kandid.calculation.bridge.BatchJob;
import kandid.soup.ChromosomeType;
import kandid.soup.ColorMapEntryGene;
import kandid.soup.HsbftGene;
import kandid.soup.Matrix4X3Gene;
import kandid.soup.NormalMapEntryGene;
import kandid.soup.PovBlackHoleWarpGene;
import kandid.soup.PovBlendGene;
import kandid.soup.PovFinishGene;
import kandid.soup.PovIsoSurfaceChromosome;
import kandid.soup.PovNormalGene;
import kandid.soup.PovPigmentGene;
import kandid.soup.PovReflectionGene;
import kandid.soup.PovRidgedMultifractalGene;
import kandid.soup.PovRotateGene;
import kandid.soup.PovTextureLayerGene;
import kandid.soup.PovThingChromosome;
import kandid.soup.PovThingNFChromosome;
import kandid.soup.PovTransformMapGene;
import kandid.soup.PovTurbulenceWarpGene;
import kandid.soup.PovWarpGene;
import kandid.soup.util.ChromosomeStamp;

/**
 * @author thomas jourdan
 *
 */
public class PovJob extends BatchJob {

  protected String[] buildEnvironment() {
    return null;
  }

  /* (non-Javadoc)
   * @see kandid.calculation.bridge.Bridge#buildCommand(kandid.soup.ChromosomeType, java.lang.String)
   */
  protected String[] buildCommand() {
    // +Isimpleworld3.pov +Osimpleworld3.png +W512 +H512 +Q9
    assert(imageFormat.equals("png") || imageFormat.equals("")) : "unsupported image format " + imageFormat;
    String[] command = new String[14];
    command[0] = childProgram;
    command[1] = "+I" + scriptFileName;
    if (doExport) {
      command[2] = "+O" + imageFileName;
    }
    else {
      command[2] = "+O-";
    }
    command[3] = "+W" + canvasSize.width / (preview ? BridgeBase.previewScale : 1);
    command[4] = "+H" + canvasSize.height / (preview ? BridgeBase.previewScale : 1);
    command[5] = "+Q9";
    command[6] = "-V";
    command[7] = doExport ? "+FN" : "+FT";
    command[8] = "Debug_Console=Off";
    command[9] = "Fatal_Console=On";
    command[10] = "Render_Console=Off";
    command[11] = "Statistic_Console=Off";
    command[12] = "Warning_Console=On";
    command[13] = "Display=Off";

    return command;
  }

  protected String getProgramName() {
    // set up command
    return "povray";
  }

  /* (non-Javadoc)
   * @see kandid.calculation.newbridge.BatchJob#getScriptExtension()
   */
  protected String getScriptExtension() {
    return ".pov";
  }

  /**
   * Normalizes the weight factors of an color map.
   * @param colorMap
   * @return array of segment start values.
   */
  protected static double[] normalizeColorMap(List colorMap) {
    Iterator iter = colorMap.iterator();
    int size = colorMap.size()-1;
    double[] segmentStart = new double[size];
    ColorMapEntryGene colorMapEntryGene = (ColorMapEntryGene)iter.next();
    double sum = colorMapEntryGene.getWeight().getValue();
    int ix = 0;
    while (iter.hasNext()) {
      colorMapEntryGene = (ColorMapEntryGene)iter.next();
      segmentStart[ix] = sum;
      sum += colorMapEntryGene.getWeight().getValue();
      ++ix;
    }
    // normalize
    for (ix = 0; ix < size; ++ix) {
      segmentStart[ix] /= sum;
    }
    return segmentStart;
  }
  
  /**
   * Normalizes the wieght factors of an color map. First value of the returend array is always 0.0, last value is 1.0
   * @param colorMap
   * @return array of segment start values: first elemet = 0.0, last element = 1.0
   */
  protected static double[] normalizeColorMap01(List aMap) {
    Iterator iter = aMap.iterator();
    int size = aMap.size();
    double[] segmentStart = new double[size];
    double sum = 0.0;
    int ix = 0;
    while (iter.hasNext()) {
      Object aEntry = iter.next();
      double weight = 1.0;
      if(aEntry instanceof ColorMapEntryGene)
        weight = ((ColorMapEntryGene)aEntry).getWeight().getValue();
      else if(aEntry instanceof NormalMapEntryGene)
        weight = ((NormalMapEntryGene)aEntry).getWeight().getValue();
      else
        assert false : "invalid type " + aEntry.getClass().getName();
      segmentStart[ix] = sum;
      if((ix+1) < segmentStart.length) {
        sum += weight;
      }
      ++ix;
    }
    // normalize
    for (ix = 0; ix < size; ++ix) {
      segmentStart[ix] /= sum;
    }
    return segmentStart;
  }
  
  /**
   * @param script
   * @param blend
   * @return
   */
  private String exBlend(String script, PovBlendGene blend) {
    script += "        " + blend.getBlendMapModifier().value() + "\n";
    script += "        frequency " + blend.getBlendMapFrequency().getValue() + "\n";
    script += "        phase " + blend.getBlendMapPhase().getValue() + "\n";
    return script;
  }
  
  /**
   * @param script
   * @param rgbftGene
   * @return
   */
  private String exColor(String script, HsbftGene rgbftGene) {
    ColorF32 hsb = new ColorF32((float)(double)rgbftGene.getHue(), (float)(double)rgbftGene.getSaturation(), (float)(double)rgbftGene.getBrightness());
    ColorF32 rgb = new ColorF32();
    ColorF32.hsb2rgb(hsb, rgb);
    script += " rgbft <" + rgb.r + ", " 
                         + rgb.g + ", " 
                         + rgb.b + ", " 
                         + rgbftGene.getFilter().getValue() + ", " 
                         + rgbftGene.getTransmit().getValue() 
                         + "> ";
    return script;
  }
  
  /* (non-Javadoc)
   * @see kandid.calculation.newbridge.BatchJob#expandScript(kandid.soup.ChromosomeType)
   */
  protected String expandScript(ChromosomeType chromosome) {
    String ident = ChromosomeStamp.removeType(chromosome.getIdent());
    PovThingChromosome povThingChromosome = (PovThingChromosome)chromosome;

    String script = "";
    script += "// Script for Persistence Of Vision raytracer version 3.7.\n";
    script += "// Generated by Kandid version " + Kandid.release + "\n";
    script += "// https://metagrowing.org/\n";
    script += "//\n";
    
    script += "#declare Texture1_" + ident + " =\n";      
    List textureLayerList = povThingChromosome.getTextureLayer();
    assert textureLayerList.size() >= 2;
    for (Iterator textureLayerIter = textureLayerList.iterator(); textureLayerIter.hasNext();) {
      PovTextureLayerGene povTextureLayerGene = (PovTextureLayerGene) textureLayerIter.next();
      
      script += "  texture {\n";
      PovPigmentGene pigment = povTextureLayerGene.getPigment();
      script += "      pigment {\n";
      if(pigment == null) {
        script += "        rgb <1, 1, 1>\n";
      }
      else {
        script += "        " + pigment.getColorMap().getMapPattern().value() + "\n";
        
        // color map  
        script += "        color_map {\n";        
        List colorMapList = pigment.getColorMap().getColor();
        double[] segmentStart = normalizeColorMap01(colorMapList);
        int cmx = 0;
        Iterator colorMapIter = colorMapList.iterator();
        for (; cmx < segmentStart.length && colorMapIter.hasNext();) {
          ColorMapEntryGene colorMapEntryGene = (ColorMapEntryGene)colorMapIter.next();
          script += "        [ " + segmentStart[cmx];
          script = exColor(script, colorMapEntryGene.getHsbft());
          script += "]\n";
          ++cmx;
        }        
        script += "        }\n";
        
        // warp
        script = exWarp(script, pigment.getWarp());

        // transform
        script = exTransform(script, pigment.getTransform());

        // perlin noise generator
        script += "        noise_generator 3\n";
        
        // wave form for color map
        script = exBlend(script, pigment.getBlend());

      }
      script += "      }\n";
      script += "    }\n";
    }
    script +=   "\n";
        
    PovNormalGene normal = null;
    if(povThingChromosome instanceof PovThingNFChromosome) {
      normal = ((PovThingNFChromosome)povThingChromosome).getNormal();
      if(normal != null) {
        script += "#declare Normal1_" + ident + " =\n";
        script += "   normal {\n";
        if(normal.getNormalMap().getAverage().isValue()) {
          script += "     average\n";
        }
        else {
          script += "     " + normal.getNormalMap().getMapPattern().value() + "\n";
        }
        
        // normal map  
        script += "     normal_map {\n";        
        List normalMapList = normal.getNormalMap().getNormal();
        double[] segmentStart = normalizeColorMap01(normalMapList);
        int nmx = 0;
        Iterator normalMapIter = normalMapList.iterator();
        for (; nmx < segmentStart.length && normalMapIter.hasNext();) {
          NormalMapEntryGene normalMapEntryGene = (NormalMapEntryGene)normalMapIter.next();
          script +=
          "     [ " + segmentStart[nmx]
                    + " " + normalMapEntryGene.getMapPattern().value()
                    + " " + normalMapEntryGene.getDepth().getValue()
                    + " bump_size "+ normalMapEntryGene.getBumpSize().getValue() + "\n";
          // warp
          script = exWarp(script, normalMapEntryGene.getWarp());
      
          // transform
          script = exTransform(script, normalMapEntryGene.getTransform());
      
          // noise generator
          script += "     noise_generator 3\n";
              
          script +=        "     ]\n";
          ++nmx;
        }        
        script += "     }\n";
        
        // warp
        script = exWarp(script, normal.getWarp());
      
        // transform
        script = exTransform(script, normal.getTransform());
      
        // perlin noise generator
        script += "     noise_generator 3\n";
        
        // wave form for normal map
        script = exBlend(script, normal.getBlend());
        script += "   }\n";
      }
    }
        
    PovFinishGene finish = null;
    if(povThingChromosome instanceof PovThingNFChromosome) {
      finish = ((PovThingNFChromosome)povThingChromosome).getFinish();
      if(finish != null) {
        script += "#declare Finish1_" + ident + " =\n";
        script += "  finish {\n";
        script += "    ambient " + finish.getAmbient().getValue() + "\n";
        script += "    diffuse " + finish.getDiffuse().getValue() + "\n";
        script += "    specular " + finish.getSpecular().getValue() + "\n";
        script += "    roughness " + finish.getRoughness().getValue() + "\n";
        if(finish.getMetallic().isValue()) {
          script += "    metallic\n";
        }
        PovReflectionGene reflection = finish.getReflection();
        script += "    reflection {\n";
        script = "    " + exColor(script, reflection.getColor()) + "\n";
        if(reflection.getMetallic().isValue()) {
          script += "      metallic\n";
        }
        script += "    }\n";
        script += "    conserve_energy\n";
        script += "  }\n";
      }
    }
        
    script += "\n";
    script += "global_settings {\n";
    script += "   assumed_gamma 1.5\n";
    script += "}\n";
    script += "\n";

    if(povThingChromosome instanceof PovIsoSurfaceChromosome) {
      PovIsoSurfaceChromosome povIsoSurfaceChromosome = (PovIsoSurfaceChromosome)povThingChromosome;
//!!      script += "#declare f_sphere = function { internal(61) }\n";
      script += "#declare f_ridged_mf = function { internal(59) }\n";
      script += "#declare f_noise3d = function { internal(76) }\n";
      script += "#declare f_snoise3d = function {2*f_noise3d(x, y, z) - 1}\n";
      script += "\n";

      PovRidgedMultifractalGene ridgedMultifractal = povIsoSurfaceChromosome.getRidgedMultifractal();    
      script += "#declare fx_" + ident + " = ";
      script += (new SdlExport()).generateSDLFunction(ridgedMultifractal.getFx());
      script += "\n";

      script += "#declare fy_" + ident + " = ";
      script += (new SdlExport()).generateSDLFunction(ridgedMultifractal.getFy());
      script += "\n";

      script += "#declare fz_" + ident + " = ";
      script += (new SdlExport()).generateSDLFunction(ridgedMultifractal.getFz());
      script += "\n";

//      script += "#declare f_radius_" + ident + " = ";
//      script += (new SdlExport()).generateSDLFunction(povIsoSurfaceChromosome.getSdlRadius());
//      script += "\n";
//    
//      script += "#declare f_offset_" + ident + " = ";
//      script += (new SdlExport()).generateSDLFunction(povIsoSurfaceChromosome.getSdlOffset());
//      script += "\n";    

      script += "#declare Thing_1 = isosurface {\n";
//!!      script += "  function { f_sphere(x, y, z, (f_radius_" + ident + " (x, y, z))) + f_offset_" + ident + "(x, y, z) }\n";
//  script += "  function { f_noise3d(f_radius_" + ident + " (x, y, z), f_offset_" + ident + "(x, y, z), z) }\n";
      script += "  function { f_ridged_mf(fx_" + ident + "(x, y, z), fy_" + ident + "(x, y, z), fz_" + ident + "(x, y, z), ";
      script += ridgedMultifractal.getH().getValue() + ", ";
      script += ridgedMultifractal.getLacunarity().getValue() + ", ";
      script += ridgedMultifractal.getOctaves().getValue() + ", ";
      script += ridgedMultifractal.getOffset().getValue() + ", ";
      script += ridgedMultifractal.getGain().getValue() + ", ";
      script += "3) }\n";
//  !!    script += "  contained_by { box { <-1, -1, -1>, <1, 1, 1> } }\n";
      script += "  contained_by { sphere { <0, 0, 0>, 1.5 } }\n";
      script += "  open\n";
      script += "  threshold " + povIsoSurfaceChromosome.getThreshold().getValue() + "\n";
      script += "  accuracy 0.1\n";
      script += "  evaluate 1.1, 1.30, 0.7\n";
      script += "}\n";
      script += "\n";
    }
    else {
//      script += "#declare Thing_1 = box { <-1, -1, -1>, <1, 1, 1> }";
      script += "#declare Thing_1 = sphere { <0, 0, 0>, 1.5 }";
      script += "\n";
    }

    script += "object {\n";
    script += "   Thing_1\n";
    script += "   texture {\n";
    script += "     Texture1_" + ident + "\n";
    script += "   }\n";
    if(normal != null) {
      script += "   normal {\n";
      script += "     Normal1_" + ident + "\n";
      script += "   }\n";
    }
    if(finish != null) {
      script += "   finish {\n";
      script += "     Finish1_" + ident + "\n";
      script += "   }\n";
    }
    script += "   scale 1\n";
    if(povThingChromosome instanceof PovIsoSurfaceChromosome) {
      PovIsoSurfaceChromosome povIsoSurfaceChromosome = (PovIsoSurfaceChromosome)povThingChromosome;
      PovRotateGene rotate = povIsoSurfaceChromosome.getRotate();
      script += "   rotate <" + rotate.getRx().getValue() + ", " + rotate.getRy().getValue() + ", " + rotate.getRz().getValue() + ">\n";
    }
    else {
      script += "   rotate <0, 0, 0>\n";
    }
    script += "   translate y*0.5\n";
    script += "}\n";
    script += "\n";
//!!    script += "plane {\n";
//!!    script += "   <0, 1, 0>, 0\n";
//!!    script += "\n";
//!!    script += "   pigment {\n";
//!!    script += "      checker\n";
//!!    script += "      color rgb <0.882353, 0.882353, 0.882353>\n";
//!!    script += "      color rgb <0.498039, 0.498039, 0.498039>\n";
//!!    script += "   }\n";
//!!    script += "   scale 1\n";
//!!//!!    script += "   rotate <0, 0, 0>\n";
//!!    script += "   translate y*(-0.5)\n";
//!!    script += "}\n";
    script += "\n";
    script += "light_source {\n";
    script += "   <0, 4, 0>, rgb <1, 1, 1>\n";
    script += "   area_light <1, 0, 0>, <0, 1, 0>, 3, 3\n";
    script += "}\n";
    script += "\n";
    script += "light_source {\n";
    script += "   <4, 5, -5>, rgb <1, 1, 1>\n";
    script += "}\n";
    script += "\n";
    script += "camera {\n";
    script += "   perspective\n";
    script += "   location <2, 3.3, -1.5>\n";
    script += "   sky <0, 1, 0>\n";
    script += "   direction <0, 0, 1>\n";
    script += "   right <1, 0, 0>\n";
    script += "   up <0, 1, 0>\n";
    script += "  look_at <0, 0.5, 0>\n";
    script += "}\n";
    return script;
  }
  
  /**
   * @param script
   * @param transform
   * @return
   */
  private String exTransform(String script, PovTransformMapGene transform) {
    Matrix4X3Gene matrix = transform.getMatrix();
    script += "        transform {\n";
    script += "          matrix <\n";
    script += "            " + matrix.getVal00().getValue() + ", " + matrix.getVal01().getValue() + ", " + matrix.getVal02().getValue() + ",\n";
    script += "            " + matrix.getVal10().getValue() + ", " + matrix.getVal11().getValue() + ", " + matrix.getVal12().getValue() + ",\n";
    script += "            " + matrix.getVal20().getValue() + ", " + matrix.getVal21().getValue() + ", " + matrix.getVal22().getValue() + ",\n";
    script += "            " + matrix.getVal30().getValue() + ", " + matrix.getVal31().getValue() + ", " + matrix.getVal32().getValue() + "\n";
    script += "          >\n";
    script += (transform.getInvers().isValue() ? "          inverse\n" : "");
    script += "        }\n";
    return script;
  }
  
  /**
   * @param script
   * @param warp
   * @return
   */
  private String exWarp(String script, PovWarpGene warp) {
    // turbulence
    PovTurbulenceWarpGene povWarpGene = warp.getTurbulence();
    script += "        warp {\n";        
    script += "          turbulence\n";
    script += "          <" + povWarpGene.getTurbulenceVector().getX().getValue() + ", " + povWarpGene.getTurbulenceVector().getY().getValue() + ", " + povWarpGene.getTurbulenceVector().getZ().getValue() + ">\n";
    script += "          omega " + povWarpGene.getOmega().getValue() + "\n";
    script += "          lambda " + povWarpGene.getLambda().getValue() + "\n";
    script += "        }\n";
    
    // black hole    
    List blackHoleList = warp.getBlackHole();
    for (Iterator blackHoleItr = blackHoleList.iterator(); blackHoleItr.hasNext();) {
      PovBlackHoleWarpGene blackHoleWarp = (PovBlackHoleWarpGene) blackHoleItr.next();
      script += "        warp {\n";        
      script += "          black_hole\n";
      script += "          <" + blackHoleWarp.getLocation().getX().getValue() + ", " + blackHoleWarp.getLocation().getY().getValue() + ", " + blackHoleWarp.getLocation().getZ().getValue() + ">, " + blackHoleWarp.getRadius().getValue() + "\n";
      script += "          falloff " + blackHoleWarp.getFalloff().getValue() + "\n";
      script += "          strength " + blackHoleWarp.getStrength().getValue() + "\n";
      script += (blackHoleWarp.getInvers().isValue() ? "          inverse\n" : "");
      script += "        }\n";
    }
    return script;
  }

  protected void showExecError() {
    if ((showExecError & 2) == 0) {
      showExecError = 2;
      JOptionPane.showMessageDialog(
        Kandid.getFrame(),
        "<HTML>'"
          + getProgramName()
          + "' not found."
          + "<br>"
          + "<br>POV-Ray 'Persistence of Vision' is required."
          + "<br>"
          + "<br>You can get this software and its dependencies as a Debian package."
          + "<br><b>povray.deb</b>"
          + "<br>"
          + "<br>The genetic engine inside Kandid can be used as a front end to evolve layered textures for "
          + "<br>the Persistence of Vision ray tracer. Persistence of Vision acts, behind the scene, as an compiler "
          + "<br>reading a POV-Ray SDL (Scene Description Language), generated by Kandid, producing images. "
          + "<br>It is not necessary to edit the parameter files by hand nor starting povray manually. "
          + "<br>From the users view there is no distinction between the build in and the external calculations."
          + "<br>"
          + "<br>The integration of POV-Ray (version 3.7) and Kandid (version 1.1.0) was tested under "
          + "<br>Debian 13 Trixie using openjdk (version 21.0.8 2025-07-15.)"
          + "<br>"
          + "<br>http://www.povray.org/",
        "POV-Ray, additional software reqiered",
        JOptionPane.ERROR_MESSAGE);
    }
  }

}
