/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.bridge;

import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

import kandid.Console;
import kandid.Kandid;
import kandid.util.Debug;

/**
 * @author thomas jourdan
 *
 */
public abstract class BridgeBase {

  protected BatchJob batchJob;
  protected Process childProcess;
  public final static int previewScale = 3;
  protected AffineTransform scale;

  protected static boolean niceFound = false;
  private static boolean niceTest = false;

  /**
   * 
   */
  public BridgeBase(BatchJob batchJob) {
    this.batchJob = batchJob;
    scale = new AffineTransform();
    scale.setToScale(previewScale, previewScale);
    File scratchDir = new File(Kandid.scratchFolder);
    if (!scratchDir.exists()) {
      scratchDir.mkdirs();
    }
  }
  
  /**
   * Read image, produced by the external program
   * @return
   * @throws IOException
   */
  public abstract BufferedImage getImage() throws IOException;
  
  /**
   * Delete scratch files.
   */
  public abstract void cleanUp();
  
  /**
   * Execute external program, to produce an image
   */
  public synchronized void execute() {
    if(Debug.enabled && batchJob.state != BatchJob.statePending) {
      System.err.println("execute() wrong state = " + batchJob.state + " " + batchJob.imageFileName);
      Debug.stackTrace(new Exception("" + Debug.currentTime()));
    }
    
    try {
      writeScript();
    }
    catch (IOException exc) {
      // script can not be written
      batchJob.state = BatchJob.stateDestroied;
      Debug.stackTrace(exc);
      return;
    }
    
    try {      
      if (batchJob.environment == null && tryNice()) {
        String[] niceCmd = new String[2 + batchJob.comandline.length];
        niceCmd[0] = "nice";
        niceCmd[1] = "-5";
        for (int px = 0; px < batchJob.comandline.length; px++) {
          niceCmd[2 + px] = batchJob.comandline[px];
        }
        // start child process
        childProcess = Runtime.getRuntime().exec(niceCmd, batchJob.environment);
      }
      else {
        // start child process
        childProcess = Runtime.getRuntime().exec(batchJob.comandline, batchJob.environment);
      }
      batchJob.state = BatchJob.stateRunning;
      if (Debug.enabled) System.out.println("childProcess.execute(): " + batchJob.imageFileName +  " " + Debug.currentTime());
  
      (new StderrReader(childProcess, batchJob)).start();
    }
    catch (IOException exc) {
      // child process can not be started
      batchJob.state = BatchJob.stateDestroied;
      Debug.stackTrace(exc);
      batchJob.showExecError();  // TODO
    }
  }  
  
  /**
   * Wait until the external program is ready or destroied.
   */
  public void waitFor() {
    if(Debug.enabled && !(batchJob.state == BatchJob.stateRunning || batchJob.state == BatchJob.stateReady || batchJob.state == BatchJob.stateDestroied)) {
      System.err.println("waitFor() wrong state = " + batchJob.state + " " + batchJob.imageFileName);
      Debug.stackTrace(new Exception("" + Debug.currentTime()));
    }
    if(batchJob.state == BatchJob.stateRunning || batchJob.state == BatchJob.stateReady) {
      try {
        // wait for child to finish
        if (Debug.enabled) System.out.println("childProcess.waitFor(): " + batchJob.imageFileName + " " + Debug.currentTime());
        //if(Debug.enabled) Debug.stackTrace(new Exception("childProcess.waitFor(): " + imageName + " " + Debug.currentTime()));
        if(childProcess != null) {
          batchJob.exitCode = childProcess.waitFor();
          if (batchJob.state == BatchJob.stateRunning) {
            batchJob.state = BatchJob.stateReady;
            if (Debug.enabled) System.out.println("BatchJob.statusReady  : " + batchJob.imageFileName + " " + Debug.currentTime());
          }
        }
        else {
          batchJob.state = BatchJob.stateDestroied;
        }
      }
      catch (InterruptedException exc) {
        // InterruptedException is ok
        batchJob.state = BatchJob.stateDestroied;
        batchJob.exitCode = -7;
        //Debug.stackTrace(exc);
      }
      catch (Throwable exc) {
        batchJob.state = BatchJob.stateDestroied;
        batchJob.exitCode = -8;
        Debug.stackTrace(exc);
      }
      finally {
        childProcess = null;
        // display exit code and error stream of subprocess
        if (batchJob.exitCode != batchJob.expextedExitCode) {
          if (Debug.enabled) {
            System.out.println("'" + batchJob.comandline[0] + "' exited with " + batchJob.exitCode+ ": " + batchJob.imageFileName + Debug.currentTime());
            if(batchJob.errorMessage.length() > 0) System.out.println(batchJob.errorMessage.toString());
          }
        }
        if(batchJob.errorMessage.length() > 0) {
          if (Debug.enabled) System.out.println(batchJob.errorMessage.toString());
          Console.append(batchJob.errorMessage.toString());
        }
      }
    }
  }

  /**
   * Kill external program.
   */
  public synchronized void destroy() {
    if (batchJob.state == BatchJob.stateRunning) {
      batchJob.state = BatchJob.stateDestroied;
      if (childProcess != null) {
        if (Debug.enabled) System.out.println("childProcess.destroy(): " + batchJob.imageFileName +  " " + Debug.currentTime());
        childProcess.destroy();
        childProcess = null;
      }
    }
    batchJob.state = BatchJob.stateDestroied;
    cleanUp();
  }
  
  /**
   * Tests if the operating system supports the nice command.
   * @return true if the nice command is found.
   */
  public static boolean tryNice() {
    if(!niceTest) {
      niceTest = true;
      try {
        String[] cmdarray = new String[1];
        cmdarray[0] = "nice";
        Process childProcess = Runtime.getRuntime().exec(cmdarray);
        niceFound = true;
        if(childProcess != null)
          childProcess.destroy();
        System.out.println("detecting nice command for external processes");
      }
      catch (IOException exc) {
      }
    }
    return niceFound;
  }

  /**
   * Write script to file.
   * @throws IOException
   */
  protected void writeScript() throws IOException {
    FileWriter fileWriter = new FileWriter(new File(batchJob.scriptFileName));
    fileWriter.write(batchJob.script);
    fileWriter.close();
  }

}
