/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation;

import java.awt.Component;
import java.awt.Dimension;

import kandid.colorator.ColorF32;
import kandid.util.*;

/**
 * Base class for calculations in world coordinate system.
 * @author thomas jourdan
 */
public abstract class WorldCalculation extends PixelCalculation implements Cloneable {
  protected double xmin;
  protected double xmax;
  protected double ymin;
  protected double ymax;
  protected double xfactor;
  protected double yfactor;
  protected double xw;
  protected double yw;
  protected int    xp;
  protected int    yp;

//  /**
//   * Constructor for calculation based on world coordinates.
//   */
//  public void init(int xDim, int yDim) {
//    super.init(xDim, yDim);
//  }
//
  /* For documentation please see super class Calculation. */
  public void activateCanvas(Component viewComponent, Dimension canvasSize, boolean zoomMode) {
    super.activateCanvas(viewComponent, canvasSize, zoomMode);
    xmin    = ymin = 0.0;
    xmax    = ymax = 1.0;
    xfactor = (double)canvasSize.width;
    yfactor = (double)canvasSize.height;
    xp      = yp = 0;
  }
  
  /**
   * Initialize transformation from word coordinates to pixel coordinates.
   *
   * @param borderFactor factor for the size of a surrounding area.
   */
  protected void initTransformation(double borderFactor) {
    double xBorder = borderFactor * (xmax - xmin);
    double yBorder = borderFactor * (ymax - ymin);
    xmax += xBorder;
    xmin -= xBorder;
    ymax += yBorder;
    ymin -= yBorder;
    double diffX = xmax - xmin;
    double diffY = ymax - ymin;
    if (diffX > 0.0) {
      xfactor = (double)canvasSize.width / diffX;
    }
    if (diffY > 0.0) {
      yfactor = (double)canvasSize.height / diffY;
    }
  }

  /**
   * Sets the actual 'pixel' in to the buffered image.
   *
   * @param color 24 Bit color value
   */
  protected void setWPixel(ColorF32 colorF32) {
    xp = (int)((xw - xmin) * xfactor);
    yp = (int)((yw - ymin) * yfactor);
    try {
      bufAwtImg.setRGB(xp, yp, ColorF32.toRGB8(colorF32));
      ColorF32.copy(colorF32, bufColorF32Img[yp][xp]);
    } catch (ArrayIndexOutOfBoundsException exc) {
      if(Debug.enabled) System.out.println("bufImg coordinates out of bounds (" + xp + ", " + yp + ")");
    }
  }

  /**
   * Transform from world coordinate system to pixel coordinate system.
   */
  protected void transform() {
    xp = (int)((xw - xmin) * xfactor);
    yp = (int)((yw - ymin) * yfactor);
  }
}

