/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation;

import java.awt.Component;
import java.awt.Dimension;

import kandid.soup.ChromosomeType;
import kandid.colorator.*;
import kandid.util.*;

/**
 * Base class for calculations, using successive refinement, in world coordinate system.
 * @author thomas jourdan
 */
public abstract class RefinementCalculation extends PixelCalculation implements Cloneable {
  protected int    depth;
  protected double xwMin;
  protected double xwMax;
  protected double ywMin;
  protected double ywMax;
  protected int    xp;
  protected int    yp;
  protected double xw;
  protected double yw;
  protected int    xpDelta;
  protected int    ypDelta;
  protected double xwDelta;
  protected double ywDelta;
  protected int    actDepth;
  //!!  protected int    steps;
  protected int     xSteps;
  protected int     ySteps;
  protected boolean more;
  protected boolean ready;

//  /**
//   * Constructor for calculation, using successive refinement, based on world coordinates.
//   */
//  public void init(int xDim, int yDim) {
//    super.init(xDim, yDim);
//  }
//
  /* For documentation please see super class Calculation. */
  public void setChromosome(ChromosomeType chromosome, Colorator colorator) {
    super.setChromosome(chromosome, colorator);
  }

  /* For documentation please see super class Calculation. */
  public void activateCanvas(Component viewComponent, Dimension canvasSize, boolean zoomMode) {
    super.activateCanvas(viewComponent, canvasSize, zoomMode);
    ready = false;
    actDepth = depth;
    setDeltas();
  }
  
  /**
   * Go to a finer refinement.
   */
  protected boolean enterDepth() {
    if (actDepth > 0) {
      --actDepth;
      if (actDepth == 0) {
        setFineDeltas();
      }
      else {
        setDeltas();
      }
      more = true;
      return true;
    }
    ready = true;
    return false;
  }

  /**
   * Calculate the deltas, for the actual refinement.
   */
  private void setDeltas() {
    xpDelta = Util.powerOf2(actDepth);
    ypDelta = Util.powerOf2(actDepth);
    xSteps  = canvasSize.width / xpDelta;
    ySteps  = canvasSize.height / ypDelta;
    xwDelta = (xwMax - xwMin) / (double)(xSteps);
    ywDelta = (ywMax - ywMin) / (double)(ySteps);
    xp      = xpDelta / 2;
    yp      = xpDelta / 2;
    xw      = xwMin + (xwDelta / 2.0);
    yw      = ywMin + (ywDelta / 2.0);
  }

  /**
   * Calculate the deltas, for the last step of refinement.
    */
  private void setFineDeltas() {
    xpDelta = 1;
    ypDelta = 1;
    xSteps  = canvasSize.width;
    ySteps  = canvasSize.height;
    xwDelta = (xwMax - xwMin) / (double)(xSteps);
    ywDelta = (ywMax - ywMin) / (double)(ySteps);
    xp      = 0;
    yp      = 0;
    xw      = xwMin;
    yw      = ywMin;
  }

  /**
   * Sets the actual 'pixel' in to the buffered image.
   * In the last refinement step this is really a pixel.
   * Otherwise its an rectangle.
   *
   * @param color 24 Bit color value
   */
  protected void setPixel(ColorF32 colorF32) {
    try {
      int rgb8 = ColorF32.toRGB8(colorF32);
      if (actDepth >= 1) {
       for (int ypp = 0; ypp < ypDelta; ++ypp) {
         for (int xpp = 0; xpp < xpDelta; ++xpp) {
           bufAwtImg.setRGB(xp + xpp, yp + ypp, rgb8);
         }
       }
     }
     else {
       bufAwtImg.setRGB(xp, yp, rgb8);
       ColorF32.copy(colorF32, bufColorF32Img[yp][xp]);
     }
    } catch (ArrayIndexOutOfBoundsException exc) {
      if(Debug.enabled) System.out.println("bufImg coordinates out of bounds (" + xp + ", " + yp + ")");
    }
  }

  /**
   * Go to next position.
   * Calculates new position for world coordinate system and pixel coordinate system. 
   */
  protected void nextLocation() {
    if (actDepth == 0) {
      if ((xp + 1) < canvasSize.width) {
        ++xp;
        xw += xwDelta;
      }
      else {
        xp = 0;
        xw = xwMin;
        if ((yp + 1) < canvasSize.height) {
          ++yp;
          yw += ywDelta;
        }
        else {
          more = false;
        }
      }
    }
    else {
      if ((xp + xpDelta + xpDelta) < canvasSize.width) {
        xp += xpDelta;
        xw += xwDelta;
      }
      else {
        xp = xpDelta / 2;
        xw = xwMin + (xwDelta / 2.0);
        if ((yp + ypDelta + ypDelta) < canvasSize.height) {
          yp += ypDelta;
          yw += ywDelta;
        }
        else {
          more = false;
        }
      }
    }
  }

  /* For documentation please see super class Calculation. */
  public boolean getReady() {
    return aborted || ready;
  }

  /* For documentation please see super class Calculation. */
  public int getPercent() {
    // this getPercent() calculation is not accurate.
    if(canvasSize != null) {
      if (actDepth == 0)
        return (100*yp) / canvasSize.height;
    }
    return 0;
  }

  /* For documentation please see super class Calculation. */
  protected String paramString() {
    return " xp=" + xp + " yp=" + yp + " xw=" + xw + " yw=" + yw + " xpDelta=" + xpDelta + " ypDelta=" + ypDelta + " xwDelta=" + xwDelta + " ywDelta=" + ywDelta + " actDepth=" + actDepth + " xSteps=" + xSteps + " ySteps=" + ySteps;
  }

//  /* For documentation please see super class Calculation. */
//  public String toString() {
//    return getClass().getName() + "[" + super.paramString() + paramString() + "]";
//  }
}

