/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation;

import java.awt.*;

import kandid.soup.ChromosomeType;
import kandid.colorator.*;
import kandid.util.*;

/**
 * Class IterationCalculation
 * @author thomas jourdan
 */
public abstract class IterationCalculation extends WorldCalculation implements Cloneable {
  private int      actCount;
  private int      maxCount;
  private int      countState;
  protected ColorF32 pointColor;
  protected Weight weight;

  /* For documentation please see super class Calculation. */
  public void setChromosome(ChromosomeType chromosome, Colorator colorator) {
    super.setChromosome(chromosome, colorator);
    pointColor = ColorF32.black;
  }

  /* For documentation please see super class Calculation. */
  public void activateCanvas(Component viewComponent, Dimension canvasSize, boolean zoomMode) {
    super.activateCanvas(viewComponent, canvasSize, zoomMode);
    
    startIteration();
    findExtremeValues(7500);
    initTransformation(0.1);
    maxCount = actCount = getMaxIterations(canvasSize.width, canvasSize.height);
    countState = 0;

    Graphics gr = bufAwtImg.getGraphics();
    gr.setColor(hasWhiteBackground() ? Color.white : Color.black);
    gr.fillRect(0, 0, canvasSize.width, canvasSize.height);
  }
  
  /**
   * Method prepareWeight
   *
   * @param chromosome
   * @param colorator
   */
  protected void prepareWeight() {
//!!    prepareFlat(chromosome, colorator);
    weight = new Weight(canvasSize.width, canvasSize.height, colorator);
  }
  
  /**
   * Calculates number of runs for the next iteration.
   */
  private int nextRun(boolean paintOnScreen) {
  	if (paintOnScreen) {
  		int count = 25 * 25000;
  		switch (countState) {
  			case 0 :
  				count = 25000;
  				++countState;
  				break;
  			case 1 :
  				count = 5 * 25000;
  				++countState;
  				break;
  		}
  		if (count > actCount) {
  			count = actCount;
  			actCount = 0;
  		} else {
  			actCount -= count;
  		}
  		return count;
  	} else {
  		actCount = 0;
  		return maxCount;
  	}
  }

  /* For documentation please see super class Calculation. */
  public void calculate(Deviation dev, boolean paintOnScreen, String doExport) {
    int count = nextRun(paintOnScreen);
    while (count > 0) {
      iterate();
      setWPixel(pointColor);
      --count;
    }
  }

  /**
   * Do the calculation, and increment the counter array.
   */
  protected void calculateWeight(boolean paintOnScreen) {
    int count = nextRun(paintOnScreen);
    while (count > 0) {
      iterate();
      transform();
      weight.inc(xp, yp);
      --count;
    }
    weight.toImageLog(bufAwtImg, bufColorF32Img);
  }

  /* For documentation please see super class Calculation. */
  public boolean getReady() {
    return aborted || actCount <= 0;
  }

  /* For documentation please see super class Calculation. */
  public int getPercent() {
    return (100 * (maxCount - actCount)) / maxCount;
  }

  /**
   * Method findExtremeValues
   *
   * @param iterations
   */
  protected void findExtremeValues(int iterations) {
    xmin = Double.MAX_VALUE;
    xmax = Double.MIN_VALUE;
    ymin = Double.MAX_VALUE;
    ymax = Double.MIN_VALUE;
    while (--iterations > 0) {
      iterate();
      if (xw < xmin) {
        xmin = xw;
      }
      if (yw < ymin) {
        ymin = yw;
      }
      if (xw > xmax) {
        xmax = xw;
      }
      if (yw > ymax) {
        ymax = yw;
      }
    }
  }

  /**
   * the first values from this iteration are not valide.
   */
  protected  void startIteration() {
    xw = 0.5;
    yw = 0.5;
    for (int ix = 0; ix < 50; ++ix) {
      // the first values from this iteration are not valide
      iterate();
    }
  }

  /**
   * Calculates the next step for this iteration.
   * Results can be fond in member variable yw and xw.
   */
  protected abstract void iterate();

  /**
   * Number of iterations depends on coloring type.
   * @param width
   * @param height
   * @return int maximum number of iterations
   */
  protected abstract int getMaxIterations(int width, int height);
}

