/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid;

import java.io.*;
import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.util.*;

import kandid.pond.ui.IlPool;
import kandid.soup.*;
import kandid.soup.util.ChromosomeStamp;
import kandid.soup.util.SoupFactory;
import kandid.calculation.*;
import kandid.colorator.*;
import kandid.extensions.LRUImageCache;
import kandid.util.*;

public class PopulationController {
  private String populationFilename;
  private int waitReady = -1;
  private ChromosomeStamp calculationStamp;
  private ChromosomeStamp coloratorStamp;
  private OrganismPanel[] organismPanels;
  private int             nRows;
  private int             nCols;
  private int             nMax;
  private PopulationFrame populationFrame;
  private GridLayout      gridLayout = new GridLayout();
  private Environment     environment = new Environment();
  private int             sizeUndo;
  private Stack           undoStack = new Stack();
  private String          calculationName;
  private String          coloratorName;
  private SoupFactory     soupFactory;
  private PopulationType      population;
  private static final int maxRetries = 32;

  protected Color background;
  protected boolean scrollbar;
  /**
   * Constructor for PopulationController.
   */
  public PopulationController() {
    soupFactory = SoupFactory.getSoupFactory();
  }

  /**
   * create new random population
   *
   * @param typeName
   * @param calculationName
   * @param coloratorName
   * @param nRows
   * @param nCols
   * @param poolFrame
   */
  public void prepare(String typeName, String calculationName, String coloratorName, int nRows, int nCols, PopulationFrame poolFrame) {
    try {
      this.calculationName = calculationName;
      this.coloratorName = coloratorName;
      this.populationFrame = poolFrame;
      this.nRows = nRows;
      this.nCols = nCols;
      nMax = nRows * nCols;
      calculationStamp = new ChromosomeStamp(typeName);
      coloratorStamp = new ChromosomeStamp(ColoratorFactory.getShortName(coloratorName));
      population = soupFactory.createPopulation(typeName);
      for (int nx = 0; nx < nMax; ++nx) {
        soupFactory.createImage(population, nx, typeName, calculationName, coloratorName);
        soupFactory.createRandomChromosome(population, nx, calculationStamp);
        soupFactory.createRandomColorator(population, nx, coloratorName, coloratorStamp);
      }
      initGUI(calculationName);
    } catch (Throwable exc) {
      Debug.stackTrace(exc);
    }
  }

  /**
   * create population from file
   * 
   * @param typeName
   * @param calculationName
   * @param coloratorName
   * @param populationName
   * @param populationFrame
   */
  public void prepare(String typeName, String populationFilename, PopulationFrame populationFrame) {
    this.populationFrame = populationFrame;
    this.populationFilename = populationFilename;
    population = (PopulationType) soupFactory.unmarshalFromFile(populationFilename);
    calculationStamp = new ChromosomeStamp(typeName);
    calculationStamp.setPopulationIDfromName(populationFilename);
    coloratorStamp = new ChromosomeStamp(ColoratorFactory.getShortName(soupFactory.getColoratorName(population, 0)));
    coloratorStamp.setPopulationIDfromName(populationFilename);
    nMax = soupFactory.getSize(population);
    switch (nMax) {
      case 12 :
        nCols = 4;
        nRows = 3;
        break;
      case 24 :
        nCols = 6;
        nRows = 4;
        break;
      case 40 :
        nCols = 8;
        nRows = 5;
        break;
      default :
        nCols = 10;
        nRows = 7;
        nMax = 70;
    }
    for (int nx = 0; nx < nMax; ++nx) {
      calculationName = soupFactory.getCalculationName(population, nx);;
      coloratorName = soupFactory.getColoratorName(population, nx);
      ColoratorType coloratorType = soupFactory.getColorator(population, nx);
      Colorator coloratorWorker = null;
      if(coloratorName != null) {
        // create simple colorator
        coloratorWorker = ColoratorFactory.createColoratorWorker(coloratorName, coloratorType);
      }
      else {
        // create look up table colorator
        coloratorWorker = ColoratorFactory.createColoratorWorker(coloratorType.getColoratorName(), coloratorType);
      }
    }
    initGUI(calculationName);
    waitReady = nMax;
  }

  /**
   * clone whole population
   * @param populationController
   * @param populationFrame
   */
  public void prepare(PopulationController oldPopulationController, PopulationFrame oldPopulationFrame) {
    calculationName = oldPopulationController.calculationName;
    coloratorName = oldPopulationController.coloratorName;
    this.populationFrame = oldPopulationFrame;

    nRows = oldPopulationController.nRows;
    nCols = oldPopulationController.nCols;
    PopulationType oldPopulation = oldPopulationController.population;
    initModel(SoupFactory.getTypeName(oldPopulation), calculationName, coloratorName);
    population.setInheritCalculation(oldPopulation.isInheritCalculation());
    population.setInheritColorator(oldPopulation.isInheritColorator());
    initGUI(calculationName);
    for (int nx = 0; nx < nMax; ++nx) {
      // clone calculation and chromosome
      Calculation calculation = oldPopulationController.organismPanels[nx].getCalculation();
      ChromosomeType oldCalculationChromosome = soupFactory.getChromosome(oldPopulation, nx);
      ChromosomeType newCalculationChromosome = soupFactory.clone(oldCalculationChromosome, false, calculationStamp.getPopulationIdent());
      soupFactory.setChromosome(population, nx, newCalculationChromosome);
      // clone colorator and chromosome
      ColoratorType oldColorator = soupFactory.getColorator(oldPopulation, nx);
      ColoratorType newColorator = soupFactory.clone(oldColorator, false, coloratorStamp.getPopulationIdent());
      soupFactory.setColorator(population, nx, newColorator);
      soupFactory.setRating(population, nx, (Fitness)calculation.getFitness().clone());
      // cloned population should start with cached images.
      LRUImageCache.getLRUImageCache().duplicate(oldCalculationChromosome.getIdent(), newCalculationChromosome.getIdent());
      // do not lost original autorship information
      cloneAuthorship(oldPopulation, nx, nx);
    }
  }

  /**
   * @param oldPopulation
   * @param sourceIndex
   * @param destinationIndex
   * @param nx
   */
  private void cloneAuthorship(PopulationType oldPopulation, int sourceIndex, int destinationIndex) {
    ImageType oldImage = soupFactory.getImage(oldPopulation, sourceIndex);
    ImageType newImage = soupFactory.getImage(population, destinationIndex);
    newImage.setAuthor(oldImage.getAuthor());
    newImage.setPublishDate(oldImage.getPublishDate());
    newImage.setAnnotation(oldImage.getAnnotation());
  }

  /**
   * Method initModel.
   * @param typeName
   * @param coloratorName
   */
  private void initModel(String typeName, String calculationName, String coloratorName) {
    nMax = nRows * nCols;
    calculationStamp = new ChromosomeStamp(typeName);
    coloratorStamp = new ChromosomeStamp(ColoratorFactory.getShortName(coloratorName));
    population = soupFactory.createPopulation(typeName);
    for (int nx = 0; nx < nMax; ++nx) {
      soupFactory.createImage(population, nx, typeName, calculationName, coloratorName);
    }
  }

  /**
   * Method initGUI.
   */
  private void initGUI(String calculationName) {
    hintGUI(calculationName);
    
    gridLayout.setColumns(nCols);
    gridLayout.setRows(nRows);
    populationFrame.populationPanel.setLayout(gridLayout);
    organismPanels = new OrganismPanel[nMax];
    for (int nx = 0; nx < nMax; ++nx) {
      OrganismPanel organismPanel = new OrganismPanel(false, nRows, nCols, scrollbar, this, nx);
      organismPanel.adjustPanelSize();
      organismPanels[nx] = organismPanel;
      populationFrame.populationPanel.add(organismPanel, null);
    }
    populationFrame.setTitle("PopulationType: " + getTitle());
    boolean simpleColorator = ColoratorFactory.isSimpleColorator(soupFactory.getColoratorName(population, 0));
    populationFrame.setInheritCalculation(population.isInheritCalculation(), simpleColorator);
    populationFrame.setInheritColorator(population.isInheritColorator(), simpleColorator);
  }

  private void hintGUI(String calculationName) {
    background = calculationName.indexOf("kandid.calculation.bridge.") == 0 ? Color.BLACK : null;
    scrollbar = calculationName.indexOf("kandid.calculation.lca.") == 0;
  }

  /**
   * start single
   *
   * @param index
   */
  private void start(boolean createNew, int index) {
    Fitness fitness = null;
    if (createNew) {
      fitness = new Fitness();
      fitness.abort = true;
    }
    else {
      fitness = soupFactory.mapRating(population, index);
    }
    organismPanels[index].start(population, index, fitness);
  }

  /**
   * start all
   */
  public void start(boolean createNew) {
    for (int nx = 0; nx < nMax; ++nx) {
      start(createNew, nx);
    }
    toggleButtons();
  }

  /**
   * stop single
   *
   * @param index
   */
  private void stop(int index) {
    organismPanels[index].stop();
  }

  /**
   * stop all
   */
  public void stop() {
    for (int nx = 0; nx < nMax; ++nx) {
      stop(nx);
    }
  }

  /**
   * stop all
   */
  public void quit() {
    for (int nx = 0; nx < nMax; ++nx) {
      stop(nx);
      //!!      organismPanels[nx].quit();
    }
  }

  /**
   * restart
   * @param index
   */
  public void apply(int index) {
    stop(index);
    start(false, index);
  }

  /**
   * store population.
   */
  public void store() {
    // set rating
    for (int nx = 0; nx < nMax; ++nx) {
      Fitness fitness = organismPanels[nx].getCalculation().getFitness();
      soupFactory.setRating(population, nx, fitness);
    }
    // store population
    populationFilename = getTitle() + "_";
    kandid.extensions.ExportState exportState = new kandid.extensions.ExportState();
    exportState.setFullname(populationFilename);
    (new File(exportState.getPath(Kandid.localImageFolder))).mkdirs();
    populationFilename = exportState.getPath(Kandid.localImageFolder) + populationFilename + getUniqStoreID(exportState.getPath(Kandid.localImageFolder)) +  ".kpop";
    soupFactory.marshalToFile(population, populationFilename);
    // store preview
    storePreview(populationFilename, "png");
    // touch timstamp
    for (int nx = 0; nx < nMax; ++nx) {
      ChromosomeType calculationChromosome = soupFactory.getChromosome(population, nx);      
      LRUImageCache.getLRUImageCache().touch(calculationChromosome.getIdent(), organismPanels[nx].getViewEngine().getDim());
    }
  }

  /**
   * store preview for this population
   */
  private void storePreview(String filename, String imagetype) {
    try {
      int populationPreviewDimension = PopulationPreview.populationPreviweWidth / nCols;
      BufferedImage populationPreviewImg = new BufferedImage(nCols*populationPreviewDimension, nRows*populationPreviewDimension, BufferedImage.TYPE_INT_RGB);
      int nx = 0;    
      for (int rx = 0; rx < nRows; ++rx) {
        for (int cx = 0; cx < nCols; ++cx) {
          ViewEngine viewEngine = organismPanels[nx].getViewEngine();
          Dimension calculatedDim = viewEngine.getGateway().getCalculation().getPreferredSize();
          if (calculatedDim == null) {
            calculatedDim = viewEngine.getDim();
          }
          Image calculatedImage = viewEngine.getGateway().getImage();
          Graphics2D g2d = populationPreviewImg.createGraphics();
          AffineTransform scale = new AffineTransform();
          scale.setToIdentity();
//          scale.setToScale(calculatedDim.getWidth() / populationPreviewDimension,
//                           calculatedDim.getHeight() / populationPreviewDimension);
          scale.translate(cx*populationPreviewDimension, rx*populationPreviewDimension);
          scale.scale(populationPreviewDimension / calculatedDim.getWidth(),
                      populationPreviewDimension /calculatedDim.getHeight());
          g2d.drawImage(calculatedImage, scale, null);
          ++nx;
        }
      }
      File imageOutFile = new File(filename + "." + imagetype);
      javax.imageio.ImageIO.write(populationPreviewImg, imagetype, imageOutFile);
    }
    catch (IOException exc) {
      Debug.stackTrace(exc);
    }
  }

  private String getUniqStoreID(String path) {
    File currDir = new File(path);
    String[] fileList = currDir.list(new kandid.extensions.KandidFilenameFilter(SoupFactory.getTypeName(population), ".kpop"));
    if (fileList != null && fileList.length > 0) {
      Arrays.sort(fileList);
      String lastEntry = fileList[fileList.length-1];
      int begin = lastEntry.lastIndexOf('_');
      int end = lastEntry.lastIndexOf('.');
      String sCount = lastEntry.substring(begin+1, end);
      int count = Integer.parseInt(sCount);
      return ""+(count+1);
    }
    return "1";
  }
  
  /**
   * command for single image
   *
   * @param cmd
   * @param to
   */
  private void command(int cmd, int to) {
    organismPanels[to].command(cmd);
  }

  /**
   * command for all images, except calling image
   *
   * @param cmd
   * @param from
   */
  private void commandAll(int cmd, int from) {
    for (int nx = 0; nx < nMax; ++nx) {
      if (nx != from) {
        organismPanels[nx].command(cmd);
      }
    }
  }

  /**
   * find partner for me
   *
   * @param me
   *
   * @return
   */
  private int findPartner(int me) {
    int partner = -1;
    int partnerList[] = new int[nMax];
    int pi = 0;
    if (Debug.enabled)
      assert (me >= 0) && (me < nMax) : "me is out of range: " + me;
    Fitness myf = organismPanels[me].getCalculation().getFitness();
    if (Debug.enabled)
      assert myf.best || myf.parent1 : "my fitness is not best or parent1";
    if (myf.best) {
      for (int nx = 0; nx < nMax; ++nx) {
        if (nx != me) {
          Fitness pf = organismPanels[nx].getCalculation().getFitness();
          if (!pf.abort) {
            partnerList[pi] = nx;
            ++pi;
          }
        }
      }
    } else if (myf.parent1) {
      for (int nx = 0; nx < nMax; ++nx) {
        if (nx != me) {
          Fitness pf = organismPanels[nx].getCalculation().getFitness();
          if (pf.parent2) {
            partnerList[0] = nx;
            pi = 1;
            break;
          }
        }
      }
    }

    switch (pi) {
      case 0 :
        break;
      case 1 :
        partner = partnerList[0];
        break;
      default :
        partner = partnerList[CentralRandomizer.getInt(pi - 1)];
    }
    if (Debug.enabled)
      assert partner != me : "wrong partner";
    if (Debug.enabled)
      assert (partner == -1) || !organismPanels[partner].getCalculation().getFitness().abort : "wrong partner";
    return partner;
  }

  /**
   * two partners
   */
  public void sexualReproduction() {
    int first = -1;
    int second = -1;
    for (int nx = 0; nx < nMax; ++nx) {
      Fitness ff = organismPanels[nx].getCalculation().getFitness();
      if (ff.best || ff.parent1) {
        first = nx;
        break;
      }
    }

    if (first >= 0) {
      undoPush();
      for (int nx = 0; nx < nMax; ++nx) {
        Fitness pf = organismPanels[nx].getCalculation().getFitness();
        if (pf.abort) {
          try {
            // try to stop old calculations
            organismPanels[nx].getCalculation().setAborted(true);
//            // remove from cache
//            ChromosomeType chromosome = soupFactory.getChromosome(population, nx);
//            LRUImageCache.getLRUImageCache().removeAll(chromosome.getIdent());
          } catch (Throwable exc) {
            Debug.stackTrace(exc);
          }
        }
      }
      // prepare and start new calculations
      for (int nx = 0; nx < nMax; ++nx) {
        second = findPartner(first);
        if (first >= 0 && first != second) {
//          ChromosomeType chromosome1 = organismPanels[first].getCalculation().getChromosome();
//          ChromosomeType chromosome2 = organismPanels[second].getCalculation().getChromosome();
//          Colorator colorator1 = organismPanels[first].getCalculation().getColorator();
//          Colorator colorator2 = organismPanels[second].getCalculation().getColorator();
          Fitness pf = organismPanels[nx].getCalculation().getFitness();
          if (pf.abort) {
            try {
              ChromosomeType calculationChromosome1 = soupFactory.getChromosome(population, first);
              ChromosomeType calculationChromosome2 = soupFactory.getChromosome(population, second);
              int retryCount = 0;
              while (retryCount < maxRetries) {
                ++retryCount;
                int mutationCounter = 0;
                // reproduce calculation chromosome
                ChromosomeType calculationChromosome = null;
                if(population.isInheritCalculation()) {
                  calculationChromosome = soupFactory.merge(calculationChromosome1, calculationChromosome2, environment);
                  mutationCounter += soupFactory.mutate(calculationChromosome, environment);
                }
                else {
                  calculationChromosome = soupFactory.getChromosome(population, nx);
                }
                calculationChromosome.setIdent(ChromosomeStamp.forceNextChromosomeId(calculationChromosome.getIdent()));
                soupFactory.setChromosome(population, nx, calculationChromosome);
                // reproduce colorator chromosome
                ColoratorType colorator1 = soupFactory.getColorator(population, first);
                if(colorator1 != null) {
                  ChromosomeType coloratorChromosome = null;
                  if(population.isInheritColorator()) {
                    ChromosomeType coloratorChromosome1 = soupFactory.getColorTable(colorator1);
                    ColoratorType colorator2 = soupFactory.getColorator(population, second);
                    ChromosomeType coloratorChromosome2 = soupFactory.getColorTable(colorator2);
                    coloratorChromosome = soupFactory.merge(coloratorChromosome1, coloratorChromosome2, environment);
                    mutationCounter += soupFactory.mutate(coloratorChromosome, environment);
                  }
                  else {
                    coloratorChromosome = soupFactory.getColorTable(soupFactory.getColorator(population, nx));
                  }
                  ColoratorType newColorator = soupFactory.clone(colorator1, true, null);
                  soupFactory.setColorTable(newColorator, coloratorChromosome);
                  soupFactory.setColorator(population, nx, newColorator);
                }
                
                boolean successful = !organismPanels[nx].getCalculation().reject(calculationChromosome);
                if (successful || retryCount >= maxRetries) {
                  start(true, nx);
                  organismPanels[nx].toggleButtons();
                  Console.append("sexual reproduction " + calculationChromosome.getIdent() + " : " + mutationCounter + " mutations and " + environment.getSwitchCounter() + " crossing over, " + "parents " + calculationChromosome1.getIdent() + " and " + calculationChromosome2.getIdent() + ", " + retryCount + " reties, " + (successful ? "successful" : "failed"));
                  if (Debug.enabled) System.out.println(retryCount + " reties");
                  break;
                }
              }
            } catch (Throwable exc) {
              Debug.stackTrace(exc);
            }
          }
        }
      } // end for
    }
    toggleButtons();
    waitReady = -1;
    if (Debug.enabled) System.out.println(Debug.memoryText());
  }

  /**
   * clone
   */
  public void cloneReproduction() {
    int first = findBest();
    if (first >= 0) {
      undoPush();
      ChromosomeType calculationChromosome1 = soupFactory.getChromosome(population, first);
      for (int nx = 0; nx < nMax; ++nx) {
        Fitness pf = organismPanels[nx].getCalculation().getFitness();
        if (pf.abort) {
          try {
            // try to stop old calculations
            organismPanels[nx].getCalculation().setAborted(true);
          } catch (Throwable exc) {
            Debug.stackTrace(exc);
          }
        }
      }
      // prepare and start new calculations
      for (int nx = 0; nx < nMax; ++nx) {
        Fitness pf = organismPanels[nx].getCalculation().getFitness();
        if (pf.abort) {
          try {
            // clone calculation chromosome
            ChromosomeType calculationChromosome = null;
            if(population.isInheritCalculation()) {
              calculationChromosome = soupFactory.clone(calculationChromosome1, true, null);
            }
            else {
              calculationChromosome = soupFactory.getChromosome(population, nx);
            }
            calculationChromosome.setIdent(ChromosomeStamp.forceNextChromosomeId(calculationChromosome.getIdent()));
            soupFactory.setChromosome(population, nx, calculationChromosome);
            // clone colorator and chromosome
            ColoratorType colorator1 = soupFactory.getColorator(population, first);
            if(colorator1 != null) {
              ChromosomeType coloratorChromosome = null;
              if(population.isInheritColorator()) {
                coloratorChromosome = soupFactory.clone(soupFactory.getColorTable(colorator1), true, null);
              }
              else {
                coloratorChromosome = soupFactory.getColorTable(soupFactory.getColorator(population, nx));
              }
              ColoratorType newColorator = soupFactory.clone(colorator1, true, null);
              soupFactory.setColorTable(newColorator, coloratorChromosome);
              soupFactory.setColorTable(newColorator, coloratorChromosome);
              soupFactory.setColorator(population, nx, newColorator);
            }
            
            cloneAuthorship(population, nx, nx);
            
            start(true, nx);
            command(Calculation.cmdSetAbort, nx);
            organismPanels[nx].toggleButtons();
            Console.append("clone " + calculationChromosome.getIdent() + " : parent " + calculationChromosome1.getIdent());
          } catch (Throwable exc) {
            Debug.stackTrace(exc);
          }
        }
      }
      toggleButtons();
    }
    waitReady = -1;
    if (Debug.enabled) System.out.println(Debug.memoryText());
  }

  /**
   * generates a new random chromosome
   */
  public void randomReproduction(boolean createNew) {
    if(!createNew)
      undoPush();
    // try to stop old calculations
    for (int nx = 0; nx < nMax; ++nx) {
      if (isPanelFree(createNew, nx)) {
        try {
          Calculation calc = organismPanels[nx].getCalculation();
          if(calc != null)
            calc.setAborted(true);
//          // remove from cache
//          ChromosomeType chromosome = soupFactory.getChromosome(population, nx);
//          LRUImageCache.getLRUImageCache().removeAll(chromosome.getIdent());
        } catch (Throwable exc) {
          Debug.stackTrace(exc);
        }
      }
    }
    // prepare and start new calculations
    for (int nx = 0; nx < nMax; ++nx) {
      // check if the organism panel is free
      if (isPanelFree(createNew, nx)) {
        int retryCount = 0;
        while (retryCount < maxRetries) {
          ++retryCount;
          // clone calculation chromosome
          ChromosomeType calculationChromosome = null;
          if(population.isInheritCalculation()) {
            calculationChromosome = soupFactory.createRandomChromosome(population, nx, calculationStamp);
          }
          else {
            calculationChromosome = soupFactory.getChromosome(population, nx);
          }
          calculationChromosome.setIdent(ChromosomeStamp.forceNextChromosomeId(calculationChromosome.getIdent()));
          soupFactory.setChromosome(population, nx, calculationChromosome);
          // clone colorator and chromosome
          ColoratorType colorator1 = soupFactory.getColorator(population, nx);
          if(colorator1 != null) {
            ColoratorType colorator = null;
            if(population.isInheritColorator()) {
              colorator = soupFactory.createRandomColorator(population, nx, colorator1.getColoratorName(), coloratorStamp);
            }
            else {
              colorator = soupFactory.getColorator(population, nx);
            }
            soupFactory.setColorator(population, nx, colorator);
          }
          
//!!          boolean successful = !organismPanels[nx].getCalculation().reject(calculationChromosome);
//!!          boolean successful = createNew || !organismPanels[nx].getCalculation().reject(calculationChromosome);
          boolean fail = (new Gateway()).createCalculation(calculationName).reject(calculationChromosome);
          if (!fail || retryCount >= maxRetries) {
//!!            organismPanels[nx].restart(calculationChromosome);
            start(true, nx);
            organismPanels[nx].toggleButtons();
            Console.append("new random " + calculationChromosome.getIdent() + ", " + retryCount + " reties, " + (fail ? "failed" : "successful"));
            if (Debug.enabled) System.out.println(retryCount + " reties");
              break;
          }
        }
      }
    }
    toggleButtons();
    waitReady = -1;
    if (Debug.enabled) System.out.println(Debug.memoryText());
  }

  private boolean isPanelFree(boolean createNew, int nx) {
    boolean isPanelFree = false;
    if(createNew) {
      isPanelFree = true;
    }
    else {
      Fitness pf = organismPanels[nx].getCalculation().getFitness();
      isPanelFree = pf.abort;
    }
    return isPanelFree;
  }

  /**
   * find image with best maker set
   *
   * @return
   */
  public int findBest() {
    int first = -1;
    for (int nx = 0; nx < nMax; ++nx) {
      Fitness ff = organismPanels[nx].getCalculation().getFitness();
      if (ff.best) {
        first = nx;
        break;
      }
    }

    return first;
  }

  /**
   * number of free locations
   *
   * @return
   */
  public int getFreeLocations() {
    int abortCount = 0;
    for (int nx = 0; nx < nMax; ++nx) {
      Fitness pf = organismPanels[nx].getCalculation().getFitness();
      if (pf.abort) {
        ++abortCount;
      }
    }

    return abortCount;
  }

  /**
   * number of not marked locations
   *
   * @return
   */
  public int getUnmarkedLocations() {
    int nomarkCount = 0;
    for (int nx = 0; nx < nMax; ++nx) {
      Fitness pf = organismPanels[nx].getCalculation().getFitness();
      if (!pf.abort && !pf.best && !pf.parent1 && !pf.parent1) {
        ++nomarkCount;
      }
    }

    return nomarkCount;
  }

  /**
   * Method hasParents
   *
   * @return
   */
  public boolean hasParents() {
    int bestCount = 0;
    int parent1Count = 0;
    int parent2Count = 0;
    for (int nx = 0; nx < nMax; ++nx) {
      Fitness pf = organismPanels[nx].getCalculation().getFitness();
      if (pf.best) {
        ++bestCount;
      }
      if (pf.parent1) {
        ++parent1Count;
      }
      if (pf.parent2) {
        ++parent2Count;
      }
    }

    return (bestCount >= 1) || ((parent1Count >= 1) && (parent2Count >= 1));
  }

  /**
   * Method hasBest
   *
   * @return
   */
  public boolean hasBest() {
    int bestCount = 0;
    for (int nx = 0; nx < nMax; ++nx) {
      Fitness pf = organismPanels[nx].getCalculation().getFitness();
      if (pf.best) {
        ++bestCount;
      }
    }

    return bestCount >= 1;
  }

  /**
   * Method canCreate
   *
   * @return
   */
  public boolean canCreate() {
    return (getFreeLocations() > 0) && hasParents() && (getUnmarkedLocations() > 0);
  }

  /**
   * Method canClone
   *
   * @return
   */
  public boolean canClone() {
    return (getFreeLocations() > 0) && hasBest();
  }

  /**
   * Method toggleButtons
   */
  private void toggleButtons() {
    populationFrame.sexualReproductionButton.setEnabled(canCreate());
    populationFrame.cloneButton.setEnabled(canClone());
    populationFrame.randomButton.setEnabled(getFreeLocations() > 0);
    populationFrame.undoButton.setEnabled(undoStack.size() >= 2);
    for (int nx = 0; nx < nMax; ++nx) {
      organismPanels[nx].toggleButtons();
    }
  }

  /**
   * Method dispatchBest
   *
   * @param caller
   * @param isSelected
   */
  public void dispatchBest(int caller, boolean isSelected) {
    if (isSelected) {
      commandAll(Calculation.cmdResetParent, caller);
      command(Calculation.cmdSetBest, caller);
    } else {
      command(Calculation.cmdResetBest, caller);
    }
    toggleButtons();
  }

  /**
   * Method dispatchParent1
   *
   * @param caller
   * @param isSelected
   */
  public void dispatchParent1(int caller, boolean isSelected) {
    if (isSelected) {
      commandAll(Calculation.cmdResetBest, caller);
      commandAll(Calculation.cmdResetParent1, caller);
      command(Calculation.cmdSetParent1, caller);
    } else {
      command(Calculation.cmdResetParent1, caller);
    }
    toggleButtons();
  }

  /**
   * Method dispatchParent2
   *
   * @param caller
   * @param isSelected
   */
  public void dispatchParent2(int caller, boolean isSelected) {
    if (isSelected) {
      commandAll(Calculation.cmdResetBest, caller);
      commandAll(Calculation.cmdResetParent2, caller);
      command(Calculation.cmdSetParent2, caller);
    } else {
      command(Calculation.cmdResetParent2, caller);
    }
    toggleButtons();
  }

  /**
   * Method dispatchAbort
   *
   * @param caller
   * @param isSelected
   */
  public void dispatchAbort(int caller, boolean isSelected) {
    if (isSelected) {
      command(Calculation.cmdSetAbort, caller);
    } else {
      command(Calculation.cmdResetAbort, caller);
    }
    toggleButtons();
  }

  /**
   * Method undoPush
   */
  public void undo() {
    if (undoStack.size() >= 2) {
      undoPop();
    }
    toggleButtons();
  }

  /**
   * Method undoPush
   */
  private void undoPush() {
    // limit size of undo stack
    if (sizeUndo >= 10) {
      for (int nx = 0; nx < nMax; ++nx) {
        undoStack.removeElementAt(0);
        undoStack.removeElementAt(0);
      }
      --sizeUndo;
    }
    // push all to undo stack
    for (int nx = 0; nx < nMax; ++nx) {
      ChromosomeType chromosome = soupFactory.getChromosome(population, nx);
      undoStack.push(soupFactory.clone(chromosome, false, null));
      ColoratorType colorator = soupFactory.getColorator(population, nx);
      undoStack.push(soupFactory.clone(colorator, false, null));
      Object fitness = organismPanels[nx].getCalculation().getFitness().clone();
      undoStack.push(fitness);
    }
    ++sizeUndo;
  }

  /**
   * Method undoPop
   */
  private void undoPop() {
    // pop all from undo stack
    int nx = nMax;
    while (nx > 0) {
      --nx;
      Fitness fitness = (Fitness) undoStack.pop();
      organismPanels[nx].getCalculation().setFitness(fitness);
      soupFactory.setRating(population, nx, fitness);
      ColoratorType colorator = (ColoratorType) undoStack.pop();
      soupFactory.setColorator(population, nx, colorator);
      ChromosomeType chromosome = (ChromosomeType) undoStack.pop();
      soupFactory.setChromosome(population, nx, chromosome);
      organismPanels[nx].start(population, nx, fitness);
    }
    --sizeUndo;
  }

  /**
   * Method editEnvironment.
   */
  public void editEnvironment() {
    EnvironmentEditor dlg = new EnvironmentEditor(populationFrame, true);
    dlg.setEnvironment(environment);
    dlg.setLocationRelativeTo(populationFrame);
    dlg.setVisible(true);
    environment = dlg.getEnvironment();
    Console.append("mutation rate " + environment.getMutationRate() + ", crossing over rate " + environment.getMergeRate());
  }

  /**
   * Method getPopulation.
   */
  public PopulationType getPopulation() {
    return population;
  }

  /**
   * Method getTitle.
   * @param index
   * @return String
   */
  public String getTitle(int index) {
    String shortColoratorName = ColoratorFactory.getShortName(soupFactory.getColoratorName(population, index));
    String ident = soupFactory.getChromosome(population, index).getIdent();
    int pos = ident.lastIndexOf('_');
    if(pos >= 0) {
      ident = ident.substring(pos);
    }
    return calculationStamp.getTypePopulationIdent(shortColoratorName) + ident;
  }
  public String getTitle() {
    String shortColoratorName = ColoratorFactory.getShortName(soupFactory.getColoratorName(population, 0));
    return calculationStamp.getTypePopulationIdent(shortColoratorName);
  }

  /**
   * Call back to signal that a calculation is ready
   * @param nx number of organism panel 
   */
  public void notifyReady(int nx) {
    if(waitReady >= 0) {
      --waitReady;
      if(waitReady == 0) {
        // update preview
        storePreview(populationFilename, "png");
      }
    }
  }

  public void windowActivated() {
    IlPool.getInstance().navigateToType(calculationName, coloratorName);
  }

}
