/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid;

import java.awt.datatransfer.*;
import javax.swing.*;

import kandid.extensions.LRUImageCache;
import kandid.pond.ui.IlPool;
import kandid.soup.ChromosomeType;
import kandid.soup.EntityType;
import kandid.soup.ImageType;
import kandid.soup.PopulationType;
import kandid.soup.RatingBase;
import kandid.soup.util.ChromosomeStamp;
import kandid.soup.util.SoupFactory;
import kandid.util.Debug;

import java.io.IOException;

public class ImageTransferHandler extends TransferHandler {
  private static final long serialVersionUID = 1L;
  private boolean sendToPool;

  public ImageTransferHandler(boolean sendToPool) {
    this.sendToPool = sendToPool;
  }

  protected Transferable createTransferable(JComponent c) {
    return new StringSelection(exportString(c));
  }

  public int getSourceActions(JComponent c) {
    return COPY;
  }

  public boolean importData(JComponent c, Transferable t) {
    if (canImport(c, t.getTransferDataFlavors())) {
      try {
        String str = (String) t.getTransferData(DataFlavor.stringFlavor);
        return importString(c, str);
      } catch (UnsupportedFlavorException ufe) {
        Debug.stackTrace(ufe);
      } catch (IOException ioe) {
        Debug.stackTrace(ioe);
      }
    }

    return false;
  }

  protected void exportDone(JComponent c, Transferable data, int action) {
  }

  public boolean canImport(JComponent c, DataFlavor[] flavors) {
    for (int i = 0; i < flavors.length; i++) {
      if (DataFlavor.stringFlavor.equals(flavors[i])) {
        if (c != null) {
          if(sendToPool)
            return true;
          if (c instanceof ViewEngine) {
            ViewEngine viewEngine = (ViewEngine) c;
            if (sendToPool || viewEngine.getCalculation().getFitness().abort) {
              if (!viewEngine.isZoomMode()) {
                if (Debug.enabled) System.out.println("canImport " + true);
                return true;
              }
            }
          }
        }
      }
    }
    if (Debug.enabled)
      System.out.println("canImport " + false);
    return false;
  }

  /**
   * Exports the images data as XML. Root element is entity.
   * @param c
   * @return
   */
  protected String exportString(JComponent c) {
    if (c != null && c instanceof ViewEngine) {
      ViewEngine viewEngine = (ViewEngine) c;
      SoupFactory soupFactory = SoupFactory.getSoupFactory();
      EntityType entity = null;
      if(viewEngine.getOrganismPanel() != null) {
        int id = viewEngine.getOrganismPanel().getPopulationID();
        PopulationType population = viewEngine.getOrganismPanel().getPopulationController().getPopulation();
        ImageType image = soupFactory.getImage(population, id);
//  !!      ImageType image = viewEngine.getImage();
        entity = soupFactory.createEntity(image);
      }
      else if(viewEngine.getPoolController() != null) {
        IlPool ilPool = viewEngine.getPoolController();
        entity = ilPool.getEntity(viewEngine);
      }
      return soupFactory.marshalToString(entity);
    }
    return "";
  }

  protected boolean importString(JComponent c, String stringRepresentation) {
    if (c != null) {
      SoupFactory soupFactory = SoupFactory.getSoupFactory();
      if (Debug.enabled)
        System.out.println(stringRepresentation);
      Object input = soupFactory.unmarshalFromString(stringRepresentation);
      if (input != null && input instanceof EntityType) {
        EntityType entity = (EntityType) input;
        // get type name of source
        String sourceTypeName = SoupFactory.getTypeName(entity);
        if (Debug.enabled)
          System.out.println("sourceTypeName      " + sourceTypeName);
        if (sendToPool) {
          if (Debug.enabled)
            System.out.println("sent to pool");
          ImageType image = soupFactory.getImage(entity);
          image.setRating(RatingBase.NONE);
          IlPool.getInstance().addImage(image);
          return true;
        } else {
          if (c instanceof ViewEngine) {
            ViewEngine viewEngine = (ViewEngine) c;
            PopulationType population = viewEngine.getOrganismPanel().getPopulationController().getPopulation();
            // get type name of destination population
            String destinationTypeName = SoupFactory.getTypeName(population);
            if (Debug.enabled)
              System.out.println("destinationTypeName " + destinationTypeName);
            if (sourceTypeName.equals(destinationTypeName)) {
              ImageType image = soupFactory.getImage(entity);
              image.setRating(RatingBase.NONE);
              // generate new chromosome id
              ChromosomeType chromosome = soupFactory.getChromosome(entity);
              String oldId = chromosome.getIdent();
              // TODO WEB database
//            HttpPostConnection.updateUsecount(image.getInstallationId(), oldId);
              String newId = ChromosomeStamp.forceNextChromosomeId(chromosome.getIdent());
              chromosome.setIdent(newId);
              // inform cache about new chromsome id
              LRUImageCache.getLRUImageCache().duplicate(oldId, newId);
              // inset image to population
              int ix = viewEngine.getOrganismPanel().getPopulationID();
              soupFactory.replaceImage(population, ix, image);
              viewEngine.getOrganismPanel().getPopulationController().apply(ix);
              return true;
            }
          }
        }
      }
    }
    return false;
  }

}
