/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid;

import java.awt.*;
import java.awt.image.*;
import java.awt.geom.*;
import javax.swing.*;

import kandid.soup.ChromosomeType;
import kandid.util.*;
import kandid.calculation.*;
import kandid.calculation.bridge.PixelBridgeCalculation;
import kandid.colorator.*;
import kandid.extensions.*;

/**
 * Class Gateway
 * 
 * @author thomas jourdan
 */
public class Gateway {
  public static final int SCREEN = 1;
  public static final int BITMAP_FILE = 3;
  public static final int GENOMEDB = 6;

  private static LRUImageCache lruImageCache = LRUImageCache.getLRUImageCache();

  private Dimension canvasSize;
  private Calculation calc;
  private ChromosomeType chromosome;
  private Component viewComponent;
  private Image image;
  private Image cachedImage;
  private Graphics2D g2d;
  private int outputType;

  public void activateScreenCanvas(Dimension canvasSize, Component viewComponent, boolean zoomMode) {
    if (Debug.enabled)
      assert viewComponent != null : "viewComponent is undefined";
    activateCanvas(SCREEN, canvasSize, viewComponent, null, zoomMode);
  }

  public void activateBitmapCanvas(Dimension canvasSize) {
    activateCanvas(BITMAP_FILE, canvasSize, Kandid.getFrame(), null, true);
  }

  private void activateCanvas(int outputType, Dimension canvasSize, Component viewComponent, Graphics2D setG2d, boolean zoomMode) {
    if (Debug.enabled) {
      assert canvasSize.width > 0 : "canvasSize.width must be a positive integer";
      assert canvasSize.height > 0 : "canvasSize.height must be a positive integer";
    }
    this.outputType = outputType;
    this.canvasSize = canvasSize;
    this.viewComponent = viewComponent;
    if(outputType == BITMAP_FILE) {
      // force recalculate on export, ignore cache
      image = cachedImage = null;
    }
    else {
      image = cachedImage = lruImageCache.get(chromosome.getIdent(), canvasSize);
    }

    if (cachedImage == null) {
      if (calc instanceof kandid.calculation.VectorCalculation) {
        initVectorGraphics(setG2d);
        ((kandid.calculation.VectorCalculation) calc).activateCanvas(viewComponent, canvasSize, g2d, zoomMode);
      } else if (calc instanceof kandid.calculation.PixelCalculation) {
        initBitmapGraphics(setG2d);
        ((kandid.calculation.PixelCalculation) calc).activateCanvas(viewComponent, canvasSize, zoomMode);
        image = ((PixelCalculation) calc).getImage();
        if (Debug.enabled)
          assert image != null : "image is null";
      } else {
        if (Debug.enabled)
          assert false;
      }
    } else {
      if (Debug.enabled)
        System.out.println("from cache " + chromosome.getIdent() + " size " + canvasSize);
    }
  }

  public void setChromosome(ChromosomeType chromosome, Calculation calculation, Colorator colorator) {
    if (Debug.enabled)
      assert chromosome != null : "chromosome is undefined";
    if (Debug.enabled)
      assert calculation != null : "calculation is undefined";
    this.calc = calculation;
    this.calc.setChromosome(chromosome, colorator);
    this.chromosome = chromosome;
  }

  /**
   * Method createCalculation
   *
   * @param calculationName
   *
   * @return
   */
  public Calculation createCalculation(String calculationName) {
    try {
      Class classDefinition = Class.forName(calculationName);
      Calculation calculation = (Calculation) classDefinition.getDeclaredConstructor().newInstance();
//!!      calculation.setGatway(this);
      return calculation;
    } catch (Throwable exc) {
      Debug.stackTrace(exc);
    }
    if (Debug.enabled)
      assert false : "No appropriate calculation for class '" + calculationName + "'.";
    return null;
  }

  /**
   * Method initBitmapGraphics
   *
   * @param setG2d
   */
  private void initBitmapGraphics(Graphics2D setG2d) {
    if (outputType == SCREEN) {
    } else if (outputType == BITMAP_FILE) {
    } else {
      if (Debug.enabled)
        assert false : "outputType is undefined";
    }
  }

  /**
   * Method initVectorGraphics
   *
   * @param setG2d
   */
  private void initVectorGraphics(Graphics2D setG2d) {
    if (outputType == SCREEN) {
      image = viewComponent.createImage(canvasSize.width, canvasSize.height);
      if (Debug.enabled)
        assert image != null : "image is null";
      g2d = (Graphics2D) image.getGraphics();
    } else if (outputType == BITMAP_FILE) {
      image = new BufferedImage(canvasSize.width, canvasSize.height, BufferedImage.TYPE_INT_RGB);
      if (Debug.enabled)
        assert image != null : "image is null";
      g2d = (Graphics2D) image.getGraphics();
    } else {
      if (Debug.enabled)
        assert false : "outputType is undefined";
    }
  }

  /**
   * Method getImage
   *
   * @return
   */
  public Image getImage() {
    if (cachedImage != null) {
      return cachedImage;
    }
    if (calc instanceof PixelBridgeCalculation) {
      PixelBridgeCalculation pixelBridgeCalculation = (PixelBridgeCalculation) calc;
      image = pixelBridgeCalculation.getImage();
    }
    return image;
  }

  /**
   * Method getGraphics
   *
   * @return
   */
  public Graphics2D getGraphics() {
    if (Debug.enabled)
      assert g2d != null : "Graphics2D is undefined";
    return g2d;
  }

  /**
   * Method calculate
   */
  public void calculate(Deviation dev, boolean paintOnScreen, String exportFilename) {
    try {
      calc.calculate(dev, paintOnScreen, exportFilename);
    } catch (Throwable exc) {
      Console.append("Error in calculation '" + calc.toString() + "'.");
      Debug.stackTrace(exc);
    }
  }

  /**
   * Method updateImage
   *
   * @param iViewLabel
   */
  public void updateImage(JLabel iViewLabel) {
    if (!calc.isAborted()) {
      if (calc instanceof PixelCalculation) {
        PixelCalculation pixCalc = (PixelCalculation) calc;
        BufferedImage calculatedImage = pixCalc.getImage();
        AffineTransform aft = pixCalc.getAffineTransform();
        if (aft != null) {
          if (aft.getScaleX() >= 1.0 || aft.getScaleY() >= 1.0) {
            iViewLabel.setIcon(new ImageIcon(calculatedImage));
          } else {
            AffineTransformOp affineTransformOp = new AffineTransformOp(aft, AffineTransformOp.TYPE_BILINEAR);
            BufferedImage scaledImage = new BufferedImage(canvasSize.width, canvasSize.height,
                BufferedImage.TYPE_INT_RGB);
            affineTransformOp.filter(calculatedImage, scaledImage);
            iViewLabel.setIcon(new ImageIcon(scaledImage));
          }
        }
      } else if (calc instanceof VectorCalculation) {
        iViewLabel.setIcon(new ImageIcon(image));
      }
    }
  }

//  /**
//   * Method exportImage
//   */
//  public void exportImage() {
//    if (calc instanceof PixelBridgeCalculation) {
//      PixelBridgeCalculation pixelBridgeCalculation = (PixelBridgeCalculation)calc;
//      g2d.drawImage(pixelBridgeCalculation.getImage(), 0, 0, Kandid.getInstance());
//    }
//    else if (calc instanceof PixelCalculation) {
////      if (Debug.enabled)
////        assert image != null : "image is null";
////      g2d.drawImage(image, 0, 0, Kandid.getInstance());
//      PixelCalculation pixelCalculation = (PixelCalculation)calc;
//      g2d.drawRenderedImage(pixelCalculation.getImage(), new AffineTransform());
//    } else if (calc instanceof kandid.calculation.VectorCalculation) {
//    }
//  }

//  public void exportImage(Graphics2D pg2d) {
//    if (calc instanceof PixelCalculation) {
//      PixelCalculation pixelCalculation = (PixelCalculation)calc;
//      BufferedImage bufferedImage = pixelCalculation.getImage();
//      if(bufferedImage != null) 
//        pg2d.drawRenderedImage(bufferedImage, new AffineTransform());
//    }
//  }

  /**
   * @return
   */
  public boolean getReady() {
    if (cachedImage != null) {
      return true;
    }
    if (Debug.enabled)
      assert calc != null;
    if (image instanceof BufferedImage && calc.getReady() && !calc.isAborted())
      lruImageCache.put(calc, chromosome.getIdent(), (BufferedImage) image);
    return calc.getReady();
  }

  /**
   * @return
   */
  public boolean getDeferred() {
    if (cachedImage != null)
      return true;
    if (Debug.enabled)
      assert calc != null;
    return calc.getDeferred();
  }

  /**
   * Method getCalculation
   *
   * @return
   */
  public Calculation getCalculation() {
//    if (Debug.enabled)
//      assert calc != null;
    return calc;
  }

}
