<!--
  Copyright (C) 2002 - 2025 Thomas Jourdan

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
-->

<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" 
                xmlns:xsd="http://www.w3.org/2001/XMLSchema"
                xmlns:jxb="http://java.sun.com/xml/ns/jaxb"
                xmlns:kandid="http://kandid.sourceforge.net/2003/XMLSchema"
                version="1.0">

  <xsl:output method="text"/>

  <xsl:template match="xsd:schema">
    <xsl:variable name="chromosomeType" select="descendant::xsd:element[@name='chromosome']/@type"/>
    // This file was generated by the scalarvm.xsl style sheet. 
    // Any modifications to this file will be lost upon recompilation of the style sheet or the source schema. 
    
    package kandid.calculation.vm.scalar;
      import kandid.soup.ExprScalarArgsListGene;
      import kandid.soup.ExprScalarConstGene;
      import kandid.soup.ExprScalarListGene;
      import kandid.soup.ExprScalarOprBase;
      import kandid.soup.ExprScalarOprGene;
      import kandid.soup.ExprScalarVarBase;
      import kandid.soup.ExprScalarVarGene;
      import kandid.soup.GeneType;
      import kandid.soup.ScalarExprGene;
      import kandid.util.Complex;
      import kandid.util.PovNoise;

    public class ScalarVM {
      private static final int maxTapeLen = 16 * 1024;
      protected int codeTape[];
      protected int iOper[];
      protected double dOper[];
      protected int tapeLength;
			protected double dStack[] = new double[1024];
      protected PovNoise povNoise;

      public static final double epsilon = 1E-10;

      <xsl:call-template name="operator"/>

      public ScalarVM(long seed) {
        povNoise = new PovNoise(seed);
      }
			
			/**
			 * Method compile.
			 * @param scalarExprGene
			 */
			public void compile(ScalarExprGene scalarExprGene) {
				codeTape = new int[maxTapeLen];
				iOper = new int[maxTapeLen];
				dOper = new double[maxTapeLen];
				tapeLength = 0;
				compile0(scalarExprGene);
				codeTape[tapeLength] = oprStop;
				++tapeLength;
			}
		
      <xsl:call-template name="compile0"/>
			
      /**
       * @param obj
       */
      private void compileElement(GeneType obj) {
        if (obj instanceof ExprScalarVarGene) {
          ExprScalarVarGene var = (ExprScalarVarGene) obj;
          if (var.getScalarVar().equals(ExprScalarVarBase.X)) {
            codeTape[tapeLength++] = oprPushX;
          }
          else if (var.getScalarVar().equals(ExprScalarVarBase.Y)) {
            codeTape[tapeLength++] = oprPushY;
          }
          else if (var.getScalarVar().equals(ExprScalarVarBase.XI)) {
            codeTape[tapeLength++] = oprPushXI;
          }
          else if (var.getScalarVar().equals(ExprScalarVarBase.YI)) {
            codeTape[tapeLength++] = oprPushYI;
          }
          else if (var.getScalarVar().equals(ExprScalarVarBase.R)) {
            codeTape[tapeLength++] = oprPushR;
          }
          else if (var.getScalarVar().equals(ExprScalarVarBase.RI)) {
            codeTape[tapeLength++] = oprPushRI;
          }
          else if (var.getScalarVar().equals(ExprScalarVarBase.T)) {
            codeTape[tapeLength++] = oprPushT;
          }
          else if (var.getScalarVar().equals(ExprScalarVarBase.TI)) {
            codeTape[tapeLength++] = oprPushTI;
          }
          else if (var.getScalarVar().equals(ExprScalarVarBase.XP_2)) {
            codeTape[tapeLength++] = oprPushXp2;
          }
          else if (var.getScalarVar().equals(ExprScalarVarBase.YP_2)) {
            codeTape[tapeLength++] = oprPushYp2;
          }
          else if (var.getScalarVar().equals(ExprScalarVarBase.XP_3)) {
            codeTape[tapeLength++] = oprPushXp3;
          }
          else if (var.getScalarVar().equals(ExprScalarVarBase.YP_3)) {
            codeTape[tapeLength++] = oprPushYp3;
          }
          else if (var.getScalarVar().equals(ExprScalarVarBase.XCOS)) {
            codeTape[tapeLength++] = oprPushXcos;
          }
          else if (var.getScalarVar().equals(ExprScalarVarBase.YCOS)) {
            codeTape[tapeLength++] = oprPushYcos;
          }
          else if (var.getScalarVar().equals(ExprScalarVarBase.XSINH)) {
            codeTape[tapeLength++] = oprPushXsinh;
          }
          else if (var.getScalarVar().equals(ExprScalarVarBase.YSINH)) {
            codeTape[tapeLength++] = oprPushYsinh;
          }
          else {
            assert false : "invalid var compiling scalar expression" + var.getScalarVar();
          }
        }
        else if (obj instanceof ExprScalarConstGene) {
          ExprScalarConstGene aConst = (ExprScalarConstGene) obj;
          codeTape[tapeLength] = oprPushConst;
          dOper[tapeLength] = aConst.getScalarConst().getValue();
          ++tapeLength;
        }
        else if (obj instanceof ScalarExprGene) {
          ScalarExprGene scalarExpressionArg = (ScalarExprGene) obj;
          compile0(scalarExpressionArg);
          codeTape[tapeLength++] = oprPushAccu;
        }
        else {
          assert false : "invalid object compiling scalar expression" + obj.getClass().getName();
        }
      }
    
      <xsl:call-template name="execute"/>
			
      <xsl:call-template name="toString"/>
    }
  </xsl:template>


  <xsl:template name="operator">
      protected static final int oprStop      =  0;
      protected static final int oprPop       =  1;
      protected static final int oprPushConst =  2;
      protected static final int oprPushAccu  =  3;
      protected static final int oprPushX     =  4;
      protected static final int oprPushY     =  5;
      protected static final int oprPushXI    =  6;
      protected static final int oprPushYI    =  7;
      protected static final int oprPushR     =  8;
      protected static final int oprPushRI    =  9;
      protected static final int oprPushT     = 10;
      protected static final int oprPushTI    = 11;
      protected static final int oprPushXp2   = 12;
      protected static final int oprPushYp2   = 13;
      protected static final int oprPushXp3   = 14;
      protected static final int oprPushYp3   = 15;
      protected static final int oprPushXcos  = 16;
      protected static final int oprPushYcos  = 17;
      protected static final int oprPushXsinh = 18;
      protected static final int oprPushYsinh = 19;
      protected static final int oprIf  = 20;
      protected static final int oprJmp = 21;
      <xsl:for-each select="descendant::xsd:simpleType[@name = 'exprScalarOprBase']">
        <xsl:for-each select="descendant::xsd:restriction/xsd:enumeration/xsd:annotation/xsd:appinfo/kandid:execute">
            protected static final int opr^<xsl:value-of select="../../../@value"/> = (oprJmp + <xsl:value-of select="position()"/>);
        </xsl:for-each>
      </xsl:for-each>
  </xsl:template>


  <xsl:template name="compile0">
    /**
     * Method compile0.
     * @param scalarExprGene
     */
    private void compile0(ScalarExprGene scalarExprGene) {
      ExprScalarListGene exprListGene = scalarExprGene.getList();
      ExprScalarOprGene opr = exprListGene.getOpr();
      ExprScalarArgsListGene args = exprListGene.getArgs();
      java.util.List &lt;GeneType&gt; argsList = args.getVarOrConstOrScalarExpression();
      int argsLen = argsList.size();
      assert argsLen > 0 : "argument list to short " + argsLen;
      if (opr.getScalarOpr().equals(ExprScalarOprBase.IF_POSITIVE)) {
        // condition
        compileElement(argsList.get(0));
        int patchIf = tapeLength;
        codeTape[tapeLength++] = oprIf;
        // then
        compileElement(argsList.get(1 % argsLen));
        int patchJmp = tapeLength;
        codeTape[tapeLength++] = oprJmp;
        iOper[patchIf] = tapeLength;
        // else
        compileElement(argsList.get(2 % argsLen));
        iOper[patchJmp] = tapeLength;
        codeTape[tapeLength++] = oprPop;
      }
      <xsl:for-each select="descendant::xsd:simpleType[@name = 'exprScalarOprBase']">
        <xsl:for-each select="descendant::xsd:restriction/xsd:enumeration/xsd:annotation/xsd:appinfo/kandid:execute">
          else if(opr.getScalarOpr().equals(ExprScalarOprBase.<xsl:value-of select="translate(../../../@value, 'abcdefghijklmnopqrstuvwxyz', 'ABCDEFGHIJKLMNOPQRSTUVWXYZ')"/>)) {
          <xsl:if test="../kandid:args">
            int ax = <xsl:value-of select="../kandid:args"/>;
            while (ax > 0) {
              --ax;
              compileElement(argsList.get(ax % argsLen));
            }
            codeTape[tapeLength] = opr^<xsl:value-of select="../../../@value"/>;
            iOper[tapeLength] = <xsl:value-of select="../kandid:args"/>;
            ++tapeLength;
          </xsl:if>
          <xsl:if test="not(../kandid:args)">
            while (argsLen > 0) {
              --argsLen;
              compileElement(argsList.get(argsLen));
            }
            codeTape[tapeLength] = opr^<xsl:value-of select="../../../@value"/>;
            iOper[tapeLength] = argsList.size();
            ++tapeLength;
          </xsl:if>
          }
        </xsl:for-each>
      </xsl:for-each>
      else {
        assert false : "invalid operator " + opr.getScalarOpr();
      }
    }
  </xsl:template>


  <xsl:template name="execute">
  /**
   * Executes code tape and returns the result from accumulator.
   *
   * @param x
   * @param y
   * @param t
   * @return accumulator
   */
  public double execute(double x, double y, double t, double xscale, double yscale) {
    x *= Math.exp(xscale);
    y *= Math.exp(yscale);
    Complex cm = new Complex(x, y);
    Complex cm_pow2 = Complex.multiply(cm, cm);
    Complex cm_pow3 = Complex.multiply(cm, cm_pow2);
    
    Complex cm_cos = Complex.cos(cm);

    Complex cm_sinh = Complex.sinh(cm);
    
    final double xi = -x;
    final double yi = -y;
    final double r = Math.sqrt(x * x + y * y);
    final double ri = 1.0 - r;
    final double ti = -t;

    double accu = 0.0;
    int dSp = 0;
    int ip = 0;
    int argLength = 0;
    while (ip &lt; tapeLength) {
      switch (codeTape[ip]) {
        case oprStop :
          return accu;
        case oprPop :
          accu = dStack[--dSp];
          break;
        case oprPushConst :
          dStack[dSp++] = dOper[ip];
          break;
        case oprPushAccu :
          dStack[dSp++] = accu;
          break;
        case oprPushX :
          dStack[dSp++] = x;
          break;
        case oprPushY :
          dStack[dSp++] = y;
          break;
        case oprPushXI :
          dStack[dSp++] = xi;
          break;
        case oprPushYI :
          dStack[dSp++] = yi;
          break;
        case oprPushR :
          dStack[dSp++] = r;
          break;
        case oprPushRI :
          dStack[dSp++] = ri;
          break;
        case oprPushT :
          dStack[dSp++] = t;
          break;
        case oprPushTI :
          dStack[dSp++] = ti;
          break;
        case oprPushXp2 :
          dStack[dSp++] = cm_pow2.re;
          break;
        case oprPushYp2 :
          dStack[dSp++] = cm_pow2.im;
          break;
        case oprPushXp3 :
          dStack[dSp++] = cm_pow3.re;
          break;
        case oprPushYp3 :
          dStack[dSp++] = cm_pow3.im;
          break;
        case oprPushXcos :
          dStack[dSp++] = cm_cos.re;
          break;
        case oprPushYcos :
          dStack[dSp++] = cm_cos.im;
          break;
        case oprPushXsinh :
          dStack[dSp++] = cm_sinh.re;
          break;
        case oprPushYsinh :
          dStack[dSp++] = cm_sinh.im;
          break;
        case oprIf :
          accu = dStack[--dSp];
          if(accu &lt; 0.0) {
            ip = iOper[ip]-1;
          }
          break;
        case oprJmp :
          ip = iOper[ip]-1;
          break;
      <xsl:for-each select="descendant::xsd:simpleType[@name = 'exprScalarOprBase']">
        <xsl:for-each select="descendant::xsd:restriction/xsd:enumeration/xsd:annotation/xsd:appinfo/kandid:execute">
        case opr^<xsl:value-of select="../../../@value"/> :
          argLength = iOper[ip];
          <xsl:if test="../kandid:args">
          assert argLength == <xsl:value-of select="../kandid:args"/> : "argument length of operator '<xsl:value-of select="../../../@value"/>' should be <xsl:value-of select="../kandid:args"/>";
          </xsl:if>
          {
            <xsl:value-of select="."/>
          }
          break;
        </xsl:for-each>
      </xsl:for-each>
        default :
          assert false : "invalid code " + codeTape[ip] + " on tape position " + ip;
          break;
      }
      ++ip;
    }
    return accu;
    }
  </xsl:template>

  <xsl:template name="toString">
  /**
   * @see java.lang.Object#toString()
   */
  public String toString() {
    StringBuffer asm = new StringBuffer();
    int ip = 0;
    while (ip &lt; tapeLength) {
      asm.append(ip + "\t");
      switch (codeTape[ip]) {
        case oprStop :
          asm.append("stop");
          break;
        case oprPop :
          asm.append("pop");
          break;
        case oprPushConst :
          asm.append("pushConst  ");
          asm.append(dOper[ip]);
          break;
        case oprPushAccu :
          asm.append("pushAccu");
          break;
        case oprPushX :
          asm.append("pushX");
          break;
        case oprPushY :
          asm.append("pushY");
          break;
        case oprPushXI :
          asm.append("pushXI");
          break;
        case oprPushYI :
          asm.append("pushYI");
          break;
        case oprPushR :
          asm.append("pushR");
          break;
        case oprPushRI :
          asm.append("pushRI");
          break;
        case oprPushT :
          asm.append("pushT");
          break;
        case oprPushTI :
          asm.append("pushTI");
          break;
        case oprPushXp2 :
          asm.append("pushXp2");
          break;
        case oprPushYp2 :
          asm.append("pushYp2");
          break;
        case oprPushXp3 :
          asm.append("pushXp3");
          break;
        case oprPushYp3 :
          asm.append("pushYp3");
          break;
        case oprPushXcos :
          asm.append("pushXcos");
          break;
        case oprPushYcos :
          asm.append("pushYcos");
          break;
        case oprPushXsinh :
          asm.append("pushXsinh");
          break;
        case oprPushYsinh :
          asm.append("pushYsinh");
          break;
        case oprIf :
          asm.append("if\t");
          asm.append(iOper[ip]);
          break;
        case oprJmp :
          asm.append("jmp\t");
          asm.append(iOper[ip]);
          break;
    <xsl:for-each select="descendant::xsd:simpleType[@name = 'exprScalarOprBase']">
      <xsl:for-each select="descendant::xsd:restriction/xsd:enumeration/xsd:annotation/xsd:appinfo/kandid:execute">
        case opr^<xsl:value-of select="../../../@value"/> :
          asm.append("<xsl:value-of select="../../../@value"/>\t");
          asm.append(iOper[ip]);
          break;
      </xsl:for-each>
    </xsl:for-each>
        default :
          asm.append("invalid code\t");
          asm.append(iOper[ip]);
          assert false : "invalid code " + codeTape[ip] + " on tape position " + ip;
          break;
      }
      asm.append('\n');
      ++ip;
    }
    return asm.toString();
  }
  </xsl:template>

  <xsl:template match="*">
  </xsl:template>

</xsl:stylesheet>

