/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

  Copying, distribution and legal info is in the file povlegal.doc which
  should be distributed with this file. If povlegal.doc is not available
  or for more info please contact:

       Drew Wells [POV-Team Leader] 
       CIS: 73767,1244  Internet: 73767.1244@compuserve.com
       Phone: (213) 254-4041
 
----------------------------------------------------------------------------*/
package kandid.util;

import javax.vecmath.Vector3d;

/**
  This module implements texturing functions such as noise, turbulence and
  waves.

  The noise function used here is the one described by Ken Perlin in
  "Hypertexture", SIGGRAPH '89 Conference Proceedings page 253.

 */
public class PovNoise {

	static final int SINTABSIZE = 1000;
	static final int MAXSIZE = 267;

	static final int NUMBER_OF_WAVES = 10;

	static final int MINX = -10000; /* Ridiculously large scaling values */
	static final int MINY = MINX;
	static final int MINZ = MINX;

	static final double REALSCALE = (2.0 / 65535.0);

	boolean initialised;
	static double[] sintab;
	static double[] RTable;

	int[] hashTable;
	double[] frequency;
	Vector3d[] Wave_Sources;

	/**
	 * 
	 */
	public PovNoise(long seed) {
		hashTable = new int[4096];
		frequency = new double[NUMBER_OF_WAVES];
		Wave_Sources = new Vector3d[NUMBER_OF_WAVES];
		for (int ix = 0; ix < Wave_Sources.length; ix++) {
			Wave_Sources[ix] = new Vector3d();
		}
		initializeNoise(seed);
	}

	private void initializeNoise(long seed) {
		initStaticTables();

		initTextureTable(seed);

		Vector3d point;
		TextureRandomizer zVarWave = new TextureRandomizer(seed);
		for (int i = 0; i < NUMBER_OF_WAVES; i++) {
			point = getDNoise((double) i, 0.0, 0.0);
			VNormalize(Wave_Sources[i], point);
			frequency[i] = zVarWave.gleichVerteilt() + 0.01;
		}
	}

	private void initTextureTable(long seed) {
		int i, j, temp;
		TextureRandomizer zVarHashTable = new TextureRandomizer(seed);

		for (i = 0; i < 4096; i++)
			hashTable[i] = i;
		for (i = 4095; i >= 0; i--) {
			j = zVarHashTable.wuerfeln() % 4096;
			temp = hashTable[i];
			hashTable[i] = hashTable[j];
			hashTable[j] = temp;
		}
	}

	private void initStaticTables() {
		if (!initialised) {
			sintab = new double[SINTABSIZE];
			for (int i = 0; i < SINTABSIZE; i++)
				sintab[i] = Math.sin(i / (double) SINTABSIZE * (Math.PI * 2.0));

			RTable = new double[MAXSIZE];
			Vector3d rp = new Vector3d();
			for (int i = 0; i < MAXSIZE; i++) {
				rp.x = rp.y = rp.z = (double) i;
				rp = RS(rp);
				RTable[i] = (int) R(rp) * REALSCALE - 1.0;
			}
			initialised = true;
		}
	}

	private static Vector3d RS(Vector3d v) {
		v.x *= .12345;
		v.y *= .12345;
		v.z *= .12345;

		return v;
	}

	private static int R(Vector3d v) {
		int bits[] = new int[3 * 8];
		long lx = Double.doubleToLongBits(v.x);
		bits[0 + 0] = (int) ((lx >> (0 * 8)) & 0xff);
		bits[0 + 1] = (int) ((lx >> (1 * 8)) & 0xff);
		bits[0 + 2] = (int) ((lx >> (2 * 8)) & 0xff);
		bits[0 + 3] = (int) ((lx >> (3 * 8)) & 0xff);
		bits[0 + 4] = (int) ((lx >> (4 * 8)) & 0xff);
		bits[0 + 5] = (int) ((lx >> (5 * 8)) & 0xff);
		bits[0 + 6] = (int) ((lx >> (6 * 8)) & 0xff);
		bits[0 + 7] = (int) ((lx >> (7 * 8)) & 0xff);

		long ly = Double.doubleToLongBits(v.y);
		bits[8 + 0] = (int) ((ly >> (0 * 8)) & 0xff);
		bits[8 + 1] = (int) ((ly >> (1 * 8)) & 0xff);
		bits[8 + 2] = (int) ((ly >> (2 * 8)) & 0xff);
		bits[8 + 3] = (int) ((ly >> (3 * 8)) & 0xff);
		bits[8 + 4] = (int) ((ly >> (4 * 8)) & 0xff);
		bits[8 + 5] = (int) ((ly >> (5 * 8)) & 0xff);
		bits[8 + 6] = (int) ((ly >> (6 * 8)) & 0xff);
		bits[8 + 7] = (int) ((ly >> (7 * 8)) & 0xff);

		long lz = Double.doubleToLongBits(v.z);
		bits[16 + 0] = (int) ((lz >> (0 * 8)) & 0xff);
		bits[16 + 1] = (int) ((lz >> (1 * 8)) & 0xff);
		bits[16 + 2] = (int) ((lz >> (2 * 8)) & 0xff);
		bits[16 + 3] = (int) ((lz >> (3 * 8)) & 0xff);
		bits[16 + 4] = (int) ((lz >> (4 * 8)) & 0xff);
		bits[16 + 5] = (int) ((lz >> (5 * 8)) & 0xff);
		bits[16 + 6] = (int) ((lz >> (6 * 8)) & 0xff);
		bits[16 + 7] = (int) ((lz >> (7 * 8)) & 0xff);

		return (Crc16(bits));
	}

	/*
	 * Note that passing a Vector3d array to Crc16 and interpreting it as
	 * an array of chars means that machines with different floating-point
	 * representation schemes will evaluate Noise(point) differently.
	 */

	private static int Crc16(int buf[]) {
		int count = buf.length;
		int crc = 0;
		int ix = 0;

		while (count-- > 0) {
			crc = (crc >> 8) ^ crctab[(crc ^ buf[ix]) % crctab.length];
			++ix;
		}

		return ((int) crc);
	}

	/*
		Robert's Skinner's Perlin-style "Noise" function - modified by AAC
		to ensure uniformly distributed clamped values between 0 and 1.0...
	*/

	//void setup_lattice(double *x, double *y, double *z,
	//                   int *ix, int *iy, int *iz,
	//                   int *jx, int *jy, int *jz,
	//                   double *sx, double *sy, double *sz,
	//                   double *tx, double *ty, double *tz)
	//{
	//   /* ensures the values are positive. */
	//   *x -= MINX;
	//   *y -= MINY;
	//   *z -= MINZ;
	//
	//   /* its equivalent integer lattice point. */
	//   *ix = (int)*x; *iy = (int)*y; *iz = (int)*z;
	//   *jx = *ix + 1; *jy = *iy + 1; *jz = *iz + 1;
	//
	//   *sx = SCURVE(*x - *ix); *sy = SCURVE(*y - *iy); *sz = SCURVE(*z - *iz);
	//
	//   /* the complement values of sx,sy,sz */
	//   *tx = 1.0 - *sx; *ty = 1.0 - *sy; *tz = 1.0 - *sz;
	//   return;
	//}

	private static double cycloidal(double value) {
		int indx;

		if (value >= 0.0) {
			indx = (int) ((value - Math.floor(value)) * SINTABSIZE);
			return (sintab[indx]);
		}
		else {
			indx = (int) ((0.0 - (value + Math.floor(0.0 - value))) * SINTABSIZE);
			return (0.0 - sintab[indx]);
		}
	}

	public double getNoise(double x, double y, double z) {
		int ix, iy, iz, jx, jy, jz;
		double sx, sy, sz, tx, ty, tz;
		double sum;
		int m;

		//!!   setup_lattice(&x, &y, &z, &ix, &iy, &iz, &jx, &jy, &jz, &sx, &sy, &sz, &tx, &ty, &tz);
		/* ensures the values are positive. */
		x -= MINX;
		y -= MINY;
		z -= MINZ;

		/* its equivalent integer lattice point. */
		ix = (int) x;
		iy = (int) y;
		iz = (int) z;
		jx = ix + 1;
		jy = iy + 1;
		jz = iz + 1;

		sx = SCURVE(x - ix);
		sy = SCURVE(y - iy);
		sz = SCURVE(z - iz);

		/* the complement values of sx,sy,sz */
		tx = 1.0 - sx;
		ty = 1.0 - sy;
		tz = 1.0 - sz;

		/*
		 *  interpolate!
		 */
		m = Hash3d(ix, iy, iz) & 0xFF;
		sum = INCRSUM(m, (tx * ty * tz), (x - ix), (y - iy), (z - iz));

		m = Hash3d(jx, iy, iz) & 0xFF;
		sum += INCRSUM(m, (sx * ty * tz), (x - jx), (y - iy), (z - iz));

		m = Hash3d(ix, jy, iz) & 0xFF;
		sum += INCRSUM(m, (tx * sy * tz), (x - ix), (y - jy), (z - iz));

		m = Hash3d(jx, jy, iz) & 0xFF;
		sum += INCRSUM(m, (sx * sy * tz), (x - jx), (y - jy), (z - iz));

		m = Hash3d(ix, iy, jz) & 0xFF;
		sum += INCRSUM(m, (tx * ty * sz), (x - ix), (y - iy), (z - jz));

		m = Hash3d(jx, iy, jz) & 0xFF;
		sum += INCRSUM(m, (sx * ty * sz), (x - jx), (y - iy), (z - jz));

		m = Hash3d(ix, jy, jz) & 0xFF;
		sum += INCRSUM(m, (tx * sy * sz), (x - ix), (y - jy), (z - jz));

		m = Hash3d(jx, jy, jz) & 0xFF;
		sum += INCRSUM(m, (sx * sy * sz), (x - jx), (y - jy), (z - jz));

		sum = sum + 0.5; /* range at this point -0.5 - 0.5... */

		if (sum < 0.0)
			sum = 0.0;
		if (sum > 1.0)
			sum = 1.0;

		return (sum);
	}

	public Vector3d getDNoise(double x, double y, double z) {
		int ix, iy, iz, jx, jy, jz;
		double px, py, pz, s;
		double sx, sy, sz, tx, ty, tz;
		int m;
		Vector3d result = new Vector3d();

		//!!   setup_lattice(&x, &y, &z, &ix, &iy, &iz, &jx, &jy, &jz, &sx, &sy, &sz, &tx, &ty, &tz);
		/* ensures the values are positive. */
		x -= MINX;
		y -= MINY;
		z -= MINZ;

		/* its equivalent integer lattice point. */
		ix = (int) x;
		iy = (int) y;
		iz = (int) z;
		jx = ix + 1;
		jy = iy + 1;
		jz = iz + 1;

		sx = SCURVE(x - ix);
		sy = SCURVE(y - iy);
		sz = SCURVE(z - iz);

		/* the complement values of sx,sy,sz */
		tx = 1.0 - sx;
		ty = 1.0 - sy;
		tz = 1.0 - sz;

		/*
		 *  interpolate!
		 */
		m = Hash3d(ix, iy, iz) & 0xFF;
		px = x - ix;
		py = y - iy;
		pz = z - iz;
		s = tx * ty * tz;
		result.x = INCRSUM(m, s, px, py, pz);
		result.y = INCRSUM(m + 4, s, px, py, pz);
		result.z = INCRSUM(m + 8, s, px, py, pz);

		m = Hash3d(jx, iy, iz) & 0xFF;
		px = x - jx;
		s = sx * ty * tz;
		result.x += INCRSUM(m, s, px, py, pz);
		result.y += INCRSUM(m + 4, s, px, py, pz);
		result.z += INCRSUM(m + 8, s, px, py, pz);

		m = Hash3d(jx, jy, iz) & 0xFF;
		py = y - jy;
		s = sx * sy * tz;
		result.x += INCRSUM(m, s, px, py, pz);
		result.y += INCRSUM(m + 4, s, px, py, pz);
		result.z += INCRSUM(m + 8, s, px, py, pz);

		m = Hash3d(ix, jy, iz) & 0xFF;
		px = x - ix;
		s = tx * sy * tz;
		result.x += INCRSUM(m, s, px, py, pz);
		result.y += INCRSUM(m + 4, s, px, py, pz);
		result.z += INCRSUM(m + 8, s, px, py, pz);

		m = Hash3d(ix, jy, jz) & 0xFF;
		pz = z - jz;
		s = tx * sy * sz;
		result.x += INCRSUM(m, s, px, py, pz);
		result.y += INCRSUM(m + 4, s, px, py, pz);
		result.z += INCRSUM(m + 8, s, px, py, pz);

		m = Hash3d(jx, jy, jz) & 0xFF;
		px = x - jx;
		s = sx * sy * sz;
		result.x += INCRSUM(m, s, px, py, pz);
		result.y += INCRSUM(m + 4, s, px, py, pz);
		result.z += INCRSUM(m + 8, s, px, py, pz);

		m = Hash3d(jx, iy, jz) & 0xFF;
		py = y - iy;
		s = sx * ty * sz;
		result.x += INCRSUM(m, s, px, py, pz);
		result.y += INCRSUM(m + 4, s, px, py, pz);
		result.z += INCRSUM(m + 8, s, px, py, pz);

		m = Hash3d(ix, iy, jz) & 0xFF;
		px = x - ix;
		s = tx * ty * sz;
		result.x += INCRSUM(m, s, px, py, pz);
		result.y += INCRSUM(m + 4, s, px, py, pz);
		result.z += INCRSUM(m + 8, s, px, py, pz);
		return result;
	}

	public double getTurbulence(double x, double y, double z, int octaves) {
		int i; /* added -dmf */
		double t = 0.0;
		double scale, value;

		for (i = 0, scale = 1; i < octaves; i++, scale *= 0.5) {
			value = getNoise(x / scale, y / scale, z / scale);
			t += Math.abs(value) * scale;
		}
		return (t);
	}

	public Vector3d getDTurbulence(double x, double y, double z, int octaves) {
		int i; /* added -dmf */
		double scale;
		Vector3d value = new Vector3d();
		Vector3d result = new Vector3d();

		result.x = 0.0;
		result.y = 0.0;
		result.z = 0.0;

		value.x = value.y = value.z = 0.0;

		for (i = 0, scale = 1; i < octaves; i++, scale *= 0.5) {
			value = getDNoise(x / scale, y / scale, z / scale);
			result.x += value.x * scale;
			result.y += value.y * scale;
			result.z += value.z * scale;
		}
		return result;

	}

	public Vector3d getWaves(double x, double y, double z, double Frequency, double Bump_Amount, int numberOfWaves) {
		int i;
		Vector3d point = new Vector3d();
		double length, scalar, index, sinValue;
		Vector3d normal = new Vector3d();

		normal.x = 0.0;
		normal.y = 0.0;
		normal.z = 0.0;
		if (numberOfWaves > NUMBER_OF_WAVES)
			numberOfWaves = NUMBER_OF_WAVES;
		for (i = 0; i < numberOfWaves; i++) {
			point.x = x;
			point.y = y;
			point.z = z;
			VSub(point, point, Wave_Sources[i]);
			length = VDot(point, point);
			//      if (length == 0.0)
			if (length <= 0.0)
				length = 1.0;

			length = Math.sqrt(length);
			index = (length * Frequency * frequency[i]) /* !! + Texture -> Phase !! */;
			sinValue = cycloidal(index);

			scalar = sinValue * Bump_Amount / frequency[i];
			VScale(point, point, scalar / length / (double) numberOfWaves);
			VAdd(normal, normal, point);
		}
		VNormalize(normal, normal);
		return normal;
	}

	private static double SCURVE(final double a) {
		return ((a) * (a) * (3.0 - 2.0 * (a)));
	}

	private int Hash3d(final int a, final int b, final int c) {
		return hashTable[(int) (hashTable[(int) (hashTable[(int) ((a) & 0xfff)] ^ ((b) & 0xfff))] ^ ((c) & 0xfff))];
	}

	private static double INCRSUM(final int m, final double s, final double x, final double y, final double z) {
		return ((s) * (RTable[m] * 0.5 + RTable[m + 1] * (x) + RTable[m + 2] * (y) + RTable[m + 3] * (z)));
	}

	/* Vector3d Add */
	private static void VAdd(Vector3d a, Vector3d b, Vector3d c) {
		a.x = b.x + c.x;
		a.y = b.y + c.y;
		a.z = b.z + c.z;
	}

	/* Vector3d Subtract */
	private static void VSub(Vector3d a, Vector3d b, Vector3d c) {
		a.x = b.x - c.x;
		a.y = b.y - c.y;
		a.z = b.z - c.z;
	}

	/* Normalize a Vector3d - calculates a Vector3d (length of 1) that points at (b) */
	private static void VNormalize(Vector3d a, Vector3d b) {
		double VTemp = Math.sqrt(b.x * b.x + b.y * b.y + b.z * b.z);
		a.x = b.x / VTemp;
		a.y = b.y / VTemp;
		a.z = b.z / VTemp;
	}

	/* Scale - Multiply Vector3d by a Scalar */
	private static void VScale(Vector3d a, Vector3d b, final double k) {
		a.x = b.x * k;
		a.y = b.y * k;
		a.z = b.z * k;
	}

	/* Dot Product - Gives Scalar angle (a) between two vectors (b) and (c) */
	private static double VDot(Vector3d b, Vector3d c) {
		return b.x * c.x + b.y * c.y + b.z * c.z;
	}

	static final int crctab[] =
		{
			0x0000,
			0xc0c1,
			0xc181,
			0x0140,
			0xc301,
			0x03c0,
			0x0280,
			0xc241,
			0xc601,
			0x06c0,
			0x0780,
			0xc741,
			0x0500,
			0xc5c1,
			0xc481,
			0x0440,
			0xcc01,
			0x0cc0,
			0x0d80,
			0xcd41,
			0x0f00,
			0xcfc1,
			0xce81,
			0x0e40,
			0x0a00,
			0xcac1,
			0xcb81,
			0x0b40,
			0xc901,
			0x09c0,
			0x0880,
			0xc841,
			0xd801,
			0x18c0,
			0x1980,
			0xd941,
			0x1b00,
			0xdbc1,
			0xda81,
			0x1a40,
			0x1e00,
			0xdec1,
			0xdf81,
			0x1f40,
			0xdd01,
			0x1dc0,
			0x1c80,
			0xdc41,
			0x1400,
			0xd4c1,
			0xd581,
			0x1540,
			0xd701,
			0x17c0,
			0x1680,
			0xd641,
			0xd201,
			0x12c0,
			0x1380,
			0xd341,
			0x1100,
			0xd1c1,
			0xd081,
			0x1040,
			0xf001,
			0x30c0,
			0x3180,
			0xf141,
			0x3300,
			0xf3c1,
			0xf281,
			0x3240,
			0x3600,
			0xf6c1,
			0xf781,
			0x3740,
			0xf501,
			0x35c0,
			0x3480,
			0xf441,
			0x3c00,
			0xfcc1,
			0xfd81,
			0x3d40,
			0xff01,
			0x3fc0,
			0x3e80,
			0xfe41,
			0xfa01,
			0x3ac0,
			0x3b80,
			0xfb41,
			0x3900,
			0xf9c1,
			0xf881,
			0x3840,
			0x2800,
			0xe8c1,
			0xe981,
			0x2940,
			0xeb01,
			0x2bc0,
			0x2a80,
			0xea41,
			0xee01,
			0x2ec0,
			0x2f80,
			0xef41,
			0x2d00,
			0xedc1,
			0xec81,
			0x2c40,
			0xe401,
			0x24c0,
			0x2580,
			0xe541,
			0x2700,
			0xe7c1,
			0xe681,
			0x2640,
			0x2200,
			0xe2c1,
			0xe381,
			0x2340,
			0xe101,
			0x21c0,
			0x2080,
			0xe041,
			0xa001,
			0x60c0,
			0x6180,
			0xa141,
			0x6300,
			0xa3c1,
			0xa281,
			0x6240,
			0x6600,
			0xa6c1,
			0xa781,
			0x6740,
			0xa501,
			0x65c0,
			0x6480,
			0xa441,
			0x6c00,
			0xacc1,
			0xad81,
			0x6d40,
			0xaf01,
			0x6fc0,
			0x6e80,
			0xae41,
			0xaa01,
			0x6ac0,
			0x6b80,
			0xab41,
			0x6900,
			0xa9c1,
			0xa881,
			0x6840,
			0x7800,
			0xb8c1,
			0xb981,
			0x7940,
			0xbb01,
			0x7bc0,
			0x7a80,
			0xba41,
			0xbe01,
			0x7ec0,
			0x7f80,
			0xbf41,
			0x7d00,
			0xbdc1,
			0xbc81,
			0x7c40,
			0xb401,
			0x74c0,
			0x7580,
			0xb541,
			0x7700,
			0xb7c1,
			0xb681,
			0x7640,
			0x7200,
			0xb2c1,
			0xb381,
			0x7340,
			0xb101,
			0x71c0,
			0x7080,
			0xb041,
			0x5000,
			0x90c1,
			0x9181,
			0x5140,
			0x9301,
			0x53c0,
			0x5280,
			0x9241,
			0x9601,
			0x56c0,
			0x5780,
			0x9741,
			0x5500,
			0x95c1,
			0x9481,
			0x5440,
			0x9c01,
			0x5cc0,
			0x5d80,
			0x9d41,
			0x5f00,
			0x9fc1,
			0x9e81,
			0x5e40,
			0x5a00,
			0x9ac1,
			0x9b81,
			0x5b40,
			0x9901,
			0x59c0,
			0x5880,
			0x9841,
			0x8801,
			0x48c0,
			0x4980,
			0x8941,
			0x4b00,
			0x8bc1,
			0x8a81,
			0x4a40,
			0x4e00,
			0x8ec1,
			0x8f81,
			0x4f40,
			0x8d01,
			0x4dc0,
			0x4c80,
			0x8c41,
			0x4400,
			0x84c1,
			0x8581,
			0x4540,
			0x8701,
			0x47c0,
			0x4680,
			0x8641,
			0x8201,
			0x42c0,
			0x4380,
			0x8341,
			0x4100,
			0x81c1,
			0x8081,
			0x4040 };

}
