/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 * Copyright (C) 1998, 2003 Slava Pestov, setHTTPProxy is from JEdit open source project.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.Authenticator;
import java.net.PasswordAuthentication;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import kandid.Console;
import kandid.preference.HttpProxyType;
import kandid.preference.PreferenceType;
import kandid.preference.util.PasswordHelper;
import kandid.preference.util.PreferenceModel;

public class NetworkHelper {
  /**
   * Returns the email address.
   * @return String
   */
  public static String getUsableEmail(String email) {
    return email.replaceAll(" \\(dot\\) ", ".").replaceAll(" \\(at\\) ", "@");
  }

  /**
   * Method setHTTPProxy.
   * @param preferences
   */
  public static void setHTTPProxy() {
    PreferenceType preferences = PreferenceModel.getInstance().getPreference();
    if(preferences != null) {
      HttpProxyType httpProxy = preferences.getHttpProxy();
      if(httpProxy != null) {
        boolean httpEnabled = httpProxy.isEnabled();
        String host = httpProxy.getHost();
        int port = httpProxy.getPort();
        String nonProxyHosts = httpProxy.getNonProxyHosts();
        String username = httpProxy.getUsername();
        String password = PasswordHelper.construct(httpProxy.getPassword());
        setHTTPProxy(httpEnabled, host, port, nonProxyHosts, username, password);
      }
    }
  }
  
  /**
   * Method setHTTPProxy.
   * @param httpProxyEnabled
   * @param host
   * @param port
   * @param nonProxyHosts
   * @param username
   * @param password
   */
  protected static void setHTTPProxy(boolean httpProxyEnabled, String host, int port, String nonProxyHosts, String username, String password) {

    if (!httpProxyEnabled) {
      Console.append("HTTP proxy disabled");
      System.getProperties().remove("proxySet");
      System.getProperties().remove("proxyHost");
      System.getProperties().remove("proxyPort");
      System.getProperties().remove("http.proxyHost");
      System.getProperties().remove("http.proxyPort");
      System.getProperties().remove("http.nonProxyHosts");
      Authenticator.setDefault(null);
    } else {
      // set proxy host
      if (host == null)
        return;
      Console.append("HTTP proxy enabled");

      System.setProperty("http.proxyHost", host);
      Console.append("HTTP proxy host: " + host);
      // set proxy port
      if (port > 0) {
        System.setProperty("http.proxyPort", ""+port);
        Console.append("HTTP proxy port: " + port);
      }

      // set non proxy hosts list
      if (nonProxyHosts != null)
        System.setProperty("http.nonProxyHosts", nonProxyHosts);

      // set proxy authentication
      // null not supported?
      if (password == null)
        password = "";

      if (username == null || username.length() == 0) {
        Console.append("HTTP proxy without user");
        Authenticator.setDefault(new FirewallAuthenticator(null));
      } else {
        Console.append("HTTP proxy user: " + username);
        PasswordAuthentication pw = new PasswordAuthentication(username, password.toCharArray());
        Authenticator.setDefault(new FirewallAuthenticator(pw));
      }
    }
  }
  
  
  public static String convertToAscii(byte[] inp) {
    StringBuffer buf = new StringBuffer(2*inp.length);
    for (int ix = 0; ix < inp.length; ix++) {
      buf.append(toHex(inp[ix]));
    }
    return buf.toString();    
  }


  private static String toHex(int value ) {
    String hex = Integer.toHexString( value );
    if (hex.length() == 1 ) {
      return "0" + hex;
    }
    if (hex.length() > 2 ) {
      return hex.substring(hex.length() - 2);
    }
    return hex;
  }


  public static byte[] convertToBinary(String inp) {
    if((inp.length() & 1) != 0)
      throw new IllegalArgumentException("length must be even " + inp.length());
    byte[] buf = new byte[inp.length() / 2];
    StringBuffer byteValue = new StringBuffer(2);
    byteValue.setLength(2);
    for (int ix = 0; ix < buf.length; ix++) {
      byteValue.setCharAt(0, inp.charAt(2*ix));
      byteValue.setCharAt(1, inp.charAt(2*ix+1));
      buf[ix] = (byte)Integer.parseInt(byteValue.toString(), 16);
    }
    return buf;    
  }


  public static String expand(byte[] inp) {
    StringBuffer buf = new StringBuffer();
    try {
      ByteArrayInputStream bis = new ByteArrayInputStream(inp);
      GZIPInputStream gis = new GZIPInputStream(bis);
      int ic = gis.read();
      while (ic >= 0) {
        buf.append((char) ic);
        ic = gis.read();
      }
      gis.close();
    } catch (IOException exc) {
      Debug.stackTrace(exc);
    }
    return buf.toString();
  }


  public static byte[] compress(String inp) {
    ByteArrayOutputStream bos = new ByteArrayOutputStream();
    try {
      GZIPOutputStream gos = new GZIPOutputStream(bos);
      byte [] bytes = inp.getBytes();
      gos.write(bytes);
      gos.close();
    } catch (IOException exc) {
      Debug.stackTrace(exc);
    }
    return bos.toByteArray();
  }


  static class FirewallAuthenticator extends Authenticator {
    PasswordAuthentication pw;

    public FirewallAuthenticator(PasswordAuthentication pw) {
      this.pw = pw;
    }

    protected PasswordAuthentication getPasswordAuthentication() {
      return pw;
    }
  }
}
