/*
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)
    partly ported to Java by Thomas Jourdan

This file is part of the GNU C++ Library.  This library is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This library is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/
package kandid.util;

/**
 * Calculating with complex numbers.
 */
public class Complex {
  public double re;
  public double im;

  /**
   * Constructor Complex
   *
   * @param re
   * @param im
   */
  public Complex(double re, double im) {
    this.re = re;
    this.im = im;
  }

  /**
   * Method equals
   *
   * @param o
   *
   * @return
   */
  public boolean equals(Object o) {
    if (o instanceof Complex) {
      Complex x = (Complex)o;
      return (x.re == re) && (x.im == im);
    }
    return false;
  }

  /**
   * Method conj
   *
   * @param x
   *
   * @return
   */
  public static Complex conj(Complex x) {
    return new Complex(x.re, -x.im);
  }

  /**
   * Method add
   *
   * @param x
   * @param y
   *
   * @return
   */
  public static Complex add(Complex x, Complex y) {
    return new Complex(x.re + y.re, x.im + y.im);
  }

  /**
   * Method add
   *
   * @param x
   * @param y
   *
   * @return
   */
  public static Complex add(Complex x, double y) {
    return new Complex(x.re + y, x.im);
  }

  /**
   * Method add
   *
   * @param x
   * @param y
   *
   * @return
   */
  public static Complex add(double x, Complex y) {
    return new Complex(x + y.re, y.im);
  }

  /**
   * Method subtract
   *
   * @param x
   * @param y
   *
   * @return
   */
  public static Complex subtract(Complex x, Complex y) {
    return new Complex(x.re - y.re, x.im - y.im);
  }

  /**
   * Method subtract
   *
   * @param x
   * @param y
   *
   * @return
   */
  public static Complex subtract(Complex x, double y) {
    return new Complex(x.re - y, x.im);
  }

  /**
   * Method subtract
   *
   * @param x
   * @param y
   *
   * @return
   */
  public static Complex subtract(double x, Complex y) {
    return new Complex(x - y.re, -y.im);
  }

  /**
   * Method multiply
   *
   * @param x
   * @param y
   *
   * @return
   */
  public static Complex multiply(Complex x, Complex y) {
    return new Complex(x.re * y.re - x.im * y.im, x.re * y.im + x.im * y.re);
  }

  /**
   * Method multiply
   *
   * @param x
   * @param y
   *
   * @return
   */
  public static Complex multiply(Complex x, double y) {
    return new Complex(x.re * y, x.im * y);
  }

  /**
   * Method multiply
   *
   * @param x
   * @param y
   *
   * @return
   */
  public static Complex multiply(double x, Complex y) {
    return new Complex(x * y.re, x * y.im);
  }

  /**
   * Method real
   *
   * @param x
   *
   * @return
   */
  public static double real(Complex x) {
    return x.re;
  }

  /**
   * Method imag
   *
   * @param x
   *
   * @return
   */
  public static double imag(Complex x) {
    return x.im;
  }

  /**
   * Method abs
   *
   * @param x
   *
   * @return
   */
  public static double abs(Complex x) {
    return hypot(x.re, x.im);
  }

  /**
   * Method norm
   *
   * @param x
   *
   * @return
   */
  public static double norm(Complex x) {
    return (x.re * x.re + x.im * x.im);
  }

  /**
   * Method arg
   *
   * @param x
   *
   * @return
   */
  public static double arg(Complex x) {
    return Math.atan2(x.im, x.re);
  }

  /**
   * Method polar
   *
   * @param r
   * @param t
   *
   * @return
   */
  public static Complex polar(double r, double t) {
    return new Complex(r * Math.cos(t), r * Math.sin(t));
  }


  /**
   * Method divide
   * from romine@xagsun.epm.ornl.gov
   *
   * @param x
   * @param y
   *
   * @return
   */
  public static Complex divide(Complex x, Complex y) {
    double den = Math.abs(y.re) + Math.abs(y.im);
    if (den == 0.0) {
      if(Debug.enabled) assert false: "Attempted division by zero.";
      return new Complex(0.0, 0.0);
    }
    double xrden = x.re / den;
    double xiden = x.im / den;
    double yrden = y.re / den;
    double yiden = y.im / den;
    double nrm   = yrden * yrden + yiden * yiden;
    return new Complex((xrden * yrden + xiden * yiden) / nrm, (xiden * yrden - xrden * yiden) / nrm);
  }

  /**
   * Method divide
   *
   * @param x
   * @param y
   *
   * @return
   */
  public static Complex divide(double x, Complex y) {
    double den = norm(y);
    if (den == 0.0) {
      if(Debug.enabled) assert false: "Attempted division by zero.";
      return new Complex(0.0, 0.0);
    }
    return new Complex((x * y.re) / den, -(x * y.im) / den);
  }

  /**
   * Method divide
   *
   * @param x
   * @param y
   *
   * @return
   */
  public static Complex divide(Complex x, double y) {
    if (y == 0.0) {
//!!      if(Debug.enabled) assert false: "Attempted division by zero.";
      return new Complex(0.0, 0.0);
    }
    return new Complex(x.re / y, x.im / y);
  }

  /**
   * Method exp
   *
   * @param x
   *
   * @return
   */
  public static Complex exp(Complex x) {
    double r = Math.exp(x.re);
    return new Complex(r * Math.cos(x.im), r * Math.sin(x.im));
  }

  /**
   * Method cosh
   *
   * @param x
   *
   * @return
   */
  public static Complex cosh(Complex x) {
    return new Complex(Math.cos(x.im) * cosh(x.re), Math.sin(x.im) * sinh(x.re));
  }

  /**
   * Method sinh
   *
   * @param x
   *
   * @return
   */
  public static Complex sinh(Complex x) {
    return new Complex(Math.cos(x.im) * sinh(x.re), Math.sin(x.im) * cosh(x.re));
  }

  /**
   * Method cos
   *
   * @param x
   *
   * @return
   */
  public static Complex cos(Complex x) {
    return new Complex(Math.cos(x.re) * cosh(x.im), -Math.sin(x.re) * sinh(x.im));
  }

  /**
   * Method sin
   *
   * @param x
   *
   * @return
   */
  public static Complex sin(Complex x) {
    return new Complex(Math.sin(x.re) * cosh(x.im), Math.cos(x.re) * sinh(x.im));
  }

  /**
   * Method log
   *
   * @param x
   *
   * @return
   */
  public static Complex log(Complex x) {
    double h = hypot(x.re, x.im);
    if (h <= 0.0) {
      if(Debug.enabled) assert false: "attempted log of zero magnitude number.";
      return new Complex(0.0, 0.0);
    }
    return new Complex(Math.log(h), Math.atan2(x.im, x.re));
  }


  /**
   * Method pow
   * Corrections based on reports from: thc@cs.brown.edu & saito@sdr.slb.com
   *
   * @param x
   * @param p
   *
   * @return
   */
  public static Complex pow(Complex x, Complex p) {
    double h = hypot(x.re, x.im);
    if (h <= 0.0) {
      if(Debug.enabled) assert false: "attempted power of zero magnitude number.";
      return new Complex(0.0, 0.0);
    }
    double a  = Math.atan2(x.im, x.re);
    double lr = Math.pow(h, p.re);
    double li = p.re * a;
    if (p.im != 0.0) {
      lr /= Math.exp(p.im * a);
      li += p.im * Math.log(h);
    }
    return new Complex(lr * Math.cos(li), lr * Math.sin(li));
  }

  /**
   * Method pow
   *
   * @param x
   * @param p
   *
   * @return
   */
  public static Complex pow(Complex x, double p) {
    double h = hypot(x.re, x.im);
    if (h <= 0.0) {
      if(Debug.enabled) assert false: "attempted power of zero magnitude number.";
      return new Complex(0.0, 0.0);
    }
    double lr = Math.pow(h, p);
    double a  = Math.atan2(x.im, x.re);
    double li = p * a;
    return new Complex(lr * Math.cos(li), lr * Math.sin(li));
  }

  /**
   * Method sqrt
   *
   * @param x
   *
   * @return
   */
  public static Complex sqrt(Complex x) {
    if ((x.re == 0.0) && (x.im == 0.0)) {
      return new Complex(0.0, 0.0);
    }
    else {
      double s = Math.sqrt((Math.abs(x.re) + hypot(x.re, x.im)) * 0.5);
      double d = (x.im / s) * 0.5;
      if (x.re > 0.0) {
        return new Complex(s, d);
      }
      else if (x.im >= 0.0) {
        return new Complex(d, s);
      }
      else {
        return new Complex(-d, -s);
      }
    }
  }

  /**
   * Method pow
   *
   * @param x
   * @param p
   *
   * @return
   */
  public static Complex pow(Complex x, int p) {
    if (p == 0) {
      return new Complex(1.0, 0.0);
    }
    else if ((x.re == 0.0) && (x.im == 0.0)) {
      return new Complex(0.0, 0.0);
    }
    else {
      Complex res = new Complex(1.0, 0.0);
      Complex b   = x;
      if (p < 0) {
        p = -p;
        b = divide(1.0, b);
      }
      for (;;) {
        if ((p % 2) != 0) {
          res = multiply(res, b);
        }
        if ((p >>= 1) == 0) {
          return res;
        }
        else {
          b = multiply(b, b);
        }
      }
    }
  }

  /**
   * Method hypot
   *
   * @param x
   * @param y
   *
   * @return
   */
  private static double hypot(double x, double y) {
    return Math.sqrt(x * x + y * y);
  }
  
  private static double sinh(double x) {
    return 0.5*(Math.exp(x)-Math.exp(-x));
  }
  
  private static double cosh(double x) {
    return 0.5*(Math.exp(x)+Math.exp(-x));
  }
}