/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.soup.util;

/**
 * 
 * @author thomas jourdan
 */
public class ChromosomeStamp {
  private String        typeName;
  private long          populationIdent;
  private static long   chromosomeIdent;
//!!  private int           uniqStoreID;

  /**
   * Method ChromosomeStamp.
   * @param typeName
   */
  public ChromosomeStamp(String typeName) {
    long timeStamp   = System.currentTimeMillis();
    this.typeName    = typeName;
    populationIdent  = timeStamp;
    chromosomeIdent  = timeStamp;
//!!    uniqStoreID      = 0;
  }

  /**
   * Returns the full stamp
   * @return String
   */
  public String getStamp() {
    return typeName + "_" + Long.toString(populationIdent, 36) + "_" + Long.toString(chromosomeIdent, 36);
  }

  /**
   * Returns the type and populationIdent.
   * @return String
   */
  public String getTypePopulationIdent(String id2) {
    return typeName + "_" + id2 + "_" + Long.toString(populationIdent, 36);
  }

  /**
   * Returns the populationIdent.
   * @return String
   */
  public String getPopulationIdent() {
    return Long.toString(populationIdent, 36);
  }

  /**
   * Increments ident counter and returns the full stamp
   * @return String
   */
  public String getNextStamp() {
    ++chromosomeIdent;
    return getStamp();
  }

  /**
   * Only for testing. Wipe population and chromosom identification.
   */
  public void wipeIdent() {
    populationIdent = 10;
    chromosomeIdent = 0;
  }

  /**
   * increments and returns the uniqStoreID.
   * @return long
   */
//!!  public int getUniqStoreID() {
//!!    return uniqStoreID++;
//!!  }

  /**
   * parse name to set the population ID
   * @param name  Voronoi_dfg9g66x_2.kpop
   */
  public void setPopulationIDfromName(String name) {
    if(name.toLowerCase().endsWith(".kpop")) {
      name = name.substring(0, name.length()-5);
      int lastFileSeperator = name.lastIndexOf("/");
      if(lastFileSeperator > -1) {
        name = name.substring(lastFileSeperator+1);
      }
      int pos0 = name.indexOf('_');
      int pos2 = name.lastIndexOf('_');
      int pos1 = name.lastIndexOf('_', pos2-1);
      if(pos1 < 0 || pos2 < 0 || pos2 == pos1) {
        throw new IllegalArgumentException();
      }
//!!      uniqStoreID = Integer.parseInt(name.substring(pos2+1))+1;
      typeName = name.substring(0, pos0);
      populationIdent = Long.parseLong(name.substring(pos1+1, pos2), 36);
    }
    else {
      throw new IllegalArgumentException();
    }
  }

  /**
   * Method forceNextChromosomeId
   * 
   * @param origStamp
   * @return
   */
  public static String forceNextChromosomeId(String origStamp) {
    int pos = origStamp.lastIndexOf('_');
    if(pos >= 3) {
      long counter = Long.parseLong(origStamp.substring(pos+1), 36);
      if(counter <= chromosomeIdent) {
        ++chromosomeIdent;
      }
      else {
        chromosomeIdent = counter+1;
      }
    }
    else throw new IllegalArgumentException();
    return origStamp.substring(0, pos) + '_' + Long.toString(chromosomeIdent, 36);
  }

  /**
   * Remove calculation type
   * 
   * @param origStamp
   * @return
   */
  public static String removeType(String origStamp) {
    int pos = origStamp.indexOf('_');
    if(pos >= 0) {
      return origStamp.substring(pos+1, origStamp.length());
    }
    return origStamp;
  }

  /**
   * Method patchPopulationIdent.
   * @param patch
   * @param populationIdent
   * @return String
   */
  public static String patchPopulationIdent(String populationIdent, String patch) {
    int endPos = populationIdent.lastIndexOf('_');
    int startPos = populationIdent.lastIndexOf('_', endPos-1);
    return populationIdent.substring(0, startPos+1) + patch + populationIdent.substring(endPos);
  }

}
