/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.soup.genetic.lsys;

import java.util.ArrayList;
import java.util.List;

import kandid.soup.AlphabetGene;
import kandid.soup.AxiomGene;
import kandid.soup.ContextGene;
import kandid.soup.IgnoreGene;
import kandid.soup.LsysChromosome;
import kandid.soup.PredecessorGene;
import kandid.soup.SuccessorGene;
import kandid.util.Debug;

public abstract class LsysGeneticWorker {

  public abstract String randomizeAlphabet(LsysChromosome lsysChromosome);

  /**
   * @return
   */
  protected List randomizeAlphabetList(LsysChromosome lsysChromosome) {
    List alphabethList = new ArrayList();
    alphabethList.add(new String("F"));
    if(kandid.util.CentralRandomizer.getDouble(1.0) < (1.0 / 2.0)) {
      alphabethList.add(new String("f"));
    }
    if(kandid.util.CentralRandomizer.getDouble(1.0) < (1.0 / 2.0)) {
      alphabethList.add(new String("A"));
    }
    if(kandid.util.CentralRandomizer.getDouble(1.0) < (1.0 / 3.0)) {
      alphabethList.add(new String("B"));
    }
    if(kandid.util.CentralRandomizer.getDouble(1.0) < (1.0 / 4.0)) {
      alphabethList.add(new String("C"));
    }
    if(kandid.util.CentralRandomizer.getDouble(1.0) < (1.0 / 5.0)) {
      alphabethList.add(new String("D"));
    }
    if(kandid.util.CentralRandomizer.getDouble(1.0) < (3.0 / 4.0)) {
      alphabethList.add(new String("+"));
    }
    if(kandid.util.CentralRandomizer.getDouble(1.0) < (3.0 / 4.0)) {
      alphabethList.add(new String("-"));
    }
    if(kandid.util.CentralRandomizer.getDouble(1.0) < (1.0 / 4.0)) {
      alphabethList.add(new String("|"));
    }
    if(kandid.util.CentralRandomizer.getDouble(1.0) < (1.0 / 2.0)) {
      alphabethList.add(new String("."));
    }
    if(kandid.util.CentralRandomizer.getDouble(1.0) < (1.0 / 4.0)) {
      alphabethList.add(new String("'"));
    }
    if(kandid.util.CentralRandomizer.getDouble(1.0) < (1.0 / 4.0)) {
      alphabethList.add(new String(","));
    }
    if(kandid.util.CentralRandomizer.getDouble(1.0) < (1.0 / 4.0)) {
      alphabethList.add(new String(":"));
    }
    if(kandid.util.CentralRandomizer.getDouble(1.0) < (1.0 / 4.0)) {
      alphabethList.add(new String(";"));
    }
    return alphabethList;
  }

  /**
   * @param lsysChromosome
   * @return
   */
  public String randomizeAxiom(LsysChromosome lsysChromosome) {
    String alphabeth = lsysChromosome.getAlphabet().getValue();
    int len = kandid.util.CentralRandomizer.getInt(1, 6);
    StringBuffer characters = new StringBuffer();
    while (len > 0) {
      characters.append(alphabeth.charAt(kandid.util.CentralRandomizer.getInt(0, alphabeth.length() - 1)));
      --len;
    }
    return characters.toString();
  }

  /**
   * @param lsysChromosome
   * @return
   */
  public String randomizePredecessor(LsysChromosome lsysChromosome) {
    String alphabeth = lsysChromosome.getAlphabet().getValue();
    StringBuffer characters = new StringBuffer();
    characters.append(alphabeth.charAt(kandid.util.CentralRandomizer.getInt(0, alphabeth.length() - 1)));
    return characters.toString();
  }

  /**
   * @param lsysChromosome
   * @return
   */
  public String randomizeContext(LsysChromosome lsysChromosome) {
    return "";
  }

  /**
   * @param lsysChromosome
   * @return
   */
  public String randomizeSuccessor(LsysChromosome lsysChromosome) {
    String alphabeth = lsysChromosome.getAlphabet().getValue();
    int len = kandid.util.CentralRandomizer.getInt(1, 20);
    StringBuffer characters = new StringBuffer();
    int needsBranchClosure = 0;
    int needsLeafClosure = 0;
    // generate string with balanced brackets [] {}
    while (len > 0) {
      // calculate probability for closing bracket ] 
      double probabilityBranchClosure = (double) (needsBranchClosure + needsLeafClosure) / (double) len;
      // calculate probability for closing bracket }
      double probabilityLeafClosure = (double) (needsBranchClosure + needsLeafClosure) / (double) len;
      if (needsBranchClosure > 0 && kandid.util.CentralRandomizer.getDouble(1.0) < probabilityBranchClosure + 0.01) {
        // insert closing bracket
        characters.append(']');
        --needsBranchClosure;
      }
      else if ((needsBranchClosure + needsLeafClosure)  < (len - 1) && kandid.util.CentralRandomizer.getDouble(1.0) < (1.0 / ((double) (alphabeth.length() + 2)))) {
        // insert opening bracket [ if there is enough room for the closing bracket
        // the probability is equal to other characters in the alphabet
        characters.append('[');
        ++needsBranchClosure;
      }
      else if (needsLeafClosure > 0 && kandid.util.CentralRandomizer.getDouble(1.0) < probabilityLeafClosure + 0.01) {
        // insert closing bracket
        characters.append('}');
        --needsLeafClosure;
      }
      else if ((needsBranchClosure + needsLeafClosure) < (len - 1) && kandid.util.CentralRandomizer.getDouble(1.0) < (1.0 / ((double) (alphabeth.length() + 2)))) {
        // insert opening bracket { if there is enough room for the closing bracket
        // the probability is equal to other characters in the alphabet
        characters.append('{');
        ++needsLeafClosure;
      }
      else {
        // insert normal character from alphabet
        characters.append(alphabeth.charAt(kandid.util.CentralRandomizer.getInt(0, alphabeth.length() - 1)));
      }
      assert needsBranchClosure >= 0;
      assert needsLeafClosure >= 0;
      --len;
    }
    // assert that all brackets are closed
    if (Debug.enabled) {
      assert invariant(characters) : characters.toString();
    }
    return characters.toString();
  }

  /**
   * @param lsysChromosome
   * @return
   */
  public String randomizeIgnore(LsysChromosome lsysChromosome) {
    return "";
  }

//!!  public String randomizeProduction(LsysChromosome lsysChromosome) {
//!!  }
  
  
  /**
   * @param characters
   * @return
   */
  public void mutateAlphabet(AlphabetGene aGene, LsysChromosome lsysChromosome) {
    // TODO Alphabet is not mutated.
  }

  /**
   * @param characters
   * @param lsysChromosome
   * @return
   */
  public void mutateAxiom(AxiomGene aGene, LsysChromosome lsysChromosome) {
    String alphabeth = lsysChromosome.getAlphabet().getValue();
    char[] chars = aGene.getValue().toCharArray();
    int pos = kandid.util.CentralRandomizer.getInt(0, chars.length - 1);    
    chars[pos] = alphabeth.charAt(kandid.util.CentralRandomizer.getInt(0, alphabeth.length() - 1));
    aGene.setValue(new String(chars));
  }

  /**
   * @param characters
   * @param lsysChromosome
   * @return
   */
  public void mutatePredecessor(PredecessorGene aGene, LsysChromosome lsysChromosome) {
    String alphabeth = lsysChromosome.getAlphabet().getValue();
    char[] chars = aGene.getValue().toCharArray();
    assert chars.length == 1;    
    chars[0] = alphabeth.charAt(kandid.util.CentralRandomizer.getInt(0, alphabeth.length() - 1));
    aGene.setValue(new String(chars));
  }

  /**
   * @param aGene
   * @param lsysChromosome
   */
  public void mutateContext(ContextGene aGene, LsysChromosome lsysChromosome) {
    // do nothing    
  }

  /**
   * @param characters
   * @param lsysChromosome
   * @return
   */
  public void mutateSuccessor(SuccessorGene aGene, LsysChromosome lsysChromosome) {
    StringBuffer chars = null;
    do {
      chars = new StringBuffer(aGene.getValue());
      int pos1 = kandid.util.CentralRandomizer.getInt(0, chars.length() - 1);    
      int pos2 = kandid.util.CentralRandomizer.getInt(0, chars.length() - 1);
      char temp = chars.charAt(pos1);
      chars.setCharAt(pos1, chars.charAt(pos2));
      chars.setCharAt(pos2, temp);    
    } while(!invariant(chars));
    aGene.setValue(new String(chars));
  }

  /**
   * @param aGene
   * @param lsysChromosome
   */
  public void mutateIgnore(IgnoreGene aGene, LsysChromosome lsysChromosome) {
    // do nothing    
  }

//!!  public LsysProductionGene mutateProduction(LsysProductionGene aGene, LsysChromosome lsysChromosome) {
//!!    return aGene;
//!!  }

  /**
   * @param characters
   * @return
   */
  private boolean invariant(StringBuffer characters) {
    int branchClosureCount = 0;
    int leafClosureCount = 0;
    for (int ix = 0; ix < characters.length(); ix++) {
      char character = characters.charAt(ix);
      if (character == '[') {
        ++branchClosureCount;
      }
      else if (character == ']') {
        --branchClosureCount;
      }
      else if (character == '{') {
        ++leafClosureCount;
      }
      else if (character == '}') {
        --leafClosureCount;
      }
      if (branchClosureCount < 0)
        return false;
      if (leafClosureCount < 0)
        return false;
    }
    return branchClosureCount == 0 && leafClosureCount == 0;
  }

  /**
   * @param lsysChromosome
   * @return
   */
  public abstract int getNumberOfProductions(LsysChromosome lsysChromosome);

  /**
   * @param lsysChromosome
   * @return
   */
  public boolean reject(LsysChromosome lsysChromosome) {
    return false;
  }

}
