/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.preference.ui;

import java.io.File;

import kandid.Kandid;
import kandid.preference.GenomeDBType;
import kandid.preference.HttpProxyType;
import kandid.preference.OutgoingMailType;
import kandid.preference.PreferenceType;
import kandid.preference.UserType;
import kandid.preference.util.PasswordHelper;
import kandid.preference.util.PreferenceModel;
import kandid.util.Debug;

/**
 * @author thomas jourdan
 *  
 */
public class IlPreference {
  private static IlPreference preferenceController = null;

  private UiPreference uiPreference;
  private PreferenceModel preferenceModel;
  
   public IlPreference() {
  }

  public void initialize() {
    uiPreference = new UiPreference(this);
    preferenceModel = PreferenceModel.getInstance();
    File filebaseDir = new File(Kandid.localImageFolder);
    uiPreference.getFilebaseTextField().setText(filebaseDir.getAbsolutePath());
    File incommingDir = new File(Kandid.incomingImageFolder);
    uiPreference.getIncommingTextField().setText(incommingDir.getAbsolutePath());
    File scratchDir = new File(Kandid.scratchFolder);
    uiPreference.getScratchTextField().setText(scratchDir.getAbsolutePath());
    File cacheDir = new File(Kandid.cacheFolder);
    uiPreference.getCacheTextField().setText(cacheDir.getAbsolutePath());
  }

   public void apply() {
    save();
  }

   public void ok() {
    save();
    uiPreference.setVisible(false);
    uiPreference.dispose();
  }

  public void cancel() {
    uiPreference.setVisible(false);
    uiPreference.dispose();
  }

  public void mapFromEditor() {
    PreferenceType preference = preferenceModel.getPreference();
    try {
      UserType user = preferenceModel.getObjectFactory().createUserType();
      user.setNickname(uiPreference.getNicknameTextField().getText());
      user.setEmail(uiPreference.getMailTextField().getText());
      preference.setUser(user);

      OutgoingMailType outgoingMail = preferenceModel.getObjectFactory().createOutgoingMailType();
      outgoingMail.setSmptHost(uiPreference.getSmtpTextField().getText());
      preference.setOutgoingMail(outgoingMail);

      HttpProxyType httpProxy = preferenceModel.getObjectFactory().createHttpProxyType();
      httpProxy.setEnabled(uiPreference.getEnableProxyCheckBox().isSelected());
      httpProxy.setHost(uiPreference.getProxyHostTextField().getText());
      try {
        httpProxy.setPort(Integer.parseInt(uiPreference.getProxyPortTextField().getText()));
      } catch (NumberFormatException nfexc) {
        httpProxy.setPort(80);
      }
      httpProxy.setNonProxyHosts(uiPreference.getNonProxyHostsTextField().getText());
      httpProxy.setUsername(uiPreference.getUsernameTextField().getText());
      char password[] = uiPreference.getJPasswordField().getPassword();
      httpProxy.setPassword(PasswordHelper.deconstruct(new String(password)));
      preference.setHttpProxy(httpProxy);

      GenomeDBType genomeDB = preferenceModel.getObjectFactory().createGenomeDBType();
      genomeDB.setURL(uiPreference.getWebDbUrlTextField().getText());
      try {
        genomeDB.setTimeout(Integer.parseInt(uiPreference.getTimeoutTextField().getText()));
      } catch (NumberFormatException nfexc) {
        genomeDB.setTimeout(50000);
      }
      try {
        genomeDB.setInstallationId(Integer.parseInt(uiPreference.getInstallIdTextField().getText()));
      } catch (NumberFormatException nfexc) {
        genomeDB.setInstallationId(0);
      }
      preference.setGenomeDB(genomeDB);
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
  }

  /**
   * Method mapToEditor.
   * 
   * @param preference
   * @param uiPreference
   */
  public void mapToEditor() {
    try {
      PreferenceType preference = preferenceModel.getPreference();
      UserType user = preference.getUser();
      if (user != null) {
        uiPreference.getNicknameTextField().setText(user.getNickname());
        uiPreference.getMailTextField().setText(user.getEmail());
      }

      OutgoingMailType outgoingMail = preference.getOutgoingMail();
      if (outgoingMail != null) {
        uiPreference.getSmtpTextField().setText(outgoingMail.getSmptHost());
      }
      
      HttpProxyType httpProxy = preference.getHttpProxy();
      if (httpProxy != null) {
        uiPreference.getEnableProxyCheckBox().setSelected(httpProxy.isEnabled());
        uiPreference.getProxyHostTextField().setText(httpProxy.getHost());
        uiPreference.getProxyPortTextField().setText("" + httpProxy.getPort());
        uiPreference.getNonProxyHostsTextField().setText(httpProxy.getNonProxyHosts());
        uiPreference.getUsernameTextField().setText(httpProxy.getUsername());
        uiPreference.getJPasswordField().setText(PasswordHelper.construct(httpProxy.getPassword()));
      }

      GenomeDBType genomeDB = preference.getGenomeDB();
      if (genomeDB != null) {
        uiPreference.getWebDbUrlTextField().setText(genomeDB.getURL());
        uiPreference.getWebDbUrlTextField().setCaretPosition(0);
        uiPreference.getTimeoutTextField().setText("" + genomeDB.getTimeout());
        uiPreference.getInstallIdTextField().setText("" + genomeDB.getInstallationId());
      }
      
      enableProxy();
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
  }
  
  public static IlPreference showInstance(int tab, java.awt.Frame parent) {
    preferenceController = IlPreference.getInstance();
    preferenceController.mapToEditor();

    preferenceController.uiPreference.pack();
    preferenceController.uiPreference.setLocationRelativeTo(parent);
    try {
      preferenceController.uiPreference.getJTabbedPane().setSelectedIndex(tab);
    } catch (IndexOutOfBoundsException exc) {
      Debug.stackTrace(exc);
    }
    preferenceController.uiPreference.setVisible(true);
    return preferenceController;
  }

  public static IlPreference getInstance() {
    if (preferenceController == null) {
      preferenceController = new IlPreference();
      preferenceController.initialize();
    }
    return preferenceController;
  }

  private void save() {
    mapFromEditor();
    preferenceModel.marshalToFile();
  }

//  public void setHTTPProxy() {
//    try {
//      ensureHttpProxy();
//      PreferenceType preferences = preferenceModel.getPreference();
//      if(preferences != null)
//        NetworkHelper.setHTTPProxy(preferences);
//    } catch (Exception exc) {
//      Debug.stackTrace(exc);
//      Console.println(exc.getMessage());
//    }
//  }

  public void ensureLoadGenomeDBCompleted() {
    if(!preferenceModel.checkLoadGenomeDBCompleted()) {
      showInstance(1, null);
    }
  }

  public void ensureExportGenomeDBCompleted() {
    if(!preferenceModel.checkExportGenomeDBCompleted()) {
      showInstance(0, null);
    }
  }

  public void ensureOutgoingMailCompleted() {
    if(!preferenceModel.checkOutgoingMailCompleted()) {
      showInstance(3, null);
    }
  }

  /**
   * 
   */
  public void enableProxy() {
    boolean enabled = uiPreference.getEnableProxyCheckBox().isSelected();
    uiPreference.getProxyHostTextField().setEnabled(enabled);
    uiPreference.getProxyPortTextField().setEnabled(enabled);
    uiPreference.getNonProxyHostsTextField().setEnabled(enabled);
    uiPreference.getUsernameTextField().setEnabled(enabled);
    uiPreference.getJPasswordField().setEnabled(enabled);
  }

  public static void main(String args[]) {
    IlPreference.showInstance(0, null);
  }

}