/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.pond.ui;

import java.awt.Dimension;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JTree;
import javax.swing.TransferHandler;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;

import kandid.Gateway;
import kandid.ImageTransferHandler;
import kandid.Kandid;
import kandid.ViewEngine;
import kandid.calculation.Calculation;
import kandid.catalog.CalculationType;
import kandid.catalog.CatalogType;
import kandid.catalog.ModelType;
import kandid.colorator.Colorator;
import kandid.colorator.ColoratorFactory;
import kandid.preference.ui.IlPreference;
import kandid.soup.ChromosomeType;
import kandid.soup.ColoratorType;
import kandid.soup.EntityType;
import kandid.soup.ImageType;
import kandid.soup.util.ChromosomeStamp;
import kandid.soup.util.SoupFactory;
import kandid.util.Debug;

/**
 * @author thomas jourdan
 *  
 */
public class IlPool {
  private static IlPool wizzard = null;

  private UiPool uiPool;
  private TypeDescription selectedType = null;
  private String[] filenameList;
  private Vector<String> chromosomeList;
  private int startPos;
  private ViewEngine viewEngine[];
  private EntityType entity[];
  private DefaultTreeModel typeTreeModel;
  private SoupFactory soupFactory;
  private boolean loading;

  public static IlPool showInstance() {
    getInstance();
    wizzard.uiPool.setVisible(true);
    return wizzard;
  }

  public static IlPool getInstance() {
    if (wizzard == null) {
      wizzard = new IlPool();
      wizzard.initialize();
      wizzard.uiPool.pack();
    }
    return wizzard;
  }

  private void initialize() {
    uiPool = new UiPool(this);
    uiPool.getBothRadioButton().setSelected(false);
    uiPool.getLoadButton().setEnabled(false);

    soupFactory = SoupFactory.getSoupFactory();
    chromosomeList = new Vector<>();

    viewEngine = new ViewEngine[10];
    entity = new EntityType[viewEngine.length];
    for (int ix = 0; ix < viewEngine.length; ++ix) {
      viewEngine[ix] = null;
      entity[ix] = null;
    }

    initTreeModel();

    startPos = 0;
    activateView();
    uiPool.getPoolContentPane().setTransferHandler(new ImageTransferHandler(true));
    uiPool.getPoolPanel().setTransferHandler(new ImageTransferHandler(true));
    uiPool.getTypeTree().setTransferHandler(new ImageTransferHandler(true));
    uiPool.getViewPanel().setTransferHandler(new ImageTransferHandler(true));
  }

  public void load() {
    if (selectedType != null) {
      loading = true;
      uiPool.getLoadButton().setEnabled(false);

      IlLoadWizzard ilLoadWizzard = IlLoadWizzard.showInstance();
      ilLoadWizzard.setQueryCalulationName(selectedType.calcClassName);
      ilLoadWizzard.setQueryColoratorName(selectedType.colorClassName);
    }
  }

  public void finalizeLoad(boolean isCanceled) {
    if(Debug.enabled) System.out.println("finalizeLoad.isCanceled: " + isCanceled);
    loading = false;
    uiPool.getLoadButton().setEnabled(selectedType != null);
    if(!isCanceled)
      activateView();
  }

  private void initTreeModel() {
    try {
      JAXBContext jc = JAXBContext.newInstance("kandid.catalog");
      Unmarshaller u = jc.createUnmarshaller();
      javax.xml.bind.JAXBElement uo = (JAXBElement) u.unmarshal(getClass().getResourceAsStream("/kandid/catalog/catalog.xml"));
      CatalogType catalog = (kandid.catalog.CatalogType) uo.getValue();
      java.util.List<CalculationType> calculationList = catalog.getCalculation();

      JTree typeTree = uiPool.getTypeTree();
      DefaultMutableTreeNode rootNode = new DefaultMutableTreeNode();
      rootNode.setUserObject("Pool");
      typeTreeModel = new DefaultTreeModel(rootNode);
      typeTree.setModel(typeTreeModel);

      for (Iterator<CalculationType> iter = calculationList.iterator(); iter.hasNext();) {
        CalculationType calculationType = iter.next();
        DefaultMutableTreeNode calcNode = new DefaultMutableTreeNode();
        String calculationName = calculationType.getName();
        calcNode.setUserObject(calculationName);

        java.util.List<ModelType> modelList = calculationType.getModel();
        for (Iterator<ModelType> iterModel = modelList.iterator(); iterModel.hasNext();) {
          ModelType modelType = iterModel.next();
          try {
            Class cc = Class.forName(modelType.getColoratorClass());
            Colorator colorator = (Colorator) cc.getDeclaredConstructor().newInstance();

            DefaultMutableTreeNode colorNode = new DefaultMutableTreeNode();
            TypeDescription colorDescription = new TypeDescription(modelType.getCalculationClass(), calculationName,
                modelType.getColoratorClass(), colorator.getShortName());
            colorNode.setUserObject(colorDescription);
            calcNode.add(colorNode);
          } catch (Exception exc) {
            Debug.stackTrace(exc);
          }
        }

        rootNode.add(calcNode);
      }
      typeTree.expandPath(new TreePath(rootNode));
    } catch (JAXBException exc) {
      Debug.stackTrace(exc);
    }
  }

  protected void activateView() {
  //TODO WEB database
    boolean displayExported = true;
    boolean displayIncoming = false;
//    boolean displayExported = uiPool.getBothRadioButton().isSelected() || uiPool.getOwnRadioButton().isSelected();
//    boolean displayIncoming = uiPool.getBothRadioButton().isSelected() || uiPool.getLoadedRadioButton().isSelected();

    chromosomeList.clear();

    if (selectedType != null) {
      java.util.ArrayList filenameListExported = new java.util.ArrayList();
      java.util.ArrayList filenameListIncoming = new java.util.ArrayList();
      if (displayExported) {
        filenameListExported = kandid.extensions.FileList.createImageList("./filebase", buildName(), ".kimg");
      }
      if (displayIncoming) {
        filenameListIncoming = kandid.extensions.FileList.createImageList("./incoming", buildName(), ".kimg");
      }
      List sum = new ArrayList(filenameListExported.size() + filenameListIncoming.size());
      for (int i1 = 0; i1 < filenameListIncoming.size(); ++i1) {
        sum.add(filenameListIncoming.get(i1));
      }
      for (int i2 = 0; i2 < filenameListExported.size(); ++i2) {
        sum.add(filenameListExported.get(i2));
      }
      filenameList = new String[sum.size()];
      filenameList = (String[]) sum.toArray(filenameList);

      for (int fx = 0; fx < filenameList.length; ++fx) {
        try {
          StringBuffer chromosomeBuffer = new StringBuffer();
          String chromosome = null;
          File inf = new File(filenameList[fx]);
          int len = (int) inf.length();
          FileReader in = new FileReader(inf);
          char[] buffer = new char[len];
          while (in.read(buffer) > 0) {
            chromosomeBuffer.append(buffer);
          }
          chromosome = chromosomeBuffer.toString();

          if (!chromosomeList.contains(chromosome)) {
            chromosomeList.add(chromosome);
          }
          in.close();
        } catch (IOException exc) {
          Debug.stackTrace(exc);
        }
      }
    }
    goTop();
  }

  private String buildName() {
    return selectedType.calcShortName + "_" + selectedType.colorShortName;
  }

  private void clean() {
    for (int ix = 0; ix < viewEngine.length; ++ix) {
      if (viewEngine[ix] != null) {
        viewEngine[ix].stop();
        viewEngine[ix] = null;
      }
    }
    uiPool.getViewPanel().removeAll();
  }

  private void next(int start) {
    int size = chromosomeList.size();
    for (int vx = 0; vx < viewEngine.length; ++vx) {
      int px = start + vx;
      if (px < size) {
        entity[vx] = (EntityType) soupFactory.unmarshalFromString(chromosomeList.get(px));
        ChromosomeType chromosome = soupFactory.getChromosome(entity[vx]);
        Calculation calculation = (new Gateway()).createCalculation(soupFactory.getCalculationName(entity[vx]));
        ColoratorType coloratorType = soupFactory.getColorator(entity[vx]);
        String coloratorName = soupFactory.getColoratorName(entity[vx]);
        Colorator colorator = ColoratorFactory.createColoratorWorker(coloratorName, coloratorType);
        viewEngine[vx] = new ViewEngine(100, 6, 6, null, this);
        
        String author = soupFactory.getImage(entity[vx]).getAuthor();
        if(author == null || author.length() == 0)
          author = "unknown author";
        else
          author = "Author: " + author;
        viewEngine[vx].setToolTipText(author);

        // drag and drop
        viewEngine[vx].addMouseListener(wizzard.new DragMouseAdapter());
        viewEngine[vx].setTransferHandler(new ImageTransferHandler(true));

        uiPool.getViewPanel().add(viewEngine[vx]);

        viewEngine[vx].prepare(chromosome, calculation, colorator);
        viewEngine[vx].start();
      }
      else {
        JLabel empty = new JLabel("");
        empty.setMinimumSize(new Dimension(100, 100));
        empty.setPreferredSize(new Dimension(100, 100));
        // drag and drop
        empty.setTransferHandler(new ImageTransferHandler(true));

        uiPool.getViewPanel().add(empty);

        entity[vx] = null;
        viewEngine[vx] = null;
      }
    }
    enableButtons();
    uiPool.getViewPanel().repaint();
  }

  private class DragMouseAdapter extends MouseAdapter {
    public void mousePressed(MouseEvent e) {
      JComponent c = (JComponent) e.getSource();
      TransferHandler handler = c.getTransferHandler();
      handler.exportAsDrag(c, e, TransferHandler.COPY);
    }
  }

  protected void goTop() {
    clean();
    startPos = 0;
    next(startPos);
  }

  protected void goBottom() {
    clean();
    startPos = chromosomeList.size() - viewEngine.length;
    if (startPos < 0)
      startPos = 0;
    next(startPos);
  }

  protected void goForward() {
    clean();
    startPos += viewEngine.length;
    next(startPos);
  }

  protected void goBackward() {
    clean();
    startPos -= viewEngine.length;
    if (startPos < 0)
      startPos = 0;
    next(startPos);
  }

  public EntityType getEntity(ViewEngine aViewEngine) {
    for (int vx = 0; vx < viewEngine.length; ++vx) {
      if (viewEngine[vx] == aViewEngine) {
        return entity[vx];
      }
    }
    return null;
  }
  /**
   * @param sel
   */
  public void nodeSelected(TreeSelectionEvent sel) {
    DefaultMutableTreeNode node = (DefaultMutableTreeNode) uiPool.getTypeTree().getLastSelectedPathComponent();

    selectedType = null;
    if (node != null) {
      if (node.isLeaf()) {
        TreePath path = sel.getPath();
        DefaultMutableTreeNode o2 = (DefaultMutableTreeNode) path.getPathComponent(2);
        selectedType = ((TypeDescription) o2.getUserObject());
      }
    }
    uiPool.getLoadButton().setEnabled(!loading && selectedType != null);
    if (selectedType != null) {
      if(Debug.enabled) System.out.println("nodeSelected: " + selectedType.calcClassName + ", "+ selectedType.colorClassName);
      activateView();
    }
  }

  public void addImage(ImageType image) {
    try {
      String typeName = image.getClass().getName();
      EntityType addEntity = soupFactory.createEntity(image);
      addEntity(typeName, addEntity, true);
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
  }

  private void addEntity(String typeName, EntityType entity, boolean marshalToFile) {
    String shortColoratorName = ColoratorFactory.getShortName(soupFactory.getColoratorName(entity));
    ChromosomeType chromosomeType = soupFactory.getChromosome(entity);
    String ident = chromosomeType.getIdent();
    String filename = ChromosomeStamp.patchPopulationIdent(ident, shortColoratorName + "_pool");
    if (marshalToFile) {
      kandid.extensions.ExportState exportState = new kandid.extensions.ExportState();
      exportState.setFullname(filename);
      (new File(exportState.getPath(Kandid.localImageFolder))).mkdirs();
      soupFactory.marshalToFile(entity, exportState.getPath(Kandid.localImageFolder) + filename + ".kimg");
    }

    navigateToType(soupFactory.getCalculationName(entity), soupFactory.getColoratorName(entity));
    activateView();
    enableButtons();
    uiPool.repaint();
  }

  private void enableButtons() {
    uiPool.getLastButton().setEnabled(!isLastPage());
    uiPool.getRightButton().setEnabled(!isLastPage());
    uiPool.getFirstButton().setEnabled(!isFirstPage());
    uiPool.getLeftButton().setEnabled(!isFirstPage());
  }

  private boolean isFirstPage() {
    return startPos <= 0;
  }

  private boolean isLastPage() {
    return startPos >= chromosomeList.size() - viewEngine.length;
  }

  public void navigateToType(String calcClassName, String colorClassName) {
    try {
      if (Debug.enabled)
        System.out.println("navigateToType: " + calcClassName + ", " + colorClassName);
      TreePath newPath = null;

      TreePath oldPath = uiPool.getTypeTree().getSelectionPath();

      DefaultTreeModel tm = (DefaultTreeModel) uiPool.getTypeTree().getModel();
      Object tr = tm.getRoot();
      int d1Count = tm.getChildCount(tr);
      for (int ix1 = 0; ix1 < d1Count; ++ix1) {
        Object calcNode = tm.getChild(tr, ix1);
        int d2Count = tm.getChildCount(calcNode);
        for (int ix2 = 0; ix2 < d2Count; ++ix2) {
          DefaultMutableTreeNode colorNode = (DefaultMutableTreeNode) tm.getChild(calcNode, ix2);
          TypeDescription colorDescription = (TypeDescription) colorNode.getUserObject();
          if (colorDescription.calcClassName.equals(calcClassName)
              && colorDescription.colorClassName.equals(colorClassName)) {
            if(Debug.enabled) System.out.println("found: " + calcNode + ", " + colorNode);
            TreeNode[] newNodePath = tm.getPathToRoot(colorNode);
            newPath = new TreePath(newNodePath);
          }
        }
      }

      if (newPath != null) {
        if (oldPath != null) {
          if (!newPath.equals(oldPath)) {
            uiPool.getTypeTree().removeSelectionPath(oldPath);
            uiPool.getTypeTree().setSelectionPath(newPath);
          }
        }
        else {
          uiPool.getTypeTree().setSelectionPath(newPath);
        }
      }
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
  }

  public void setVisible(boolean visible) {
    uiPool.setVisible(visible);
  }

  public static void main(String args[]) {
    IlPool ilPool = IlPool.getInstance();
    ilPool.navigateToType("kandid.calculation.vm.scalar.ScalarExpressionCalculation", "kandid.colorator.GradientColorator");
    ilPool = IlPool.showInstance();
  }

  /**
   * 
   */
  public void showPreference() {
    IlPreference.showInstance(0, uiPool);
  }

}