/*
 * Copyright 2013, Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package kandid.noise;

import javax.vecmath.Vector3d;

public class PerlinNoise {
  ImprovedNoise improvedPerlinNoise;

  public PerlinNoise(long seed) {
    this.improvedPerlinNoise = new ImprovedNoise();
    this.improvedPerlinNoise.seed(seed);
  }

  public void initializeNoise(long seed) {
    this.improvedPerlinNoise.seed(seed);
  }

  public double unoise(double x, double y, double z) {
    return 0.5 * (1.0 + this.improvedPerlinNoise.noise(x, y, z));
  }

  public double snoise(double x, double y, double z) {
    return this.improvedPerlinNoise.noise(x, y, z);
  }

  /**
   * @see https://svn.blender.org/svnroot/bf-blender/trunk/blender/source/blender/python/mathutils/mathutils_noise.c
   * @param x
   * @param y
   * @param z
   * @param vnoise
   */
  public void unoise3d(double x, double y, double z, Vector3d vnoise) {
    /* Simply evaluate noise at 3 different positions */
    vnoise.x = this.unoise(x + 9.321, y - 1.531, z - 7.951);
    vnoise.y = this.unoise(x, y, z);
    vnoise.z = this.unoise(x + 6.327, y + 0.1671, z - 2.672);
  }

  public void snoise3d(double x, double y, double z, Vector3d vnoise) {
    /* Simply evaluate noise at 3 different positions */
    vnoise.x = this.snoise(x + 9.321, y - 1.531, z - 7.951);
    vnoise.y = this.snoise(x, y, z);
    vnoise.z = this.snoise(x + 6.327, y + 0.1671, z - 2.672);
  }

  public double uturbulence(double x, double y, double z, double octaves, double ampscale, double freqscale) {
    double amp = 1.0;
    double noise = this.unoise(x, y, z);
    int ioctaves = (int)octaves;
    double fraction = octaves - ioctaves;
    for (int i = 1; i < ioctaves; i++) {
      amp *= ampscale;
      x *= freqscale;
      y *= freqscale;
      z *= freqscale;
      double t = amp * (this.unoise(x, y, z) - 0.5);
      noise += t;
    }
    amp *= ampscale;
    x *= freqscale;
    y *= freqscale;
    z *= freqscale;
    double t = fraction * amp * (this.unoise(x, y, z) - 0.5);
    noise += t;
    return noise;
  }

  public double sturbulence(double x, double y, double z, double octaves, double ampscale, double freqscale) {
    double amp = 1.0;
    double noise = this.snoise(x, y, z);
    int ioctaves = (int)octaves;
    double fraction = octaves - ioctaves;
    for (int i = 1; i < ioctaves; i++) {
      amp *= ampscale;
      x *= freqscale;
      y *= freqscale;
      z *= freqscale;
      double t = amp * (this.snoise(x, y, z) - 0.5);
      noise += t;
    }
    amp *= ampscale;
    x *= freqscale;
    y *= freqscale;
    z *= freqscale;
    double t = fraction * amp * (this.snoise(x, y, z) - 0.5);
    noise += t;
    return noise;
  }

  /**
   * @see https://svn.blender.org/svnroot/bf-blender/trunk/blender/source/blender/python/mathutils/mathutils_noise.c
   * @param x
   * @param y
   * @param z
   * @param octaves
   * @param ampscale
   * @param freqscale
   * @param vnoise
   */
  public void uturbulence3d(double x, double y, double z, int octaves, double ampscale, double freqscale, Vector3d vnoise) {
    vnoise.x = this.unoise(x + 9.321, y - 1.531, z - 7.951);
    vnoise.y = this.unoise(x, y, z);
    vnoise.z = this.unoise(x + 6.327, y + 0.1671, z - 2.672);
    double amp = 1.0;
    for (int i = 1; i < octaves; i++) {
      amp *= ampscale;
      x *= freqscale;
      y *= freqscale;
      z *= freqscale;
      double nx = this.unoise(x + 9.321, y - 1.531, z - 7.951);
      double ny = this.unoise(x, y, z);
      double nz = this.unoise(x + 6.327, y + 0.1671, z - 2.672);
      vnoise.x += amp * (nx - 0.5);
      vnoise.y += amp * (ny - 0.5);
      vnoise.z += amp * (nz - 0.5);
    }
    vnoise.x = (0.05 + vnoise.x / 1.1);
    vnoise.y = (0.05 + vnoise.y / 1.1);
    vnoise.z = (0.05 + vnoise.z / 1.1);
  }

  /**
   * @see https://svn.blender.org/svnroot/bf-blender/trunk/blender/source/blender/python/mathutils/mathutils_noise.c
   * @param x
   * @param y
   * @param z
   * @param octaves
   * @param ampscale
   * @param freqscale
   * @param vnoise
   */
  public void sturbulence3d(double x, double y, double z, int octaves, double ampscale, double freqscale, Vector3d vnoise) {
    vnoise.x = this.snoise(x + 9.321, y - 1.531, z - 7.951);
    vnoise.y = this.snoise(x, y, z);
    vnoise.z = this.snoise(x + 6.327, y + 0.1671, z - 2.672);
    double amp = 1.0;
    for (int i = 1; i < octaves; i++) {
      amp *= ampscale;
      x *= freqscale;
      y *= freqscale;
      z *= freqscale;
      double nx = this.snoise(x + 9.321, y - 1.531, z - 7.951);
      double ny = this.snoise(x, y, z);
      double nz = this.snoise(x + 6.327, y + 0.1671, z - 2.672);
      vnoise.x += amp * (nx - 0.5);
      vnoise.y += amp * (ny - 0.5);
      vnoise.z += amp * (nz - 0.5);
    }
    vnoise.x = (0.05 + vnoise.x / 1.1);
    vnoise.y = (0.05 + vnoise.y / 1.1);
    vnoise.z = (0.05 + vnoise.z / 1.1);
  }

}
