/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.extensions;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.io.File;

import javax.swing.ImageIcon;
import javax.swing.JCheckBox;
import javax.swing.JEditorPane;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.SwingConstants;

import kandid.Console;
import kandid.Gateway;
import kandid.Kandid;
import kandid.calculation.Calculation;
import kandid.calculation.bridge.PixelBridgeCalculation;
import kandid.calculation.bridge.povray.PovTextureCalculation;
import kandid.colorator.Colorator;
import kandid.colorator.ColoratorFactory;
import kandid.pond.BreederType;
import kandid.pond.util.BreederSearchThread;
import kandid.preference.PreferenceType;
import kandid.preference.ui.IlPreference;
import kandid.preference.util.PreferenceModel;
import kandid.soup.ChromosomeType;
import kandid.soup.ColoratorType;
import kandid.soup.EntityType;
import kandid.soup.ImageType;
import kandid.soup.PopulationType;
import kandid.soup.util.SoupFactory;
import kandid.util.Debug;
import kandid.util.NetworkHelper;

/** Wizard for exporting images to different file formats.
 * @author thomas jourdan
 */
public class ExportWizard extends javax.swing.JFrame {
  private static final long serialVersionUID = 1L;
  
  private static final int charColumns = 50;
  private static ImageIcon exportImageIcon;
  private static ImageIcon rewindImageIcon;
  private static ImageIcon papersizeImageIcon;
  private static ImageIcon insertDbImageIcon;
  private static ImageIcon pfmImageIcon;
  private static ImageIcon jpegImageIcon;
  private static ImageIcon pngImageIcon;
  
  private static Dimension buttonDimension = new java.awt.Dimension(140, 30);

  private String title;
  private String typeName;
  private ImageType image;
  private ChromosomeType chromosome;
  private String calculationName;
  private Colorator colorator;
  private ExportState exportState;
  private int wizPos;
  private BreederSearchThread breederRequestThread;

  /**
   * Creates new form ExportWizard
   *
   * @param chromosome
   * @param calculationName
   * @param colorator
   */
  public ExportWizard(PopulationType population, int nx, Calculation calculation, String title) {
    this.title = title;

    SoupFactory soupFactory = SoupFactory.getSoupFactory();
    chromosome = soupFactory.getChromosome(population, nx);
    image = soupFactory.getImage(population, nx);
    typeName = SoupFactory.getTypeName(population);
    calculationName = calculation.getClass().getName();
    ColoratorType coloratorType = soupFactory.getColorator(population, nx);
    String coloratorName = soupFactory.getColoratorName(population, nx);
    colorator = ColoratorFactory.createColoratorWorker(coloratorName, coloratorType);

    exportState = new ExportState();
    Dimension prefDim = calculation.getPreferredSize();
    if (prefDim != null) {
      exportState.xDim = prefDim.width;
      exportState.yDim = prefDim.height;
    }
    exportState.setFullname(title);
    try {
      initComponents();
      jpegButton.setEnabled(!(calculation instanceof PovTextureCalculation));
      pfmButton.setEnabled(!(calculation instanceof PixelBridgeCalculation));
      wizPos = 1;
      manageCard(0);
      xSizeTextField2.setText("" + exportState.xDim);
      ySizeTextField.setText("" + exportState.yDim);
    } catch (Throwable exc) {
      Debug.stackTrace(exc);
    }
  }

  /** This method is called from within the constructor to
   * initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is
   * always regenerated by the Form Editor.
   */
  private void initComponents() {
    controlPanel = new javax.swing.JPanel();
    okButton = new javax.swing.JButton();
    rewindButton = new javax.swing.JButton();
    choicePanel = new javax.swing.JPanel();
    card1Panel = new javax.swing.JPanel();
    headline1Panel = new javax.swing.JPanel();
    step1Label = new javax.swing.JLabel();
    text1Label = new javax.swing.JLabel();
    step1Panel = new javax.swing.JPanel();
    insertDbButton = new javax.swing.JButton();
    pngButton = new javax.swing.JButton();
    jpegButton = new javax.swing.JButton();
    pfmButton = new javax.swing.JButton();
    insertDbLabel = new javax.swing.JLabel();
    pngLabel = new javax.swing.JLabel();
    jpegLabel = new javax.swing.JLabel();
    pfmLabel = new javax.swing.JLabel();
    card2Panel = new javax.swing.JPanel();
    headline2Panel = new javax.swing.JPanel();
    step2Label = new javax.swing.JLabel();
    text2Label = new javax.swing.JLabel();
    step2Panel = new javax.swing.JPanel();
    size512x512Button = new javax.swing.JButton();
    size1024x1024Button = new javax.swing.JButton();
    freesizeButton = new javax.swing.JButton();
    xSizeTextField2 = new javax.swing.JTextField();
    ySizeTextField = new javax.swing.JTextField();
    yLabel = new javax.swing.JLabel();
    xLabel = new javax.swing.JLabel();
    exportNameLabel = new javax.swing.JLabel();
    exportPathLabel = new javax.swing.JLabel();
    size200x200Button = new javax.swing.JButton();
    card3Panel = new javax.swing.JPanel();
    headline3Panel = new javax.swing.JPanel();
    step3Label = new javax.swing.JLabel();
    text3Label = new javax.swing.JLabel();
    step3Panel = new javax.swing.JPanel();
    card5Panel = new javax.swing.JPanel();
    headline5Panel = new javax.swing.JPanel();
    step5Label = new javax.swing.JLabel();
    text5Label = new javax.swing.JLabel();
    step5Panel = new javax.swing.JPanel();
    outputNameTextField = new javax.swing.JTextField();
    outputPathTextField = new javax.swing.JTextField();
    acceptCheckBox = new JCheckBox();
    authorTextField = new javax.swing.JTextField();

    addWindowListener(new java.awt.event.WindowAdapter() {
      public void windowClosing(java.awt.event.WindowEvent evt) {
        exitForm(evt);
      }
    });

    controlPanel.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.RIGHT));

    if (exportImageIcon == null)
      exportImageIcon = Kandid.getInstance().createImageIcon("export.png", null);
    okButton.setIcon(exportImageIcon);
    okButton.setText("Export image");
    okButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        okButtonActionPerformed(evt);
      }
    });

    controlPanel.add(okButton);

    if (rewindImageIcon == null)
      rewindImageIcon = Kandid.getInstance().createImageIcon("rewind.png", null);
    rewindButton.setIcon(rewindImageIcon);
    rewindButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        backButtonActionPerformed(evt);
      }
    });

    controlPanel.add(rewindButton);

    getContentPane().add(controlPanel, java.awt.BorderLayout.SOUTH);

    choicePanel.setLayout(new java.awt.CardLayout());

    // --- 1 ---
    card1Panel.setLayout(new java.awt.BorderLayout());

    headline1Panel.setLayout(new java.awt.GridLayout(2, 1));

    step1Label.setText("<html><b>&nbsp;&nbsp;&nbsp;Step 1</b></html>");
    headline1Panel.add(step1Label);

    text1Label.setText("<html><b>&nbsp;&nbsp;&nbsp;Select output format.</b></html>");
    headline1Panel.add(text1Label);

    card1Panel.add(headline1Panel, java.awt.BorderLayout.NORTH);

    step1Panel.setLayout(new java.awt.GridBagLayout());
    java.awt.GridBagConstraints gridBagConstraints1;

    if (pngImageIcon == null)
      pngImageIcon = Kandid.getInstance().createImageIcon("ping.png", null);
    pngButton.setIcon(pngImageIcon);
    pngButton.setText("PNG");
    pngButton.setPreferredSize(buttonDimension);
    pngButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        pngButtonActionPerformed(evt);
      }
    });
    gridBagConstraints1 = new java.awt.GridBagConstraints();
    gridBagConstraints1.insets = new java.awt.Insets(5, 10, 5, 10);
    gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
    step1Panel.add(pngButton, gridBagConstraints1);

    if (jpegImageIcon == null)
      jpegImageIcon = Kandid.getInstance().createImageIcon("jpeg.png", null);
    jpegButton.setIcon(jpegImageIcon);
    jpegButton.setText("JPEG");
    jpegButton.setPreferredSize(buttonDimension);
    jpegButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        jpegButton20ActionPerformed(evt);
      }
    });
    gridBagConstraints1 = new java.awt.GridBagConstraints();
    gridBagConstraints1.gridx = 0;
    gridBagConstraints1.gridy = 1;
    gridBagConstraints1.insets = new java.awt.Insets(5, 10, 5, 10);
    gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
    step1Panel.add(jpegButton, gridBagConstraints1);

    if (pfmImageIcon == null)
      pfmImageIcon = Kandid.getInstance().createImageIcon("pfm.png", null);
    pfmButton.setIcon(pfmImageIcon);
    pfmButton.setText("PFM");
    pfmButton.setPreferredSize(buttonDimension);
    pfmButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        pfmButtonActionPerformed(evt);
      }
    });
    gridBagConstraints1 = new java.awt.GridBagConstraints();
    gridBagConstraints1.gridx = 0;
    gridBagConstraints1.gridy = 2;
    gridBagConstraints1.insets = new java.awt.Insets(5, 10, 5, 10);
    gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
    step1Panel.add(pfmButton, gridBagConstraints1);

    if (insertDbImageIcon == null)
      insertDbImageIcon = Kandid.getInstance().createImageIcon("insertdb.png", null);
    insertDbButton.setIcon(insertDbImageIcon);
    insertDbButton.setText("Publish");
    String author = image.getAuthor();
    insertDbButton.setEnabled(author == null || author.length() == 0);
    insertDbButton.setPreferredSize(buttonDimension);
    insertDbButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        insertDbButtonActionPerformed(evt);
      }
    });
    gridBagConstraints1 = new java.awt.GridBagConstraints();
    gridBagConstraints1.gridx = 0;
    gridBagConstraints1.gridy = 5;
    gridBagConstraints1.insets = new java.awt.Insets(5, 10, 5, 10);
    gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
// TODO WEB database   step1Panel.add(insertDbButton, gridBagConstraints1);

    pngLabel.setText("Bitmap: Export Portable Network Graphics");
    gridBagConstraints1 = new java.awt.GridBagConstraints();
    gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
    step1Panel.add(pngLabel, gridBagConstraints1);

    jpegLabel.setText("Bitmap: Export Compressed JPEG Format");
    gridBagConstraints1 = new java.awt.GridBagConstraints();
    gridBagConstraints1.gridx = 1;
    gridBagConstraints1.gridy = 1;
    gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
    step1Panel.add(jpegLabel, gridBagConstraints1);

    pfmLabel.setText("Bitmap: Export Portable FloatMap Image Format");
    gridBagConstraints1 = new java.awt.GridBagConstraints();
    gridBagConstraints1.gridx = 1;
    gridBagConstraints1.gridy = 2;
    gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
    step1Panel.add(pfmLabel, gridBagConstraints1);

    insertDbLabel.setText("Publish in WEB database");
    gridBagConstraints1 = new java.awt.GridBagConstraints();
    gridBagConstraints1.gridx = 1;
    gridBagConstraints1.gridy = 5;
    gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
// TODO WEB database   step1Panel.add(insertDbLabel, gridBagConstraints1);

    card1Panel.add(step1Panel, java.awt.BorderLayout.CENTER);

    choicePanel.add(card1Panel, "jPanel8");

    // --- 2 ---
    card2Panel.setLayout(new java.awt.BorderLayout());

    headline2Panel.setLayout(new java.awt.GridLayout(2, 1));

    step2Label.setText("<html><b>&nbsp;&nbsp;&nbsp;Step 2</b></html>");
    headline2Panel.add(step2Label);

    text2Label.setText("<html><b>&nbsp;&nbsp;&nbsp;Select image size.</b></html>");
    headline2Panel.add(text2Label);

    card2Panel.add(headline2Panel, java.awt.BorderLayout.NORTH);

    step2Panel.setLayout(new java.awt.GridBagLayout());
    java.awt.GridBagConstraints gridBagConstraints2;

    if (papersizeImageIcon == null)
      papersizeImageIcon = Kandid.getInstance().createImageIcon("papersize.png", null);
    size200x200Button.setIcon(papersizeImageIcon);
    size200x200Button.setText("200 x 200");
    size200x200Button.setPreferredSize(buttonDimension);
    size200x200Button.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        size200x200ButtonActionPerformed(evt);
      }
    });
    gridBagConstraints2 = new java.awt.GridBagConstraints();
    gridBagConstraints2.insets = new java.awt.Insets(5, 10, 5, 10);
    gridBagConstraints2.anchor = java.awt.GridBagConstraints.WEST;
    step2Panel.add(size200x200Button, gridBagConstraints2);

    if (papersizeImageIcon == null)
      papersizeImageIcon = Kandid.getInstance().createImageIcon("papersize.png", null);
    size512x512Button.setIcon(papersizeImageIcon);
    size512x512Button.setText("512 x 512");
    size512x512Button.setPreferredSize(buttonDimension);
    size512x512Button.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        size512x512ButtonActionPerformed(evt);
      }
    });
    gridBagConstraints2 = new java.awt.GridBagConstraints();
    gridBagConstraints2.gridx = 0;
    gridBagConstraints2.gridy = 1;
    gridBagConstraints2.insets = new java.awt.Insets(5, 10, 5, 10);
    gridBagConstraints2.anchor = java.awt.GridBagConstraints.WEST;
    step2Panel.add(size512x512Button, gridBagConstraints2);

    if (papersizeImageIcon == null)
      papersizeImageIcon = Kandid.getInstance().createImageIcon("papersize.png", null);
    size1024x1024Button.setIcon(papersizeImageIcon);
    size1024x1024Button.setText("1024 x 1024");
    size1024x1024Button.setPreferredSize(buttonDimension);
    size1024x1024Button.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        size1024x1024ButtonActionPerformed(evt);
      }
    });
    gridBagConstraints2 = new java.awt.GridBagConstraints();
    gridBagConstraints2.gridx = 0;
    gridBagConstraints2.gridy = 2;
    gridBagConstraints2.insets = new java.awt.Insets(5, 10, 5, 10);
    gridBagConstraints2.anchor = java.awt.GridBagConstraints.WEST;
    step2Panel.add(size1024x1024Button, gridBagConstraints2);

    if (papersizeImageIcon == null)
      papersizeImageIcon = Kandid.getInstance().createImageIcon("papersize.png", null);
    freesizeButton.setIcon(papersizeImageIcon);
    freesizeButton.setText("free size");
    freesizeButton.setPreferredSize(buttonDimension);
    freesizeButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        freesizeButtonActionPerformed(evt);
      }
    });
    gridBagConstraints2 = new java.awt.GridBagConstraints();
    gridBagConstraints2.gridx = 0;
    gridBagConstraints2.gridy = 3;
    gridBagConstraints2.insets = new java.awt.Insets(5, 10, 5, 10);
    gridBagConstraints2.anchor = java.awt.GridBagConstraints.WEST;
    step2Panel.add(freesizeButton, gridBagConstraints2);

    yLabel.setText("x  ");
    gridBagConstraints2 = new java.awt.GridBagConstraints();
    gridBagConstraints2.gridx = 1;
    gridBagConstraints2.gridy = 3;
    step2Panel.add(yLabel, gridBagConstraints2);

    xSizeTextField2.setColumns(5);
    gridBagConstraints2 = new java.awt.GridBagConstraints();
    gridBagConstraints2.gridx = 2;
    gridBagConstraints2.gridy = 3;
    gridBagConstraints2.insets = new java.awt.Insets(5, 10, 5, 10);
    step2Panel.add(xSizeTextField2, gridBagConstraints2);

    xLabel.setText("y  ");
    gridBagConstraints2 = new java.awt.GridBagConstraints();
    gridBagConstraints2.gridx = 3;
    gridBagConstraints2.gridy = 3;
    step2Panel.add(xLabel, gridBagConstraints2);

    ySizeTextField.setColumns(5);
    gridBagConstraints2 = new java.awt.GridBagConstraints();
    gridBagConstraints2.gridx = 4;
    gridBagConstraints2.gridy = 3;
    gridBagConstraints2.insets = new java.awt.Insets(5, 0, 5, 0);
    step2Panel.add(ySizeTextField, gridBagConstraints2);

    card2Panel.add(step2Panel, java.awt.BorderLayout.CENTER);

    choicePanel.add(card2Panel, "jPanel9");

    // --- 3 ---
    card3Panel.setLayout(new java.awt.BorderLayout());

    headline3Panel.setLayout(new java.awt.GridLayout(2, 1));

    step3Label.setText("<html><b>&nbsp;&nbsp;&nbsp;Step 3</b></html>");
    headline3Panel.add(step3Label);

    text3Label.setText("<html><b>&nbsp;&nbsp;&nbsp;Automaticaly generated filename.</b></html>");
    headline3Panel.add(text3Label);

    card3Panel.add(headline3Panel, java.awt.BorderLayout.NORTH);

    java.awt.GridBagConstraints gridBagConstraints3;
    step3Panel.setLayout(new java.awt.GridBagLayout());
    
    exportNameLabel.setText("Name of exported image:");
    gridBagConstraints3 = new java.awt.GridBagConstraints();
    gridBagConstraints3.gridx = 0;
    gridBagConstraints3.gridy = 0;
    gridBagConstraints3.insets = new java.awt.Insets(5, 10, 5, 10);
    gridBagConstraints3.anchor = java.awt.GridBagConstraints.WEST;
    step3Panel.add(exportNameLabel, gridBagConstraints3);

    outputNameTextField.setColumns(charColumns);
    gridBagConstraints3 = new java.awt.GridBagConstraints();
    gridBagConstraints3.gridx = 0;
    gridBagConstraints3.gridy = 1;
    gridBagConstraints3.insets = new java.awt.Insets(5, 10, 5, 10);
    gridBagConstraints3.anchor = java.awt.GridBagConstraints.WEST;
    step3Panel.add(outputNameTextField, gridBagConstraints3);

    exportPathLabel.setText("Folder for exported image:");
    gridBagConstraints3 = new java.awt.GridBagConstraints();
    gridBagConstraints3.gridx = 0;
    gridBagConstraints3.gridy = 2;
    gridBagConstraints3.insets = new java.awt.Insets(5, 10, 5, 10);
    gridBagConstraints3.anchor = java.awt.GridBagConstraints.WEST;
    step3Panel.add(exportPathLabel, gridBagConstraints3);

    outputPathTextField.setColumns(charColumns);
    outputPathTextField.setEditable(false);
    gridBagConstraints3 = new java.awt.GridBagConstraints();
    gridBagConstraints3.gridx = 0;
    gridBagConstraints3.gridy = 3;
    gridBagConstraints3.insets = new java.awt.Insets(5, 10, 5, 10);
    gridBagConstraints3.anchor = java.awt.GridBagConstraints.WEST;
    step3Panel.add(outputPathTextField, gridBagConstraints3);

    card3Panel.add(step3Panel, java.awt.BorderLayout.CENTER);

    choicePanel.add(card3Panel, "jPanel");

    // --- 5 ---
    card5Panel.setLayout(new java.awt.BorderLayout());

    headline5Panel.setLayout(new java.awt.GridLayout(2, 1));

    step5Label.setText("<html><b>&nbsp;&nbsp;&nbsp;Step 2</b></html>");
    headline5Panel.add(step5Label);

    text5Label.setText("<html><b>&nbsp;&nbsp;&nbsp;Publish to Kandid WEB database.</html>");
    headline5Panel.add(text5Label);

    card5Panel.add(headline5Panel, java.awt.BorderLayout.NORTH);

    GridBagLayout gridbag = new GridBagLayout();
    GridBagConstraints c = new GridBagConstraints();
    c.gridwidth = GridBagConstraints.REMAINDER;
    c.anchor = GridBagConstraints.WEST;
    step5Panel.setLayout(gridbag);

    JEditorPane editorPane = new JEditorPane();
    editorPane.setText("Publishing this work to Kandid WEB database, will give all users the\n"
  + "following rights:\n"
  + "\n"
  +   "Creative Commons License\n"
  +     "Attribution-NonCommercial-ShareAlike 2.0\n"
  + "\n"
  + "All users are free:\n"
  + "    * to copy, distribute, display, and perform the work\n"
  + "    * to make derivative works\n"
  + "\n"
  + "Under the following conditions:\n"
  + "\n"
  + "    Attribution. Users must give the original author credit.\n"
  + "\n"
  + "    Noncommercial. Users may not use this work for commercial purposes.\n"
  + "\n"
  + "    Share Alike. If a users alters, transforms, or builds upon this work, he\n"
  + "      may distribute the resulting work only under a license identical to\n"
  + "      this one.\n"
  + "\n"
  + "   * For any reuse or distribution, the user must make clear to others the\n"
  + "      license terms of this work.\n"
  + "   * Any of these conditions can be waived if the user gets permission\n"
  + "      from the copyright holder.\n"
  + "\n"
  + "Your fair use and other rights are in no way affected by the above.");
    editorPane.setEditable(false);
    editorPane.setCaretPosition(0);
    JScrollPane editorScrollPane = new JScrollPane(editorPane);
    editorScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
    editorScrollPane.setPreferredSize(new Dimension(450, 180));
    gridbag.setConstraints(editorScrollPane, c);
    step5Panel.add(editorScrollPane);

    JLabel l1 =new JLabel(" ");
    gridbag.setConstraints( l1, c);
    step5Panel.add( l1);

    acceptCheckBox.setText("<html>I accept publishing this work under:<br>Creative Commons<br>Attribution-NonCommercial-ShareAlike&nbsp;2.0  Licence.</html>");
    acceptCheckBox.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        acceptCheckBoxActionPerformed(evt);
      }
    });
    acceptCheckBox.setVerticalTextPosition(SwingConstants.TOP);  
    gridbag.setConstraints(acceptCheckBox, c);
    step5Panel.add( acceptCheckBox);

    JLabel l2 =new JLabel(" ");
    gridbag.setConstraints( l2, c);
    step5Panel.add( l2);

    JLabel l3 =new JLabel("Author:");
    gridbag.setConstraints( l3, c);
    step5Panel.add( l3);
    authorTextField.setColumns(charColumns);
    gridbag.setConstraints( authorTextField, c);
    step5Panel.add( authorTextField);

    card5Panel.add(step5Panel, java.awt.BorderLayout.CENTER);

    choicePanel.add(card5Panel, "insertDbPanel");
    
    // --------
    getContentPane().add(choicePanel, java.awt.BorderLayout.CENTER);

    pack();
  }

  /**
   * @param evt
   */
  protected void acceptCheckBoxActionPerformed(ActionEvent evt) {
    okButton.setEnabled(acceptCheckBox.isSelected());
  }

  private void backButtonActionPerformed(java.awt.event.ActionEvent evt) {
    if (wizPos >= 1) {
      --wizPos;
      ((java.awt.CardLayout) choicePanel.getLayout()).previous(choicePanel);
    }
    manageCard(-1);
  }

  private void okButtonActionPerformed(java.awt.event.ActionEvent evt) {
    if (wizPos >= 3) {
      exportState.setFullname(outputNameTextField.getText());
      SoupFactory soupFactory = SoupFactory.getSoupFactory();
      EntityType entity = soupFactory.createEntity(typeName, image);
      switch (exportState.exportType) {
        case Gateway.BITMAP_FILE :
           (new BitmapFileEngine(chromosome, calculationName, colorator, exportState)).start();
          break;
        case Gateway.GENOMEDB :
       // TODO WEB database
//          if(acceptCheckBox.isSelected()) {
//            int installationId = 0;
//            try {
//              PreferenceType preferences = PreferenceModel.getInstance().getPreference();
//              installationId = preferences.getGenomeDB().getInstallationId();
//
//              HttpPostConnection idConnection = HttpPostConnection.verifyInstallationId(installationId);
//              idConnection.waitWithTimeout();
//              if(!idConnection.isError()) {
//                installationId = idConnection.getConnectionResult().getAutoId();
//                if(installationId > 0) {
//                  preferences.getGenomeDB().setInstallationId(installationId);
//                  PreferenceModel.getInstance().marshalToFile();
//                  Console.append("setting installation id to " + installationId + ".");
//                }
//                
//                image.setInstallationId(installationId);
//                if(authorTextField.isEditable())
//                  image.setAuthor(authorTextField.getText());
//                DateFormat dateFormat = DateFormat.getDateTimeInstance(DateFormat.SHORT, DateFormat.SHORT, Locale.GERMAN);
//                TimeZone timeZone = TimeZone.getDefault();
//                dateFormat.setTimeZone(timeZone);
//                image.setPublishDate(dateFormat.format(new Date()));
//                
//                HttpPostConnection insertConnection = HttpPostConnection.insertChromosome(installationId, 
//                    soupFactory.getChromosome(entity).getIdent(),
//                    calculationName, colorator.getClass().getName(), entity) ;
//                insertConnection.waitWithTimeout();
//                insertConnection.getConnectionResult();
//              }            
//            } catch (Exception exc) {
//              Debug.stackTrace(exc);
//            }
//          }
         break;
        default :
          if (Debug.enabled)
            assert false : "wrong exportType " + exportState.exportType;
      }
	  (new File(exportState.getPath(Kandid.localImageFolder))).mkdirs();
	  soupFactory.marshalToFile(entity, exportState.getPath(Kandid.localImageFolder) + exportState.getFilename() + ".kimg");
      closeWizard();
    }
  }

  private void jpegButton20ActionPerformed(java.awt.event.ActionEvent evt) {
    exportState.exportType = Gateway.BITMAP_FILE;
    exportState.fileExtension = "jpg";
    manageCard(1);
  }

  private void pngButtonActionPerformed(java.awt.event.ActionEvent evt) {
    exportState.exportType = Gateway.BITMAP_FILE;
    exportState.fileExtension = "png";
    manageCard(1);
  }

  private void pfmButtonActionPerformed(java.awt.event.ActionEvent evt) {
    exportState.exportType = Gateway.BITMAP_FILE;
    exportState.fileExtension = "pfm";
    manageCard(1);
  }

  private void insertDbButtonActionPerformed(java.awt.event.ActionEvent evt) {
    try {
      IlPreference.getInstance().ensureExportGenomeDBCompleted();
      PreferenceType preferences = PreferenceModel.getInstance().getPreference();
      
      NetworkHelper.setHTTPProxy();
      
      exportState.exportType = Gateway.GENOMEDB;
      exportState.setFullname(title);
      outputNameTextField.setText(exportState.getFullname());
      okButton.setText("Add to WEB database");

      String author = image.getAuthor();
      if(author != null && author.length() > 0) {
        authorTextField.setText(author);
        authorTextField.setEditable(false);
      }
      else {
        String nickname = preferences.getUser().getNickname();
        String email = preferences.getUser().getEmail();
        String configuredAutor = "";
        if(nickname != null && nickname.length() > 0) {
          configuredAutor += nickname;
        }
        else {
          configuredAutor += "anonymous";
        }
        if(email != null && email.length() > 0) {
          configuredAutor += " " + email;
        }
        authorTextField.setText(configuredAutor);
      }
      manageCard(4);
    } catch (Exception exc) {
      Debug.stackTrace(exc);
      Console.append(exc.getMessage());
    }
  }

  /**
   * Callback after a breeder is found.
   */
  public void signalBreederSearchReady() {
    if (breederRequestThread != null) {
      BreederType breederType = breederRequestThread.getBreederType();
      if (breederType != null) {
        exportState.to = NetworkHelper.getUsableEmail(breederType.getEmail());
//        mailToTextField.setText(exportState.to);
        manageCard(3);
      }
    }
  }

  private void freesizeButtonActionPerformed(java.awt.event.ActionEvent evt) {
    try {
      exportState.xDim = Integer.parseInt(xSizeTextField2.getText());
    } catch (NumberFormatException exc) {
      String errmsg = "Please enter a number greater 0 for x.";
      JOptionPane.showMessageDialog(null, errmsg, "Number format error", JOptionPane.ERROR_MESSAGE);
    }
    try {
      exportState.yDim = Integer.parseInt(ySizeTextField.getText());
    } catch (NumberFormatException exc) {
      String errmsg = "Please enter a number greater 0 for y.";
      JOptionPane.showMessageDialog(null, errmsg, "Number format error", JOptionPane.ERROR_MESSAGE);
    }
    exportState.setFullname(title + "_x" + exportState.xDim + "_y" + exportState.yDim);
    outputNameTextField.setText(exportState.getFullname());
    outputPathTextField.setText(exportState.getAbsolutPath(Kandid.localImageFolder));
    manageCard(1);
  }

  private void size1024x1024ButtonActionPerformed(java.awt.event.ActionEvent evt) {
    exportState.xDim = 1024;
    exportState.yDim = 1024;
    exportState.setFullname(title + "_x" + exportState.xDim + "_y" + exportState.yDim);
    outputNameTextField.setText(exportState.getFullname());
    outputPathTextField.setText(exportState.getAbsolutPath(Kandid.localImageFolder));
    manageCard(1);
  }

  private void size512x512ButtonActionPerformed(java.awt.event.ActionEvent evt) {
    exportState.xDim = 512;
    exportState.yDim = 512;
    exportState.setFullname(title + "_x" + exportState.xDim + "_y" + exportState.yDim);
    outputNameTextField.setText(exportState.getFullname());
    outputPathTextField.setText(exportState.getAbsolutPath(Kandid.localImageFolder));
    manageCard(1);
  } 

  private void size200x200ButtonActionPerformed(java.awt.event.ActionEvent evt) {
    exportState.xDim = 200;
    exportState.yDim = 200;
    exportState.setFullname(title + "_x" + exportState.xDim + "_y" + exportState.yDim);
    outputNameTextField.setText(exportState.getFullname());
    outputPathTextField.setText(exportState.getAbsolutPath(Kandid.localImageFolder));
    manageCard(1);
  } 

  /** Exit the wizard */
  private void exitForm(java.awt.event.WindowEvent evt) { 
    closeWizard();
  }

  /**
   * manage card layout
   */

  private void manageCard(int forward) {
    if (forward > 0) {
      int fc = forward;
      while (fc > 0 && wizPos < 5) {
        ((java.awt.CardLayout) choicePanel.getLayout()).next(choicePanel);
        ++wizPos;
        --fc;
      }
    } else if (forward < 0) {
      //!!      int fc = -forward;
      //!!      while (fc > 0 && wizPos > 1) {
      //!!        ((java.awt.CardLayout) choicePanel.getLayout()).previous(choicePanel);
      //!!        --wizPos;
      //!!        --fc;
      //!!      }
       ((java.awt.CardLayout) choicePanel.getLayout()).first(choicePanel);
      wizPos = 1;
    }
    if (wizPos <= 1)
      okButton.setText("Export image");
    okButton.setEnabled(wizPos >= 3 && wizPos != 5);
    rewindButton.setEnabled(wizPos > 1);
    outputNameTextField.setEnabled(!(exportState.exportType == Gateway.GENOMEDB));
  }

  /**
   * Method closeWizard
   */
  void closeWizard() {
    wizPos = 1;
    setVisible(false);
    dispose();
  }

  private javax.swing.JPanel controlPanel;
  private javax.swing.JButton okButton;
  private javax.swing.JButton rewindButton;
  private javax.swing.JPanel choicePanel;
  private javax.swing.JPanel card1Panel;
  private javax.swing.JPanel headline1Panel;
  private javax.swing.JLabel step1Label;
  private javax.swing.JLabel text1Label;
  private javax.swing.JPanel step1Panel;
  private javax.swing.JButton insertDbButton;
  private javax.swing.JButton pngButton;
  private javax.swing.JButton jpegButton;
  private javax.swing.JButton pfmButton;
  private javax.swing.JLabel insertDbLabel;
  private javax.swing.JLabel pngLabel;
  private javax.swing.JLabel jpegLabel;
  private javax.swing.JLabel pfmLabel;
  private javax.swing.JPanel card2Panel;
  private javax.swing.JPanel headline2Panel;
  private javax.swing.JLabel step2Label;
  private javax.swing.JLabel text2Label;
  private javax.swing.JPanel step2Panel;
  private javax.swing.JButton size512x512Button;
  private javax.swing.JButton size1024x1024Button;
  private javax.swing.JButton freesizeButton;
  private javax.swing.JTextField xSizeTextField2;
  private javax.swing.JTextField ySizeTextField;
  private javax.swing.JLabel yLabel;
  private javax.swing.JLabel xLabel;
  private javax.swing.JButton size200x200Button;
  private javax.swing.JPanel card3Panel;
  private javax.swing.JPanel headline3Panel;
  private javax.swing.JLabel step3Label;
  private javax.swing.JLabel text3Label;
  private javax.swing.JPanel step3Panel;
  private javax.swing.JPanel card5Panel;
  private javax.swing.JPanel headline5Panel;
  private javax.swing.JLabel step5Label;
  private javax.swing.JLabel text5Label;
  private javax.swing.JPanel step5Panel;
  private javax.swing.JTextField outputNameTextField;
  private javax.swing.JTextField outputPathTextField;
  private JCheckBox  acceptCheckBox;
  private javax.swing.JTextField authorTextField;
  private javax.swing.JLabel exportNameLabel;
  private javax.swing.JLabel exportPathLabel;

}
