/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.extensions;

import java.awt.*;
import java.io.*;
import kandid.soup.*;
import kandid.*;
import kandid.colorator.*;
import kandid.util.*;

/**
 * Exports current image to bit map file format.
 * This class uses javax.imageio for exporting.
 * Supported file formats are PNG and JPEG. See javax.imageio API documentation.
 * @author thomas jourdan
 */
public abstract class BitmapEngine extends ExportEngine implements Runnable {

  /**
   * Constructor BitmapEngine
   *
   * @param chromosome chromosome of this image
   * @param calculationName name of the calculation for this image
   * @param colorator color strategy for this image
   * @param exportState detail information like file name and size
   */
  public BitmapEngine(ChromosomeType chromosome, String calculationName, Colorator colorator, ExportState exportState) {
    super(chromosome, calculationName, colorator, exportState);
  }
  
  protected abstract void generateOutput() throws IOException;

  /**
   * start export engine
   */
  public void start() {
    if (engine == null) {
      engine = new Thread(this);
      engine.start();
    }
  }

  /**
   * stop export engine
   */
  public void stop() {
    engine = null;
  }

  /**
   * calculate bitmap and write it to file system
   */
  public void run() {
    long startTime = System.currentTimeMillis();
    Thread me = Thread.currentThread();
    me.setPriority(Thread.MIN_PRIORITY);
    try {
      (new File(exportState.getPath(Kandid.localImageFolder))).mkdirs();
      // calculate image
      kandid.Console.append("exporting " + exportState.getAbsolutPath(Kandid.localImageFolder) + exportState.getFullname());
      gw.setChromosome(chromosome, gw.createCalculation(calculationName), colorator);
      gw.activateBitmapCanvas(new Dimension(exportState.xDim, exportState.yDim));
      gw.getCalculation().setImgaeFormat(exportState.fileExtension);
      while (!gw.getReady() && (me == engine)) {
        gw.calculate(false, exportState.getPath(Kandid.localImageFolder) + exportState.getFilename());
      }
      generateOutput();
      if(Debug.enabled) System.out.println("engine ready " + (System.currentTimeMillis()-startTime) + "ms");
    } catch (Throwable exc) {
//!!      Console.println("error while exporting " + exportState.getPath() + exportState.getFullname());
      kandid.Console.append("error while exporting ");
      kandid.Console.append(exc.toString());
      Debug.stackTrace(exc);
    }
  }
}